"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostedRotationType = exports.HostedRotation = exports.RotationSchedule = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const secretsmanager_generated_1 = require("./secretsmanager.generated");
/**
 * A rotation schedule.
 */
class RotationSchedule extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if ((!props.rotationLambda && !props.hostedRotation) || (props.rotationLambda && props.hostedRotation)) {
            throw new Error('One of `rotationLambda` or `hostedRotation` must be specified.');
        }
        new secretsmanager_generated_1.CfnRotationSchedule(this, 'Resource', {
            secretId: props.secret.secretArn,
            rotationLambdaArn: (_a = props.rotationLambda) === null || _a === void 0 ? void 0 : _a.functionArn,
            hostedRotationLambda: (_b = props.hostedRotation) === null || _b === void 0 ? void 0 : _b.bind(props.secret, this),
            rotationRules: {
                automaticallyAfterDays: props.automaticallyAfter && props.automaticallyAfter.toDays() || 30,
            },
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
    }
}
exports.RotationSchedule = RotationSchedule;
/**
 * A hosted rotation.
 */
class HostedRotation {
    constructor(type, props, masterSecret) {
        this.type = type;
        this.props = props;
        this.masterSecret = masterSecret;
        if (type.isMultiUser && !masterSecret) {
            throw new Error('The `masterSecret` must be specified when using the multi user scheme.');
        }
    }
    /**
     * MySQL Single User.
     */
    static mysqlSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.MYSQL_SINGLE_USER, options);
    }
    /**
     * MySQL Multi User.
     */
    static mysqlMultiUser(options) {
        return new HostedRotation(HostedRotationType.MYSQL_MULTI_USER, options, options.masterSecret);
    }
    /**
     * PostgreSQL Single User.
     */
    static postgreSqlSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.POSTGRESQL_SINGLE_USER, options);
    }
    /**
     * PostgreSQL Multi User.
     */
    static postgreSqlMultiUser(options) {
        return new HostedRotation(HostedRotationType.POSTGRESQL_MULTI_USER, options, options.masterSecret);
    }
    /**
     * Oracle Single User.
     */
    static oracleSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.ORACLE_SINGLE_USER, options);
    }
    /**
     * Oracle Multi User.
     */
    static oracleMultiUser(options) {
        return new HostedRotation(HostedRotationType.ORACLE_MULTI_USER, options, options.masterSecret);
    }
    /**
     * MariaDB Single User.
     */
    static mariaDbSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.MARIADB_SINGLE_USER, options);
    }
    /**
     * MariaDB Multi User.
     */
    static mariaDbMultiUser(options) {
        return new HostedRotation(HostedRotationType.MARIADB_MULTI_USER, options, options.masterSecret);
    }
    /**
     * SQL Server Single User.
     */
    static sqlServerSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.SQLSERVER_SINGLE_USER, options);
    }
    /**
     * SQL Server Multi User.
     */
    static sqlServerMultiUser(options) {
        return new HostedRotation(HostedRotationType.SQLSERVER_MULTI_USER, options, options.masterSecret);
    }
    /**
     * Redshift Single User.
     */
    static redshiftSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.REDSHIFT_SINGLE_USER, options);
    }
    /**
     * Redshift Multi User.
     */
    static redshiftMultiUser(options) {
        return new HostedRotation(HostedRotationType.REDSHIFT_MULTI_USER, options, options.masterSecret);
    }
    /**
     * MongoDB Single User.
     */
    static mongoDbSingleUser(options = {}) {
        return new HostedRotation(HostedRotationType.MONGODB_SINGLE_USER, options);
    }
    /**
     * MongoDB Multi User.
     */
    static mongoDbMultiUser(options) {
        return new HostedRotation(HostedRotationType.MONGODB_MULTI_USER, options, options.masterSecret);
    }
    /**
     * Binds this hosted rotation to a secret.
     */
    bind(secret, scope) {
        var _a, _b, _c, _d, _e, _f, _g;
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-secretsmanager-rotationschedule-hostedrotationlambda.html
        core_1.Stack.of(scope).addTransform('AWS::SecretsManager-2020-07-23');
        if (!this.props.vpc && this.props.securityGroups) {
            throw new Error('`vpc` must be specified when specifying `securityGroups`.');
        }
        if (this.props.vpc) {
            this._connections = new ec2.Connections({
                securityGroups: this.props.securityGroups || [new ec2.SecurityGroup(scope, 'SecurityGroup', {
                        vpc: this.props.vpc,
                    })],
            });
        }
        // Prevent master secret deletion when rotation is in place
        if (this.masterSecret) {
            this.masterSecret.denyAccountRootDelete();
        }
        return {
            rotationType: this.type.name,
            kmsKeyArn: (_a = secret.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            masterSecretArn: (_b = this.masterSecret) === null || _b === void 0 ? void 0 : _b.secretArn,
            masterSecretKmsKeyArn: (_d = (_c = this.masterSecret) === null || _c === void 0 ? void 0 : _c.encryptionKey) === null || _d === void 0 ? void 0 : _d.keyArn,
            rotationLambdaName: this.props.functionName,
            vpcSecurityGroupIds: (_f = (_e = this._connections) === null || _e === void 0 ? void 0 : _e.securityGroups) === null || _f === void 0 ? void 0 : _f.map(s => s.securityGroupId).join(','),
            vpcSubnetIds: (_g = this.props.vpc) === null || _g === void 0 ? void 0 : _g.selectSubnets(this.props.vpcSubnets).subnetIds.join(','),
        };
    }
    /**
     * Security group connections for this hosted rotation.
     */
    get connections() {
        if (!this.props.vpc) {
            throw new Error('Cannot use connections for a hosted rotation that is not deployed in a VPC');
        }
        // If we are in a vpc and bind() has been called _connections should be defined
        if (!this._connections) {
            throw new Error('Cannot use connections for a hosted rotation that has not been bound to a secret');
        }
        return this._connections;
    }
}
exports.HostedRotation = HostedRotation;
/**
 * Hosted rotation type.
 */
class HostedRotationType {
    /**
     * @param name The type of rotation
     * @param isMultiUser Whether the rotation uses the mutli user scheme
     */
    constructor(name, isMultiUser) {
        this.name = name;
        this.isMultiUser = isMultiUser;
    }
}
exports.HostedRotationType = HostedRotationType;
/**
 * MySQL Single User.
 */
HostedRotationType.MYSQL_SINGLE_USER = new HostedRotationType('MySQLSingleUser');
/**
 * MySQL Multi User.
 */
HostedRotationType.MYSQL_MULTI_USER = new HostedRotationType('MySQLMultiUser', true);
/**
 * PostgreSQL Single User.
 */
HostedRotationType.POSTGRESQL_SINGLE_USER = new HostedRotationType('PostgreSQLSingleUser');
/**
 * PostgreSQL Multi User.
 */
HostedRotationType.POSTGRESQL_MULTI_USER = new HostedRotationType('PostgreSQLMultiUser', true);
/**
 * Oracle Single User.
 */
HostedRotationType.ORACLE_SINGLE_USER = new HostedRotationType('OracleSingleUser');
/**
 * Oracle Multi User.
 */
HostedRotationType.ORACLE_MULTI_USER = new HostedRotationType('OracleMultiUser', true);
/**
 * MariaDB Single User.
 */
HostedRotationType.MARIADB_SINGLE_USER = new HostedRotationType('MariaDBSingleUser');
/**
 * MariaDB Multi User.
 */
HostedRotationType.MARIADB_MULTI_USER = new HostedRotationType('MariaDBMultiUser', true);
/**
 * SQL Server Single User.
 */
HostedRotationType.SQLSERVER_SINGLE_USER = new HostedRotationType('SQLServerSingleUser');
/**
 * SQL Server Multi User.
 */
HostedRotationType.SQLSERVER_MULTI_USER = new HostedRotationType('SQLServerMultiUser', true);
/**
 * Redshift Single User.
 */
HostedRotationType.REDSHIFT_SINGLE_USER = new HostedRotationType('RedshiftSingleUser');
/**
 * Redshift Multi User.
 */
HostedRotationType.REDSHIFT_MULTI_USER = new HostedRotationType('RedshiftMultiUser', true);
/**
 * MongoDB Single User.
 */
HostedRotationType.MONGODB_SINGLE_USER = new HostedRotationType('MongoDBSingleUser');
/**
 * MongoDB Multi User.
 */
HostedRotationType.MONGODB_MULTI_USER = new HostedRotationType('MongoDBMultiUser', true);
//# sourceMappingURL=data:application/json;base64,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