"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const crypto = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * (experimental) Specify a test that the canary should run.
 *
 * @experimental
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * (experimental) Specify a custom test with your own code.
     *
     * @param options The configuration options.
     * @returns `Test` associated with the specified Code object
     * @experimental
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
/**
 * (experimental) Runtime options for a canary.
 *
 * @experimental
 */
class Runtime {
    /**
     * @param name The name of the runtime version.
     * @experimental
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
/**
 * (experimental) `syn-1.0` includes the following:.
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-1.0
 * @experimental
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * (experimental) `syn-nodejs-2.0` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 81.0.4044.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-2.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_0 = new Runtime('syn-nodejs-2.0');
/**
 * (experimental) Define a new Canary.
 *
 * @experimental
 */
class Canary extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.stringValue({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_b = (_a = props.artifactsBucketLocation) === null || _a === void 0 ? void 0 : _a.bucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : this.createDefaultRole((_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_e = props.artifactsBucketLocation) === null || _e === void 0 ? void 0 : _e.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_f = props.startAfterCreation) !== null && _f !== void 0 ? _f : true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_g = props.failureRetentionPeriod) === null || _g === void 0 ? void 0 : _g.toDays(),
            successRetentionPeriod: (_h = props.successRetentionPeriod) === null || _h === void 0 ? void 0 : _h.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * (experimental) Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricDuration(options) {
        return this.metric('Duration', options);
    }
    /**
     * (experimental) Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricSuccessPercent(options) {
        return this.metric('SuccessPercent', options);
    }
    /**
     * (experimental) Measure the number of failed canary runs over a given time period.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricFailed(options) {
        return this.metric('Failed', options);
    }
    /**
     * @param metricName - the name of the metric
     * @param options - configuration options for the metric
     *
     * @returns a CloudWatch metric associated with the canary.
     * @default avg over 5 minutes
     */
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            metricName,
            namespace: 'CloudWatchSynthetics',
            dimensions: { CanaryName: this.canaryName },
            statistic: 'avg',
            ...options,
        }).attachTo(this);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: ['arn:aws:logs:::*'],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _a, _b, _c;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName,
            s3Key: (_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey,
            s3ObjectVersion: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _a, _b, _c, _d;
        return {
            durationInSeconds: String(`${(_b = (_a = props.timeToLive) === null || _a === void 0 ? void 0 : _a.toSeconds()) !== null && _b !== void 0 ? _b : 0}`),
            expression: (_d = (_c = props.schedule) === null || _c === void 0 ? void 0 : _c.expressionString) !== null && _d !== void 0 ? _d : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = this.node.uniqueId.toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
}
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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