# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['momento',
 'momento.auth',
 'momento.config',
 'momento.config.transport',
 'momento.errors',
 'momento.internal',
 'momento.internal._utilities',
 'momento.internal.aio',
 'momento.internal.synchronous',
 'momento.requests',
 'momento.responses',
 'momento.responses.control',
 'momento.responses.dictionary_data',
 'momento.responses.list_data',
 'momento.responses.scalar_data',
 'momento.responses.set_data']

package_data = \
{'': ['*']}

install_requires = \
['grpcio>=1.50.0,<2.0.0',
 'momento-wire-types>=0.39,<0.40',
 'pyjwt>=2.4.0,<3.0.0']

setup_kwargs = {
    'name': 'momento',
    'version': '0.22.0',
    'description': 'SDK for Momento',
    'long_description': '<head>\n  <meta name="Momento Python Client Library Documentation" content="Python client software development kit for Momento Serverless Cache">\n</head>\n<img src="https://docs.momentohq.com/img/logo.svg" alt="logo" width="400"/>\n\n[![project status](https://momentohq.github.io/standards-and-practices/badges/project-status-official.svg)](https://github.com/momentohq/standards-and-practices/blob/main/docs/momento-on-github.md)\n[![project stability](https://momentohq.github.io/standards-and-practices/badges/project-stability-alpha.svg)](https://github.com/momentohq/standards-and-practices/blob/main/docs/momento-on-github.md) \n\n# Momento Python Client Library\n\n\nPython client SDK for Momento Serverless Cache: a fast, simple, pay-as-you-go caching solution without\nany of the operational overhead required by traditional caching solutions!\n\n\n\n## Getting Started :running:\n\n### Requirements\n\n- [Python 3.7](https://www.python.org/downloads/) or above is required\n- A Momento Auth Token is required, you can generate one using the [Momento CLI](https://github.com/momentohq/momento-cli)\n\n### Examples\n\nReady to dive right in? Just check out the [examples](./examples/README.md) directory for complete, working examples of\nhow to use the SDK.\n\n### Installation\n\nThe [Momento SDK is available on PyPi](https://pypi.org/project/momento/). To install via pip:\n\n```bash\npip install momento\n```\n\n### Usage\n\nHere is a quickstart you can use in your own project:\n\n```python\nimport logging\nfrom datetime import timedelta\n\nfrom example_utils.example_logging import initialize_logging\n\nfrom momento import SimpleCacheClient\nfrom momento.auth import CredentialProvider\nfrom momento.config import Laptop\nfrom momento.responses import CacheGet, CacheSet, CreateCache, ListCaches\n\n_AUTH_PROVIDER = CredentialProvider.from_environment_variable("MOMENTO_AUTH_TOKEN")\n_CACHE_NAME = "cache"\n_ITEM_DEFAULT_TTL_SECONDS = timedelta(seconds=60)\n_KEY = "MyKey"\n_VALUE = "MyValue"\n\n_logger = logging.getLogger("momento-example")\n\n\ndef _print_start_banner() -> None:\n    _logger.info("******************************************************************")\n    _logger.info("*                      Momento Example Start                     *")\n    _logger.info("******************************************************************")\n\n\ndef _print_end_banner() -> None:\n    _logger.info("******************************************************************")\n    _logger.info("*                       Momento Example End                      *")\n    _logger.info("******************************************************************")\n\n\ndef _create_cache(cache_client: SimpleCacheClient, cache_name: str) -> None:\n    create_cache_response = cache_client.create_cache(cache_name)\n    match create_cache_response:\n        case CreateCache.Success():\n            pass\n        case CreateCache.CacheAlreadyExists():\n            _logger.info(f"Cache with name: {cache_name!r} already exists.")\n        case CreateCache.Error() as error:\n            _logger.error(f"Error creating cache: {error.message}")\n        case _:\n            _logger.error("Unreachable")\n\n\ndef _list_caches(cache_client: SimpleCacheClient) -> None:\n    _logger.info("Listing caches:")\n    list_caches_response = cache_client.list_caches()\n    while True:\n        match list_caches_response:\n            case ListCaches.Success() as success:\n                for cache_info in success.caches:\n                    _logger.info(f"- {cache_info.name!r}")\n                next_token = success.next_token\n                if next_token is None:\n                    break\n            case ListCaches.Error() as error:\n                _logger.error(f"Error creating cache: {error.message}")\n            case _:\n                _logger.error("Unreachable")\n\n        list_caches_response = cache_client.list_caches(next_token)\n    _logger.info("")\n\n\nif __name__ == "__main__":\n    initialize_logging()\n    _print_start_banner()\n    with SimpleCacheClient(Laptop.latest(), _AUTH_PROVIDER, _ITEM_DEFAULT_TTL_SECONDS) as cache_client:\n        _create_cache(cache_client, _CACHE_NAME)\n        _list_caches(cache_client)\n\n        _logger.info(f"Setting Key: {_KEY!r} Value: {_VALUE!r}")\n        set_response = cache_client.set(_CACHE_NAME, _KEY, _VALUE)\n        match set_response:\n            case CacheSet.Success():\n                pass\n            case CacheSet.Error() as error:\n                _logger.error(f"Error creating cache: {error.message}")\n            case _:\n                _logger.error("Unreachable")\n\n        _logger.info(f"Getting Key: {_KEY!r}")\n        get_response = cache_client.get(_CACHE_NAME, _KEY)\n        match get_response:\n            case CacheGet.Hit() as hit:\n                _logger.info(f"Look up resulted in a hit: {hit}")\n                _logger.info(f"Looked up Value: {hit.value_string!r}")\n            case CacheGet.Miss():\n                _logger.info("Look up resulted in a: miss. This is unexpected.")\n            case CacheGet.Error() as error:\n                _logger.error(f"Error creating cache: {error.message}")\n            case _:\n                _logger.error("Unreachable")\n    _print_end_banner()\n\n```\n\nNote 1: the above code requires an environment variable named MOMENTO_AUTH_TOKEN which must\nbe set to a valid [Momento authentication token](https://docs.momentohq.com/docs/getting-started#obtain-an-auth-token).\n\nNote 2: The above code uses [structural pattern matching](https://peps.python.org/pep-0636/), a feature introduced in Python 3.10.\nUsing a Python version less than 3.10? No problem. Here is the same example compatible across all versions of Python:\n\n```python\nimport logging\nfrom datetime import timedelta\n\nfrom example_utils.example_logging import initialize_logging\n\nfrom momento import SimpleCacheClient\nfrom momento.auth import CredentialProvider\nfrom momento.config import Laptop\nfrom momento.responses import CacheGet, CacheSet, CreateCache, ListCaches\n\n_AUTH_PROVIDER = CredentialProvider.from_environment_variable("MOMENTO_AUTH_TOKEN")\n_CACHE_NAME = "cache"\n_ITEM_DEFAULT_TTL_SECONDS = timedelta(seconds=60)\n_KEY = "MyKey"\n_VALUE = "MyValue"\n\n_logger = logging.getLogger("momento-example")\n\n\ndef _print_start_banner() -> None:\n    _logger.info("******************************************************************")\n    _logger.info("*                      Momento Example Start                     *")\n    _logger.info("******************************************************************")\n\n\ndef _print_end_banner() -> None:\n    _logger.info("******************************************************************")\n    _logger.info("*                       Momento Example End                      *")\n    _logger.info("******************************************************************")\n\n\ndef _create_cache(cache_client: SimpleCacheClient, cache_name: str) -> None:\n    create_cache_response = cache_client.create_cache(cache_name)\n    if isinstance(create_cache_response, CreateCache.Success):\n        pass\n    elif isinstance(create_cache_response, CreateCache.CacheAlreadyExists):\n        _logger.info(f"Cache with name: {cache_name!r} already exists.")\n    elif isinstance(create_cache_response, CreateCache.Error):\n        _logger.error(f"Error creating cache: {create_cache_response.message}")\n    else:\n        _logger.error("Unreachable")\n\n\ndef _list_caches(cache_client: SimpleCacheClient) -> None:\n    _logger.info("Listing caches:")\n    list_caches_response = cache_client.list_caches()\n    while True:\n        if isinstance(list_caches_response, ListCaches.Success):\n            for cache_info in list_caches_response.caches:\n                _logger.info(f"- {cache_info.name!r}")\n            next_token = list_caches_response.next_token\n            if next_token is None:\n                break\n        elif isinstance(list_caches_response, ListCaches.Error):\n            _logger.error(f"Error creating cache: {list_caches_response.message}")\n        else:\n            _logger.error("Unreachable")\n\n        list_caches_response = cache_client.list_caches(next_token)\n    _logger.info("")\n\n\nif __name__ == "__main__":\n    initialize_logging()\n    _print_start_banner()\n    with SimpleCacheClient(Laptop.latest(), _AUTH_PROVIDER, _ITEM_DEFAULT_TTL_SECONDS) as cache_client:\n        _create_cache(cache_client, _CACHE_NAME)\n        _list_caches(cache_client)\n\n        _logger.info(f"Setting Key: {_KEY!r} Value: {_VALUE!r}")\n        set_response = cache_client.set(_CACHE_NAME, _KEY, _VALUE)\n        if isinstance(set_response, CacheSet.Success):\n            pass\n        elif isinstance(set_response, CacheSet.Error):\n            _logger.error(f"Error creating cache: {set_response.message}")\n        else:\n            _logger.error("Unreachable")\n\n        _logger.info(f"Getting Key: {_KEY!r}")\n        get_response = cache_client.get(_CACHE_NAME, _KEY)\n        if isinstance(get_response, CacheGet.Hit):\n            _logger.info(f"Look up resulted in a hit: {get_response}")\n            _logger.info(f"Looked up Value: {get_response.value_string!r}")\n        elif isinstance(get_response, CacheGet.Miss):\n            _logger.info("Look up resulted in a: miss. This is unexpected.")\n        elif isinstance(get_response, CacheGet.Error):\n            _logger.error(f"Error creating cache: {get_response.message}")\n        else:\n            _logger.error("Unreachable")\n    _print_end_banner()\n```\n\n### Error Handling\n\nComing Soon!\n\n### Tuning\n\nComing Soon!\n\n----------------------------------------------------------------------------------------\nFor more info, visit our website at [https://gomomento.com](https://gomomento.com)!\n',
    'author': 'Momento',
    'author_email': 'hello@momentohq.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://gomomento.com',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<3.12',
}


setup(**setup_kwargs)
