# src/scrapy_waitrequests/sender/twocaptcha.py

import json
import scrapy

from .base_captcha import BaseCaptchaProvider
from scrapy_algova.wait_request import CaptchaWaitRequest


class TwoCaptchaProvider(BaseCaptchaProvider):
    """
    Minimal 2captcha provider using the new JSON API:

      - createTask:    https://api.2captcha.com/createTask
      - getTaskResult: https://api.2captcha.com/getTaskResult

    Pensado para Turnstile / Recaptcha "Task*".
    """

    submit_url = "https://api.2captcha.com/createTask"
    result_url = "https://api.2captcha.com/getTaskResult"

    def __init__(self, api_key: str):
        self.api_key = api_key

    # ---------- BaseCaptchaProvider API ----------

    def build_submit_request(
        self,
        *,
        website_url: str,
        website_key: str,
        final_callback,
        errback=None,
        meta=None,
        task_type: str = "TurnstileTaskProxyless",
        task_params: dict = None,
    ):
        """
        Crea el createTask request.

        task_type: e.g. "TurnstileTaskProxyless", "TurnstileTask", ...
        task_params: se fusiona dentro de "task" (proxy, action, data, pagedata, etc.).
        """

        task = {
            "type": task_type,
            "websiteURL": website_url,
            "websiteKey": website_key,
        }
        if task_params:
            task.update(task_params)

        payload = {
            "clientKey": self.api_key,
            "task": task,
        }

        return scrapy.Request(
            url=self.submit_url,
            method="POST",
            body=json.dumps(payload),
            headers={"Content-Type": "application/json"},
            callback=self._parse_submit,
            errback=errback,
            cb_kwargs={
                "final_callback": final_callback,
                "final_errback": errback,
            },
            meta=meta or {},
            dont_filter=True,
        )

    def build_result_request_params(self, captcha_id: str) -> dict:
        """
        Construye parámetros para getTaskResult.
        """

        payload = {
            "clientKey": self.api_key,
            "taskId": captcha_id,
        }

        return {
            "url": self.result_url,
            "method": "POST",
            "body": json.dumps(payload),
            "headers": {"Content-Type": "application/json"},
        }

    def parse_result(self, response):
        """
        Interpreta respuesta de getTaskResult.

        Ejemplo OK:
        {
            "errorId": 0,
            "status": "ready",
            "solution": {"token": "...", "userAgent": "..."},
            ...
        }
        """

        data = json.loads(response.text or "{}")

        error_id = data.get("errorId", 0)
        if error_id != 0:
            # algo salió mal: devolver "failed" con código de error
            return "failed", data.get("errorCode") or data

        status = data.get("status")
        if status == "processing":
            return "pending", None
        if status == "ready":
            solution = data.get("solution") or {}
            # para Turnstile el token está en solution["token"]
            token = solution.get("token")
            return "ready", token

        # cualquier cosa rara la tratamos como fallo
        return "failed", data

    # ---------- callbacks internos ----------

    def _parse_submit(self, response, final_callback, final_errback):
        """
        Maneja la respuesta de createTask:

        OK:
        {
            "errorId": 0,
            "taskId": 123456789
        }
        """

        data = json.loads(response.text or "{}")
        if data.get("errorId") != 0:
            if final_errback:
                return final_errback(
                    ValueError(
                        f"2captcha createTask failed: "
                        f"{data.get('errorCode') or data}"
                    )
                )
            return

        task_id = data["taskId"]

        # devolvemos un CaptchaWaitRequest que hará polling a getTaskResult
        yield CaptchaWaitRequest(
            provider=self,
            captcha_id=task_id,
            callback=final_callback,
            errback=final_errback,
            wait_time=5.0,
            max_tries=40,  # 40 * 5s = 200s ~3.3min
            meta=response.meta,
        )