# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fsm']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'fsmpy',
    'version': '2.1.0',
    'description': 'Minimal state machine',
    'long_description': '========\nOverview\n========\n\n.. start-badges\n\n.. list-table::\n    :stub-columns: 1\n\n    * - docs\n      - |docs|\n    * - tests\n      - | |travis|\n        | |codecov|\n    * - package\n      - |version| |wheel| |supported-versions| |supported-implementations|\n\n\n.. |docs| image:: https://img.shields.io/readthedocs/pip.svg?style=flat-square\n    :alt: Read the Docs\n    :target: https://readthedocs.org/projects/pyfsm\n\n.. |travis| image:: https://img.shields.io/travis/Woile/pyfsm.svg?style=flat-square\n    :alt: Travis-CI Build Status\n    :target: https://travis-ci.org/Woile/pyfsm\n\n.. |codecov| image:: https://img.shields.io/codecov/c/github/Woile/pyfsm.svg?style=flat-square\n    :alt: Coverage Status\n    :target: https://codecov.io/github/Woile/pyfsm\n\n.. |version| image:: https://img.shields.io/pypi/v/fsmpy.svg?style=flat-square\n    :alt: PyPI Package latest release\n    :target: https://pypi.python.org/pypi/fsmpy\n\n.. |wheel| image:: https://img.shields.io/pypi/wheel/fsmpy.svg?style=flat-square\n    :alt: PyPI Wheel\n    :target: https://pypi.python.org/pypi/fsmpy\n\n.. |supported-versions| image:: https://img.shields.io/pypi/pyversions/fsmpy.svg?style=flat-square\n    :alt: Supported versions\n    :target: https://pypi.python.org/pypi/fsmpy\n\n.. |supported-implementations| image:: https://img.shields.io/pypi/implementation/fsmpy.svg?style=flat-square\n    :alt: Supported implementations\n    :target: https://pypi.python.org/pypi/fsmpy\n\n\n.. end-badges\n\nMinimal state machine\n\n* Free software: BSD license\n\n\n.. code-block:: python\n\n    import fsm\n\n    class MyTasks(fsm.FiniteStateMachineMixin):\n        """An example to test the state machine.\n\n        Contains transitions to everywhere, nowhere and specific states.\n        """\n\n        state_machine = {\n            \'created\': \'__all__\',\n            \'pending\': (\'running\',),\n            \'running\': (\'success\', \'failed\'),\n            \'success\': None,\n            \'failed\': (\'retry\',),\n            \'retry\': (\'pending\', \'retry\'),\n        }\n\n        def __init__(self, state):\n            """Initialize setting a state."""\n            self.state = state\n\n        def on_before_pending(self):\n            print("I\'m going to a pending state")\n\n::\n\n    In [4]: m = MyTasks(state=\'created\')\n\n    In [5]: m.change_state(\'pending\')\n    I\'m going to a pending state\n    Out[5]: \'pending\'\n\n\n::\n\n    In [6]: m.change_state(\'failed\')  # Let\'s try to transition to an invalid state\n    ---------------------------------------------------------------------------\n    InvalidTransition                         Traceback (most recent call last)\n    <ipython-input-6-71d2461eee74> in <module>()\n    ----> 1 m.change_state(\'failed\')\n\n    ~/pyfsm/src/fsm/fsm.py in change_state(self, next_state, **kwargs)\n        90             msg = "The transition from {0} to {1} is not valid".format(previous_state,\n        91                                                                        next_state)\n    ---> 92             raise InvalidTransition(msg)\n        93\n        94         name = \'pre_{0}\'.format(next_state)\n\n    InvalidTransition: The transition from pending to failed is not valid\n\n.. contents::\n    :depth: 2\n\n\nInstallation\n============\n\n::\n\n    pip install fsmpy\n\n\nUsage\n======\n\n1. Define in a class the :code:`state_machine`\n2. Initialize :code:`state`, either with a value, using :code:`__init__` or as a django field\n3. Add hooks:\n\n+------------------------+-------------------------------------------------------------------------------------------------------------+\n| Method                 | Description                                                                                                 |\n+------------------------+-------------------------------------------------------------------------------------------------------------+\n| on_before_change_state | Before transitioning to the state                                                                           |\n+------------------------+-------------------------------------------------------------------------------------------------------------+\n| on_change_state        | After transitioning to the state, if no failure, runs for every state                                       |\n+------------------------+-------------------------------------------------------------------------------------------------------------+\n| pre_<state_name>       | Runs before a particular state, where :code:`state_name` is the specified name in the :code:`state_machine` |\n+------------------------+-------------------------------------------------------------------------------------------------------------+\n| post_<state_name>      | Runs after a particular state, where :code:`state_name` is the specified name in the :code:`state_machine`  |\n+------------------------+-------------------------------------------------------------------------------------------------------------+\n\nThis hooks will receive any extra argument given to :code:`change_state`\n\n\nE.g:\n\nRunning :code:`m.change_state(\'pending\', name=\'john\')` will trigger :code:`pre_pending(name=\'john\')`\n\nDjango integration\n==================\n\n.. code-block:: python\n\n    import fsm\n    from django.db import models\n\n\n    class MyModel(models.Model, fsm.FiniteStateMachineMixin):\n        """An example to test the state machine.\n\n        Contains transitions to everywhere, nowhere and specific states.\n        """\n\n        CHOICES = (\n            (\'created\', \'CREATED\'),\n            (\'pending\', \'PENDING\'),\n            (\'running\', \'RUNNING\'),\n            (\'success\', \'SUCCESS\'),\n            (\'failed\', \'FAILED\'),\n            (\'retry\', \'RETRY\'),\n        )\n\n        state_machine = {\n            \'created\': \'__all__\',\n            \'pending\': (\'running\',),\n            \'running\': (\'success\', \'failed\'),\n            \'success\': None,\n            \'failed\': (\'retry\',),\n            \'retry\': (\'pending\', \'retry\'),\n        }\n\n        state = models.CharField(max_length=30, choices=CHOICES, default=\'created\')\n\n        def on_change_state(self, previous_state, next_state, **kwargs):\n            self.save()\n\nDjango Rest Framework\n---------------------\n\nIf you are using :code:`serializers`, they usually perform the :code:`save`, so saving inside :code:`on_change_state` is not necessary.\n\nOne simple solution is to do this:\n\n.. code-block:: python\n\n    class MySerializer(serializers.ModelSerializer):\n\n        def update(self, instance, validated_data):\n            new_state = validated_data.get(\'state\', instance.state)\n            try:\n                instance.change_state(new_state)\n            except fsm.InvalidTransition:\n                raise serializers.ValidationError("Invalid transition")\n            instance = super().update(instance, validated_data)\n            return instance\n\n\nDocumentation\n=============\n\nhttps://pyfsm.readthedocs.org/\n\nDevelopment\n===========\n\nTo run the tests run::\n\n    tox\n\nNote, to combine the coverage data from all the tox environments run:\n\n.. list-table::\n    :widths: 10 90\n    :stub-columns: 1\n\n    - - Windows\n      - ::\n\n            set PYTEST_ADDOPTS=--cov-append\n            tox\n\n    - - Other\n      - ::\n\n            PYTEST_ADDOPTS=--cov-append tox\n',
    'author': 'Santiago Fraire Willemoes',
    'author_email': 'santiwilly@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Woile/pyfsm',
    'packages': packages,
    'package_data': package_data,
}


setup(**setup_kwargs)
