"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a
 * specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com)
 */
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param hostedZoneId the ID of the hosted zone to import
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param publicHostedZoneId the ID of the public hosted zone to import
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param privateHostedZoneId the ID of the private hosted zone to import
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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