# Fred ![alt text](https://raw.githubusercontent.com/derohde/Fred/master/logo/logo.png "Fred logo")
A fast, scalable and light-weight C++ Fréchet distance library, exposed to python and focused on (k,l)-clustering of polygonal curves.

### NOW AVAILABLE VIA PIP
### NOW WITH OPTIMIZED CLUSTER CENTERS

## Ingredients
`import Fred as fred`

- for verbosity, set `fred.config.verbosity`, default is `0`, possible values `0,1,2,3`

### Number of Threads

By default, Fred will automatically determine the number of threads to use. If you want to set an upper limit, set `fred.config.number_threads`. Set to `-1` to enable dynamic mode again.

### Curve
- signature: `fred.Curve(np.ndarray)`, `fred.Curve(np.ndarray, str name)`
- properties: `fred.Curve.values`: curves as `np.ndarray`, `fred.Curve.name`: get name of curve, `fred.Curve.dimensions`: dimension of curve, `fred.Curve.complexity`: number of points of curve

### Curves
- signature: `fred.Curves()`
- methods: `fred.Curves.add(curve)`: add curve, `fred.Curves[i]`: get ith curve, `len(fred.Curves)`: number curves, `fred.Curves + fred.Curves`: add two sets of curves,  `fred.Curves.simplify(l)`: return set of simplified curves
- properties:  `fred.Curves.m`: maximum complexity of the contained curves, `fred.Curves.values`: curves as `np.ndarray`

#### continous Fréchet distance
- signature: `fred.continuous_frechet(curve1, curve2)`
- returns: `fred.Continuous_Frechet_Result` with members `value`, `time_bounds`: running-time for upper and lower bound, `number_searches`: number of free space diagrams built, `time_searches`: running-time for free spaces

###### continuous Fréchet distance config
- approximation error in percent of distance: `fred.config.continuous_frechet_error`, which defaults to 1

#### discrete Fréchet distance
- signature: `fred.discrete_frechet(curve1, curve2)`
- returns: `fred.Discrete_Frechet_Result` with members `value` and `time`

#### discrete dynamic time warping distance
- signature: `fred.discrete_dynamic_time_warping(curve1, curve2)`
- returns: `fred.Discrete_Dynamic_Time_Warping_Distance` with members `value` and `time`

### Curve Simplification

All simplifications are vertex-restricted!

#### minimum error simplification
- graph approach from [**Polygonal Approximations of a Curve — Formulations and Algorithms**](https://www.sciencedirect.com/science/article/pii/B9780444704672500114)
- signature: `fred.minimum_error_simplification(fred.Curve, int complexity)`
- returns: `fred.Curve`that uses input curves vertices, with `complexity` number of vertices and that has minimum distance to input curve

#### approximate minimum link simplification
- algorithm "FS" from [**Near-Linear Time Approximation Algorithms for Curve Simplification**](https://link.springer.com/article/10.1007/s00453-005-1165-y)
- signature: `fred.approximate_minimum_link_simplification(fred.Curve, double error)`
- returns: `fred.Curve` that uses input curves vertices, is of small complexity and with distance to input curve at most `error`

#### approximate minimum error simplification
- binary search on `fred.approximate_minimum_link_simplification`
- signature: `fred.approximate_minimum_error_simplification(fred.Curve, int complexity)`
- returns: `fred.Curve`that uses input curves vertices, with `complexity` number of vertices and that has small distance to input curve

### Clustering

#### discrete (k,l)-center clustering (continuous Fréchet)
- from [**Approximating (k,l)-center clustering for curves**](https://dl.acm.org/doi/10.5555/3310435.3310616)
- signature: `fred.discrete_klcenter(k, l, curves, distances, random_first_center, fast_simplification)` with parameters 
    - `k`: number of centers
    - `l`: maximum complexity of the centers
    - `local_search`: number of iterations of local search to improve solution, defaults to `0`
    - `consecutive_call`: reuses distances and simplifications already computed in a previous call if `true`, defaults to `false`
    - `random_first_center`: determines if first center is chosen uniformly at random or first curve is used as first center, optional, defaults to true
    - `fast_simplification`: determines whether to use the minimum error simplification or the faster approximate minimum error simplification, defaults to `false`
- returns: `fred.Clustering_Result` with mebers 
    - `value`: objective value 
    - `time`: running-time 
    - `assignment`: empty if compute_assignment has not been called

#### discrete (k,l)-median clustering (continuous Fréchet)
- Algorithm from section 4.3 in [**Geometric Approximation Algorithms**](http://www.ams.org/books/surv/173/) + simplification
- signature: `fred.discrete_klmedian(k, l, curves, distances, fast_simplification)` with parameters 
    - `k`: number of centers
    - `l`: maximum complexity of the centers
    - `consecutive_call`: reuses distances and simplifications already computed in a previous call if `true`, defaults to `false`
    - `fast_simplification`: determines whether to use the minimum error simplification or the faster approximate minimum error simplification, defaults to `false`
- returns: `fred.Clustering_Result` with mebers 
    - `value`: objective value 
    - `time`: running-time 
    - `assignment`: empty if compute_assignment has not been called

#### Clustering Result
- signature: `fred.Clustering_Result`
- methods: 
    - `len(fred.Clustering_Result)`: number of centers
    - `fred.Clustering_Result[i]`: get ith center
    - `fred.Clustering_Result.compute_assignment(fred.Curves, bool consecutive_call)`: assigns every curve to its nearest center with parameter `consecutive_call`, which defaults to `false`; set to true, if you want to assign the curves used for clustering
    - `fred.Clustering_Result.optimize(fred.Curves, bool consecutive_call)`: (heuristically) optimizes cluster centers using a [stabbing algorithm](https://arxiv.org/abs/2212.01458)
- members: 
    - `value`: objective value
    - `time`: running-time
    - `assignment`: empty if compute_assignment was not called

#### Cluster Assignment
- signature: `fred.Cluster_Assignment`
- methods: 
    - `len(fred.Cluster_Assignment)`: number of centers
    - `fred.Cluster_Assignment.count(i)`: number of curves assigned to center `i`
    - `fred.Cluster_Assignment.get(i,j)`: get index of `j`th curve assigned to center `i`

### Dimension Reduction via Gaussian Random Projection 
- [Section 2 in **Random Projections and Sampling Algorithms for Clustering of High Dimensional Polygonal Curves**](https://papers.nips.cc/paper/9443-random-projections-and-sampling-algorithms-for-clustering-of-high-dimensional-polygonal-curves)
- signature: `fred.dimension_reduction(curves, epsilon, empirical_constant)` with parameters `epsilon`: (1+epsilon) approximation parameter, `empirical_constant`: use constant of empirical study (faster, but less accurate), defaults to `True`
- returns: `fred.Curves` collection of curves
  
## Installation

### Requirements

You have to have installed:
 - cmake
 - git
 - openmp available (should be a part of your compiler)
 
Thats it!

#### Installing on Windows

It's best to use the [Ubuntu subsystem](https://ubuntu.com/wsl), which is easy to install through powershell. Once you have this installed, use apt to install gcc, cmake and python. Afterwards you can follow the following procedure.

#### Installing on Mac

Apple's clang does not really support openmp, which is now kind of an integral part of Fred. You can try to install libomp via homebrew, but the best way would be to get a virtual ubuntu running.

### Installation Procedure

 - Variant 1: simply run `pip install Fred-Frechet`
 - Variant 2: clone repository and run `make` for installation into userdir

## Test
Just run `python py/test.py`.
  
## Mini Example
```python
import Fred as fred
import numpy as np
import pandas as pd

curve1d = fred.Curve([1., 2.]) # Curve stores a polygonal curve with 
                                         # at least two points of at least one 
                                         # and equal number of dimensions

curve2d1 = fred.Curve([[1., 0.], [2., 1.], [3., 0.]]) # any number of dimensions and points works
curve2d2 = fred.Curve([[1., -1.], [2., -2.], [3., -1.]], "optional name, e.g. displayed in plot") 

print(curve2d1)

fred.plot_curve(curve2d1, curve2d2)
fred.plot_curve(curve2d2, fred.minimum_error_simplification(curve2d2, 2))

print("distance is {}".format(fred.continuous_frechet(curve2d1, curve2d2).value))

print("download HUGE curves") 

import requests, zipfile, io             # you can use all libraries 
                                         # that work with numpy to read data into fred
                                         
re = requests.get("https://archive.ics.uci.edu/ml/machine-learning-databases/00447/data.zip", stream=True)
zf = zipfile.ZipFile(io.BytesIO(re.content))

ps1 = fred.Curve(pd.read_csv(zf.open('PS1.txt'), delimiter="\t", header=None).values[:50], "PS1")
ps2 = fred.Curve(pd.read_csv(zf.open('PS2.txt'), delimiter="\t", header=None).values[:50], "PS2")
ps3 = fred.Curve(pd.read_csv(zf.open('PS3.txt'), delimiter="\t", header=None).values[:50], "PS3")
ps4 = fred.Curve(pd.read_csv(zf.open('PS4.txt'), delimiter="\t", header=None).values[:50], "PS4")
ps5 = fred.Curve(pd.read_csv(zf.open('PS5.txt'), delimiter="\t", header=None).values[:50], "PS5")
ps6 = fred.Curve(pd.read_csv(zf.open('PS6.txt'), delimiter="\t", header=None).values[:50], "PS6")

curves = fred.Curves() # for clustering or if you want to apply dimension reduction
                       # you need to encapsulate your curves in a Curves object
              
curves.add(ps1)
curves.add(ps2)
curves.add(ps3)
curves.add(ps4)
curves.add(ps5)
curves.add(ps6)

fred.plot_curve(curves)

curves = fred.dimension_reduction(curves, 0.95) # fred is pretty fast but with high dimensional data
                                                # a dimension reduction massively improves running-time
                                                # even for smaller values of epsilon
                                                
fred.plot_curve(curves)
                                  
# Oneshot clustering - if you already know the value of k
                                  
clustering = fred.discrete_klcenter(2, 10, curves) # fast but coarse
          
clustering = fred.discrete_klmedian(2, 10, curves) # slow but better results

print("clustering cost is {}".format(clustering.value))

for i, center in enumerate(clustering):
    print(f"center {i} is {center}")

fred.plot_curve(clustering)

# Multiple clustering calls - if you need to find a suitable value for k
                            
for k in range(2, 6):
    
    clustering = fred.discrete_klcenter(k, 10, curves, consecutive_call=True)
    print(f"clustering cost is {clustering.value}")
            
    clustering = fred.discrete_klmedian(k, 10, curves, consecutive_call=True)
    print(f"clustering cost is {clustering.value}")

clustering.compute_assignment(curves, consecutive_call=True) # use consecutive_call = False when computing assignment for curves other
                                                             # than those used for computing the clustering

for i in range(len(clustering)):
    for j in range(clustering.assignment.count(i)):
        print(f"{curves[clustering.assignment.get(i,j)].name} was assigned to center {clustering[i].name}")

clustering.optimize_centers(curves, consecutive_call = True) # uses stabbing to get better centers

fred.plot_clustering(clustering, curves)

```
## Visual Example

```python
import Fred as fred

curves = fred.Curves()
curves.add(fred.Curve([[-10.93943942, -37.06242838],[-10.93944232, -37.06256992],[-10.93944232, -37.06256992],[-10.93944232, -37.06256992],[-10.93899402, -37.06280406],[-10.93860314, -37.06296434],[-10.93835807, -37.06278978],[-10.93829299, -37.06237239],[-10.93854427, -37.06176238],[-10.93889765, -37.06105771],[-10.9392863 , -37.06034013],[-10.93958482, -37.0597757 ],[-10.93981259, -37.05935465],[-10.94003912, -37.0589944 ],[-10.94029375, -37.05890855],[-10.94044568, -37.05899324],[-10.94071019, -37.05920933],[-10.94091498, -37.05938079],[-10.94092187, -37.05938262],[-10.94111489, -37.05955812],[-10.94143233, -37.05984643],[-10.94189287, -37.06029526],[-10.94234691, -37.06074624],[-10.9425804 , -37.06096802],[-10.94263953, -37.06101946],[-10.94263953, -37.06101946],[-10.94263953, -37.06101946],[-10.94266097, -37.0610417 ],[-10.94286097, -37.0612328 ],[-10.94315305, -37.06155233],[-10.94350765, -37.06192153],[-10.94410138, -37.0621534 ],[-10.94468481, -37.06230469],[-10.94527759, -37.06251654],[-10.94594673, -37.06281271],[-10.94671859, -37.06308869],[-10.94730157, -37.06335312],[-10.94778644, -37.06381369],[-10.94844587, -37.06445785],[-10.94904832, -37.06500803],[-10.94979403, -37.06571115],[-10.9504376 , -37.06632879],[-10.95101859, -37.06687168],[-10.95164801, -37.06724984],[-10.95194617, -37.06738994],[-10.95245685, -37.06766077],[-10.95311692, -37.06801289],[-10.95373597, -37.06834647],[-10.95425643, -37.06861906],[-10.95456037, -37.06876904],[-10.9551056 , -37.06906634],[-10.95571837, -37.06937444],[-10.95645998, -37.06951516],[-10.95722549, -37.06945968],[-10.9582645 , -37.06937347],[-10.959267  , -37.06930038],[-10.96033403, -37.06924749],[-10.9616459 , -37.069179  ],[-10.96270941, -37.06910438],[-10.96397319, -37.06903497],[-10.96497853, -37.06897429],[-10.9656844 , -37.06850669],[-10.96592044, -37.06768898],[-10.96610809, -37.06688631],[-10.96630181, -37.06600845],[-10.96646721, -37.06534033],[-10.96698213, -37.06499147],[-10.96764198, -37.06528425],[-10.96814327, -37.06557494],[-10.9687186 , -37.06599832],[-10.96927369, -37.0664603 ],[-10.96945204, -37.06660118],[-10.96959754, -37.06671691],[-10.96985275, -37.0669367 ],[-10.97016947, -37.0671957 ],[-10.97041451, -37.06739896],[-10.97048234, -37.06735379],[-10.97032484, -37.06715337],[-10.97020309, -37.06704726],[-10.97024076, -37.066873  ],[-10.97046391, -37.0665994 ],[-10.97074272, -37.06632107],[-10.97105451, -37.06647656],[-10.97140669, -37.06677885],[-10.97186827, -37.06717748],[-10.97201951, -37.06732553],[-10.97215358, -37.06744312],[-10.9723854 , -37.06761853],[-10.97258704, -37.06777718],[-10.97286671, -37.067999  ],[-10.97318196, -37.06824471],[-10.97329008, -37.06835596],[-10.97331396, -37.0683913 ],[-10.97331396, -37.0683913 ],[-10.97340625, -37.06848176],[-10.97355518, -37.06857843],[-10.97388026, -37.06881238],[-10.97409123, -37.06894244],[-10.97418474, -37.06886148],[-10.97428593, -37.0687222 ],[-10.97436295, -37.06863122],[-10.97442223, -37.06855982],[-10.97447997, -37.06848969],[-10.97454607, -37.06841423],[-10.97467334, -37.06824215],[-10.97479694, -37.06806764],[-10.97487819, -37.06798559],[-10.97494334, -37.06790972],[-10.97494334, -37.06790972],[-10.97496155, -37.06788944],[-10.97496155, -37.06788944],[-10.97496155, -37.06788944],[-10.97496155, -37.06788944]]))
curves.add(fred.Curve([[-10.90889296, -37.05237154],[-10.90904785, -37.05260632],[-10.90973536, -37.05266121],[-10.91010518, -37.05261827],[-10.91000414, -37.05256681],[-10.91048646, -37.05260013],[-10.91135943, -37.05269498],[-10.91229152, -37.05277322],[-10.91342033, -37.05270886],[-10.91431659, -37.05278532],[-10.91506897, -37.05266128],[-10.91544909, -37.05242861],[-10.91593664, -37.05267365],[-10.91667623, -37.05265982],[-10.91748816, -37.05281022],[-10.91801454, -37.05269153],[-10.91787201, -37.05279041],[-10.91785134, -37.05282591],[-10.91792023, -37.052902  ],[-10.91837675, -37.05296211],[-10.91850017, -37.0533763 ],[-10.91849212, -37.05358395],[-10.91849799, -37.05357595],[-10.9184368 , -37.05413224],[-10.91850316, -37.05443409],[-10.91850858, -37.05446375],[-10.9185052 , -37.05446315],[-10.91839066, -37.05517003],[-10.91841456, -37.05612296],[-10.9185024 , -37.05687158],[-10.9181955 , -37.05773603],[-10.91861413, -37.05873733],[-10.91855651, -37.05895014],[-10.91835643, -37.05889317],[-10.91854341, -37.05965251],[-10.91848843, -37.06081932],[-10.91843979, -37.06219825],[-10.91841746, -37.06266105],[-10.91837862, -37.06279615],[-10.91831496, -37.06300964],[-10.9183473 , -37.06299321],[-10.91837899, -37.06298774],[-10.91837791, -37.06298758],[-10.918378  , -37.06298698],[-10.91837822, -37.06298652],[-10.91841453, -37.06321011],[-10.91847828, -37.06394916],[-10.91861829, -37.06397188],[-10.91861297, -37.06398652],[-10.91861178, -37.06398627],[-10.91861355, -37.06398464],[-10.91861037, -37.06398528],[-10.91861033, -37.06398535],[-10.91861101, -37.06398616],[-10.91861229, -37.06398567],[-10.91861452, -37.06398519],[-10.91852107, -37.06433786],[-10.91854463, -37.06535452],[-10.91862404, -37.06630359],[-10.91879595, -37.06735107],[-10.91863274, -37.06821768],[-10.91867692, -37.06827064],[-10.91864985, -37.06846654],[-10.91861474, -37.06844821],[-10.91861158, -37.06845233],[-10.91861335, -37.06844958],[-10.91863853, -37.06863415],[-10.91866434, -37.06936379],[-10.91857642, -37.06997517],[-10.91855849, -37.07080263],[-10.91839213, -37.07178178],[-10.91835998, -37.07254682],[-10.91817896, -37.07257863],[-10.91805938, -37.07302097],[-10.91824085, -37.07327488],[-10.91825536, -37.07351718],[-10.9182579 , -37.07365122],[-10.91824237, -37.07365167],[-10.91824648, -37.07364797],[-10.9182473 , -37.07364716],[-10.91824587, -37.07364717],[-10.9182463 , -37.07364646],[-10.91824667, -37.07364626],[-10.91824711, -37.07364598],[-10.91829794, -37.07417675],[-10.91832389, -37.07471123],[-10.91834816, -37.07553953],[-10.91838258, -37.07596903],[-10.91830836, -37.076009  ],[-10.91831556, -37.07602092],[-10.91835704, -37.07625065],[-10.91840705, -37.07718778],[-10.91836099, -37.07770235],[-10.91843053, -37.07811014],[-10.91859487, -37.07820678],[-10.91832849, -37.07835976],[-10.91841839, -37.07871173],[-10.91858561, -37.07912875],[-10.91853693, -37.08029196],[-10.91847022, -37.08132272],[-10.91864122, -37.08232053],[-10.91878421, -37.08390708],[-10.91902161, -37.08549171],[-10.91957646, -37.08678353],[-10.91990603, -37.08762652],[-10.92011327, -37.08811781],[-10.92026249, -37.08852046],[-10.92027593, -37.08896067],[-10.92028812, -37.08971096],[-10.92086275, -37.09022295],[-10.92119806, -37.09125864],[-10.92158668, -37.09245244],[-10.9219998 , -37.09341251],[-10.92227813, -37.09389165],[-10.92251669, -37.09491899],[-10.9227356 , -37.09633969],[-10.92225014, -37.09771002],[-10.92184819, -37.09906076],[-10.92144139, -37.09987985],[-10.92144481, -37.09995651],[-10.92130557, -37.10039003],[-10.92115916, -37.1016734 ],[-10.92137987, -37.10308411],[-10.92125626, -37.10455606],[-10.92161594, -37.10520467],[-10.92292039, -37.10514096],[-10.92374638, -37.10509667],[-10.92379817, -37.10503713],[-10.92420939, -37.10468552]]))
curves.add(fred.Curve([[-10.90885896, -37.05223934],[-10.90887542, -37.05237009],[-10.90888581, -37.05236599],[-10.9088792 , -37.05236119],[-10.908879  , -37.0523613 ],[-10.90887897, -37.05236133],[-10.90887908, -37.05236184],[-10.90887925, -37.05236193],[-10.90887916, -37.05236232],[-10.90888031, -37.05238124],[-10.90901423, -37.05261118],[-10.90985044, -37.05264657],[-10.9107936 , -37.05260385],[-10.91128793, -37.05269182],[-10.91130952, -37.05271182],[-10.91131643, -37.05271003],[-10.91131638, -37.05270946],[-10.91131644, -37.05270941],[-10.91143227, -37.05271666],[-10.91178154, -37.05270394],[-10.91236017, -37.05273264],[-10.91239461, -37.05271299],[-10.91241785, -37.05269624],[-10.91241363, -37.0526976 ],[-10.91241079, -37.05269947],[-10.91287302, -37.05276819],[-10.91292392, -37.05277988],[-10.91327659, -37.05276669],[-10.91415117, -37.05275218],[-10.91495188, -37.05273417],[-10.91496974, -37.05215649],[-10.9149677 , -37.05136187],[-10.91497338, -37.05058764],[-10.91496995, -37.05030794],[-10.91496585, -37.04945366],[-10.91495289, -37.04837212],[-10.91492666, -37.04793266],[-10.9154707 , -37.04776878],[-10.9164315 , -37.04755438],[-10.91754675, -37.04716886],[-10.91769218, -37.04709118],[-10.91810692, -37.0469302 ],[-10.91830395, -37.0470265 ],[-10.91830991, -37.0471823 ],[-10.9183291 , -37.04789938],[-10.91833195, -37.04793058],[-10.91834546, -37.04793107],[-10.91834466, -37.04793116],[-10.91835246, -37.04833939],[-10.91834766, -37.04877383],[-10.91838263, -37.04880066],[-10.91838436, -37.04879317],[-10.91838504, -37.04883652],[-10.9183568 , -37.04934469],[-10.91837932, -37.04946533],[-10.91840019, -37.04948704],[-10.91842577, -37.05018411],[-10.91843672, -37.0509644 ],[-10.91843705, -37.05104008],[-10.91844653, -37.05104291],[-10.91843375, -37.05144372],[-10.91840768, -37.05230779],[-10.91838656, -37.05247549],[-10.91839701, -37.0524806 ],[-10.91839752, -37.0524812 ],[-10.91839744, -37.05248117],[-10.91840084, -37.05276094],[-10.91842085, -37.05350222],[-10.91844609, -37.05396258],[-10.91842658, -37.05428625],[-10.91843082, -37.05427083],[-10.91843453, -37.05428462],[-10.91843585, -37.05428381],[-10.91842443, -37.0544715 ],[-10.91849376, -37.05466126],[-10.91845835, -37.05466794],[-10.91846788, -37.05466888],[-10.91846515, -37.05467152],[-10.91850948, -37.0554206 ],[-10.91847707, -37.05563209],[-10.91848641, -37.05582653],[-10.91848739, -37.05660023],[-10.91851866, -37.05669263],[-10.91852299, -37.05686276],[-10.91846154, -37.05698656],[-10.91844117, -37.05698978],[-10.91843684, -37.05699138],[-10.91843657, -37.05699116],[-10.91842257, -37.05705144],[-10.91849069, -37.05770009],[-10.91848882, -37.05861348],[-10.91845581, -37.0588178 ],[-10.91844748, -37.05883368],[-10.91847247, -37.05965045],[-10.91859661, -37.06049115],[-10.91856505, -37.06149714],[-10.91869476, -37.06279455],[-10.91859068, -37.06408679],[-10.91859419, -37.06415096],[-10.91858553, -37.06414306],[-10.91858083, -37.06414215],[-10.91858146, -37.06414188],[-10.91858273, -37.06414125],[-10.91858285, -37.06414109],[-10.91858276, -37.06414071],[-10.91858274, -37.06414038],[-10.91855532, -37.0645434 ],[-10.91863649, -37.06557188],[-10.91866141, -37.06580983],[-10.91866441, -37.06610947],[-10.91867941, -37.06713852],[-10.91868266, -37.06809659],[-10.91863953, -37.06822028],[-10.9186659 , -37.06856735],[-10.91867465, -37.06856386],[-10.91867849, -37.06859656],[-10.91866908, -37.06912704],[-10.91865205, -37.06996072],[-10.91859789, -37.07080823],[-10.91863396, -37.07076615],[-10.91863365, -37.07076375],[-10.91863391, -37.07076374],[-10.91861796, -37.07089762],[-10.91845358, -37.0718163 ],[-10.91835186, -37.0724233 ],[-10.91835338, -37.0725683 ],[-10.91831338, -37.07322569],[-10.9183142 , -37.07349223],[-10.91832067, -37.07348685],[-10.91832039, -37.0734867 ],[-10.91832038, -37.07348678],[-10.91832031, -37.07348701],[-10.91831564, -37.07382692],[-10.91830293, -37.07403445],[-10.91830898, -37.07412932],[-10.91831836, -37.07419686],[-10.91832134, -37.07419849],[-10.9183278 , -37.07419625],[-10.91832944, -37.07419579],[-10.91832958, -37.07419529],[-10.91832968, -37.07419532],[-10.91832977, -37.07419536],[-10.91832858, -37.07424334],[-10.91831282, -37.07518683],[-10.91833099, -37.07639977],[-10.91840824, -37.0777744 ],[-10.91849951, -37.07861672],[-10.91845452, -37.07862479],[-10.91848634, -37.07914864],[-10.91857345, -37.0800367 ],[-10.91862898, -37.0811015 ],[-10.91859219, -37.08148031],[-10.91861545, -37.08191268],[-10.91871241, -37.08309159],[-10.91886331, -37.08459254],[-10.91930283, -37.08603994],[-10.91976034, -37.08729571],[-10.92013428, -37.08829363],[-10.92030583, -37.08884993],[-10.92032012, -37.08970807],[-10.92088373, -37.09034097],[-10.92127412, -37.09139195],[-10.92173602, -37.09259479],[-10.92215114, -37.0937036 ],[-10.9222773 , -37.0940131 ],[-10.92231795, -37.09408497],[-10.92260962, -37.09483125],[-10.92274161, -37.09607361],[-10.92232907, -37.09745904],[-10.92191838, -37.09877962],[-10.92140382, -37.10007076],[-10.92136317, -37.10019156],[-10.92128902, -37.10063571],[-10.92126559, -37.10172765],[-10.92135036, -37.10325494],[-10.92123861, -37.1046231 ],[-10.92183621, -37.10517821],[-10.92305273, -37.1051427 ],[-10.92372149, -37.1051285 ],[-10.92376114, -37.10506548]]))
curves.add(fred.Curve([[-10.91084681, -37.07166612],[-10.91084827, -37.07166627],[-10.91084866, -37.07166643],[-10.91084876, -37.07166642],[-10.91084877, -37.07166648],[-10.91084878, -37.07166644],[-10.91084879, -37.0716664 ],[-10.91108778, -37.07134133],[-10.91101406, -37.07097936],[-10.91106469, -37.07039257],[-10.91116368, -37.06980669],[-10.9112987 , -37.06916358],[-10.91136723, -37.06848953],[-10.91139371, -37.06808087],[-10.91138729, -37.06777591],[-10.91140065, -37.06747087],[-10.91147758, -37.06711645],[-10.91151886, -37.0668637 ],[-10.91146682, -37.06645663],[-10.91145419, -37.06586694],[-10.91145156, -37.06537729],[-10.91143917, -37.0650615 ],[-10.91142248, -37.06492729],[-10.9113822 , -37.06471988],[-10.91137874, -37.06437949],[-10.91128405, -37.06416269],[-10.91106123, -37.06400982],[-10.9107417 , -37.06391896],[-10.91070748, -37.06374787],[-10.91069302, -37.06379217],[-10.9106894 , -37.06379635],[-10.91068961, -37.06379618],[-10.91069017, -37.06379778],[-10.91066955, -37.06381148],[-10.91065168, -37.06382585],[-10.91065187, -37.06382756],[-10.91065149, -37.0638275 ],[-10.91046345, -37.06376038],[-10.91003268, -37.06359161],[-10.90954249, -37.06323076],[-10.90905153, -37.06283556],[-10.90851284, -37.06246959],[-10.90788971, -37.06204795],[-10.90737506, -37.0616739 ],[-10.90701436, -37.06134047],[-10.90681351, -37.06104878],[-10.90679137, -37.0610476 ],[-10.90677751, -37.06104718],[-10.9066537 , -37.06097462],[-10.90641888, -37.06108662],[-10.90608072, -37.06102893],[-10.90580368, -37.06088867],[-10.90569895, -37.06080945],[-10.90559621, -37.06078814],[-10.90538341, -37.06063794],[-10.90489687, -37.06043854],[-10.90433668, -37.06028677],[-10.90390004, -37.06011298],[-10.90344991, -37.05994839],[-10.90316524, -37.05984431],[-10.90304264, -37.05958596],[-10.90309358, -37.05917924],[-10.90299615, -37.05909411],[-10.90281366, -37.0590481 ],[-10.90246731, -37.05904368],[-10.90197105, -37.0590733 ],[-10.90152028, -37.05908249],[-10.90092242, -37.05906881],[-10.90041634, -37.05904959],[-10.89983202, -37.0590348 ],[-10.89940699, -37.05901467],[-10.89916147, -37.05897371],[-10.89906724, -37.05881088],[-10.89894604, -37.05869827],[-10.89898272, -37.05866933],[-10.89898149, -37.05866148],[-10.89898258, -37.05865768],[-10.89898321, -37.05865787],[-10.89899389, -37.05862473],[-10.89893697, -37.0585505 ],[-10.89894792, -37.05840642],[-10.89890551, -37.05836317],[-10.89886548, -37.05832611],[-10.89886218, -37.05828079],[-10.89885211, -37.05826662],[-10.89882905, -37.05829454],[-10.89881562, -37.05828908],[-10.89876544, -37.05826351],[-10.89876377, -37.05817255],[-10.89876495, -37.05809176],[-10.89877268, -37.05809386],[-10.89876802, -37.05808856],[-10.89876323, -37.05808883],[-10.89876458, -37.05808451],[-10.89877188, -37.05808067],[-10.89877632, -37.05807892],[-10.89877717, -37.05807859],[-10.89877717, -37.0580786 ],[-10.89877756, -37.05807845],[-10.89877318, -37.05801001],[-10.89877909, -37.05798185],[-10.89876268, -37.05794467],[-10.89873004, -37.05793166],[-10.89872761, -37.05795591],[-10.89873506, -37.05794047],[-10.89873677, -37.05793122],[-10.89873712, -37.05793141],[-10.89871481, -37.0578775 ],[-10.89867777, -37.0578005 ],[-10.89866617, -37.05778977],[-10.89866379, -37.05778433],[-10.89866502, -37.05778877],[-10.89866404, -37.05779084],[-10.89863354, -37.05780816],[-10.89856611, -37.05786058],[-10.89841719, -37.05791092],[-10.89815447, -37.0580658 ],[-10.89784878, -37.05829262],[-10.89761903, -37.0581144 ],[-10.89732929, -37.05789227],[-10.89717686, -37.05778185],[-10.89707984, -37.05772735],[-10.89701027, -37.0576393 ],[-10.89700872, -37.05759186],[-10.89699122, -37.05759739],[-10.89697745, -37.05758239],[-10.8969464 , -37.05737001],[-10.89682958, -37.05693202],[-10.89681246, -37.05656891],[-10.89670929, -37.05638987],[-10.8966253 , -37.05597366],[-10.89654314, -37.05545415],[-10.89646924, -37.0550843 ],[-10.89639706, -37.05463208],[-10.89628769, -37.0542623 ],[-10.89621119, -37.05378888],[-10.89616226, -37.05344222],[-10.89628005, -37.0532723 ],[-10.89639045, -37.05331839],[-10.89660797, -37.05335822],[-10.89679471, -37.05342657],[-10.89686863, -37.05337445]]))
curves.add(fred.Curve([[-10.89690567, -37.05341903],[-10.8969056 , -37.05341898],[-10.89690555, -37.05341891],[-10.89690549, -37.0534187 ],[-10.89690548, -37.05341856],[-10.8969055 , -37.05341849],[-10.89690548, -37.05341843],[-10.8969055 , -37.05341849],[-10.89690554, -37.05341838],[-10.89690564, -37.05341839],[-10.89701263, -37.05340852],[-10.89722523, -37.05342557],[-10.89723832, -37.05350449],[-10.89717068, -37.05363369],[-10.897036  , -37.05378087],[-10.89698835, -37.05385026],[-10.89678631, -37.05376199],[-10.8964454 , -37.05376647],[-10.89629112, -37.05371052],[-10.89632802, -37.05372689],[-10.89629152, -37.05375495],[-10.89628692, -37.05375099],[-10.89627716, -37.05375652],[-10.89625987, -37.05397725],[-10.89628992, -37.05433394],[-10.89641704, -37.05491169],[-10.89650205, -37.05538725],[-10.89662549, -37.05585079],[-10.89672853, -37.05639197],[-10.89683215, -37.05684729],[-10.89688334, -37.05709153],[-10.89688178, -37.05708553],[-10.89688413, -37.05708087],[-10.89690865, -37.05714155],[-10.89691783, -37.05718521],[-10.89693176, -37.05732323],[-10.89695903, -37.05751781],[-10.89707767, -37.05775834],[-10.8972072 , -37.05786438],[-10.89736296, -37.05798437],[-10.89741663, -37.05802338],[-10.89741647, -37.05801987],[-10.89741557, -37.05802015],[-10.89741965, -37.05802177],[-10.89742085, -37.05802202],[-10.89742117, -37.05802217],[-10.89742128, -37.05802227],[-10.89742143, -37.05802221],[-10.89742139, -37.05802219],[-10.89747894, -37.05807238],[-10.89773568, -37.05829255],[-10.89812081, -37.05825626],[-10.89859467, -37.05800414],[-10.89911292, -37.05752394],[-10.89970898, -37.05712928],[-10.90018724, -37.05675472],[-10.90076842, -37.05639709],[-10.90127605, -37.05604642],[-10.90165875, -37.05573016],[-10.90198215, -37.05546222],[-10.90227251, -37.05526793],[-10.90236639, -37.05522582],[-10.90236246, -37.05521877],[-10.90235807, -37.05523154],[-10.90236236, -37.05523916],[-10.90236447, -37.05524287],[-10.90236454, -37.055243  ],[-10.90236466, -37.05524333],[-10.90236475, -37.05524339],[-10.9023649 , -37.05524363],[-10.90254257, -37.05513317],[-10.90282964, -37.05494034],[-10.90306344, -37.05478325],[-10.90311258, -37.05512214],[-10.90309853, -37.05551105],[-10.90308741, -37.05611231],[-10.90308918, -37.05664367],[-10.90306858, -37.05728638],[-10.90303992, -37.05784568],[-10.90302953, -37.05844184],[-10.90303765, -37.05884853],[-10.90302128, -37.05913893],[-10.90301883, -37.05925678],[-10.90303325, -37.05952559],[-10.90306907, -37.0598469 ],[-10.90339438, -37.06001599],[-10.90392679, -37.06023751],[-10.90438371, -37.06042155],[-10.90471035, -37.06053125],[-10.90472527, -37.06054158],[-10.90471949, -37.06054326],[-10.90471906, -37.06054548],[-10.90471908, -37.06054581],[-10.90471909, -37.06054581],[-10.90483354, -37.0605773 ],[-10.90513748, -37.06065388],[-10.90539805, -37.06075924],[-10.90572617, -37.06093143],[-10.90616548, -37.06116768],[-10.90636103, -37.06128798],[-10.90637358, -37.06128514],[-10.90638002, -37.06129139],[-10.90637535, -37.06128164],[-10.90637324, -37.06127892],[-10.90638069, -37.06128578],[-10.90639188, -37.06128388],[-10.90640803, -37.06129463],[-10.90646538, -37.06144708],[-10.90640914, -37.06188596],[-10.90635141, -37.06230836],[-10.90627555, -37.06287501],[-10.90624604, -37.06336722],[-10.90614223, -37.06403633],[-10.90599761, -37.06461488],[-10.90596871, -37.06476812],[-10.90598258, -37.06478104],[-10.90598051, -37.0647775 ],[-10.90595152, -37.06481843],[-10.90588523, -37.06512019],[-10.90568886, -37.06564232],[-10.90541711, -37.06616048],[-10.90526082, -37.06673083],[-10.90529493, -37.06738995],[-10.90536405, -37.06791785],[-10.90539059, -37.06823072],[-10.90533728, -37.0684509 ],[-10.90534819, -37.06848812],[-10.9053413 , -37.06849338],[-10.90534095, -37.06849168],[-10.90534075, -37.06849177],[-10.90534526, -37.06854019],[-10.90535917, -37.06871221],[-10.90532939, -37.06907934],[-10.90538317, -37.06936346],[-10.90534533, -37.06977163],[-10.90526838, -37.07022421],[-10.90520452, -37.07062292],[-10.90512337, -37.07119707],[-10.90509762, -37.07160193],[-10.90516816, -37.07161939],[-10.90533123, -37.07164524],[-10.90547442, -37.07166854],[-10.90579741, -37.07172973],[-10.90615877, -37.07175405],[-10.90625685, -37.07177615],[-10.90625743, -37.07178249],[-10.90639338, -37.0717877 ],[-10.90667219, -37.07181285],[-10.90701824, -37.07187944],[-10.90728979, -37.07193113],[-10.90739651, -37.07193527],[-10.90750588, -37.07194521],[-10.90767892, -37.07196762],[-10.9078449 , -37.07198601],[-10.90821305, -37.0720277 ],[-10.90850588, -37.07211123],[-10.90852023, -37.07224636],[-10.90849121, -37.07251041],[-10.90843154, -37.07291201],[-10.90840754, -37.07304199],[-10.90836781, -37.07307114],[-10.90809344, -37.07305034],[-10.90785105, -37.07298895],[-10.90777115, -37.07300544],[-10.90775505, -37.07305879],[-10.9077381 , -37.07307371],[-10.90773741, -37.07307583],[-10.90773953, -37.07307682],[-10.90773936, -37.07307664],[-10.90773946, -37.07307666],[-10.90773948, -37.07307667],[-10.90773945, -37.07307666],[-10.90773944, -37.07307627],[-10.90773926, -37.07307592],[-10.90773938, -37.07307563],[-10.9077394 , -37.07307568]]))
curves.add(fred.Curve([[-10.91123058, -37.05267365],[-10.91124767, -37.05266673],[-10.91124849, -37.05266653],[-10.91124838, -37.0526666 ],[-10.91133679, -37.05267613],[-10.91201697, -37.05272876],[-10.91224702, -37.05273415],[-10.91239475, -37.05275545],[-10.91292339, -37.05277202],[-10.91292327, -37.0527765 ],[-10.9129218 , -37.05277463],[-10.91293049, -37.05277216],[-10.91293028, -37.05277515],[-10.91313676, -37.05277452],[-10.91401928, -37.05278626],[-10.91511555, -37.05281484],[-10.9154083 , -37.0528026 ],[-10.91539379, -37.05279944],[-10.91546781, -37.05279263],[-10.91603673, -37.05279344],[-10.91661873, -37.05280683],[-10.91733891, -37.0528258 ],[-10.91775113, -37.05287111],[-10.91773283, -37.05286237],[-10.91773206, -37.05286124],[-10.91773169, -37.0528582 ],[-10.91779636, -37.05286287],[-10.91833575, -37.0528786 ],[-10.91842743, -37.05339072],[-10.91844051, -37.05347797],[-10.91844687, -37.0537144 ],[-10.91846088, -37.05419054],[-10.91846194, -37.05423861],[-10.91846267, -37.05444798],[-10.91846288, -37.05507308],[-10.91845544, -37.05581652],[-10.91846079, -37.0565293 ],[-10.9184686 , -37.05657055],[-10.91847605, -37.05657923],[-10.91849796, -37.05668691],[-10.91849858, -37.05672019],[-10.9184975 , -37.05671895],[-10.91850749, -37.05715694],[-10.91851813, -37.05809036],[-10.91854393, -37.0590754 ],[-10.91853867, -37.0591914 ],[-10.91854379, -37.05961591],[-10.91857009, -37.06058723],[-10.9185627 , -37.06189653],[-10.91859115, -37.06301676],[-10.91862926, -37.06329844],[-10.91863808, -37.0633107 ],[-10.91863583, -37.06331346],[-10.91863625, -37.06331355],[-10.91864117, -37.06361367],[-10.91862685, -37.06431794],[-10.91863416, -37.06503393],[-10.91862364, -37.06556393],[-10.91863441, -37.06575273],[-10.91863947, -37.06591478],[-10.91864849, -37.06671648],[-10.9186753 , -37.0678659 ],[-10.91869008, -37.06819236],[-10.91870717, -37.06826001],[-10.91871544, -37.06866141],[-10.91872237, -37.06865415],[-10.91872258, -37.0686538 ],[-10.91872298, -37.06865352],[-10.91872397, -37.06865361],[-10.91871196, -37.06871912],[-10.91870781, -37.06922829],[-10.91863526, -37.07005487],[-10.91852447, -37.07097459],[-10.9185456 , -37.07100337],[-10.9185539 , -37.07100317],[-10.91855358, -37.07100342],[-10.91855085, -37.0710288 ],[-10.91848045, -37.07152553],[-10.91835316, -37.07246386],[-10.91838253, -37.07263478],[-10.91835019, -37.07274853],[-10.91832443, -37.07352369],[-10.9183089 , -37.07378666],[-10.91833896, -37.07454645],[-10.91833543, -37.07477267],[-10.91835102, -37.07489425],[-10.91839505, -37.0756999 ],[-10.9184516 , -37.07683635],[-10.91849241, -37.07819382],[-10.91848552, -37.07858075],[-10.91850192, -37.0787994 ],[-10.91855006, -37.07956554],[-10.91859746, -37.08065662],[-10.91859775, -37.08137351],[-10.91859778, -37.08145844],[-10.91867707, -37.08226554],[-10.91882245, -37.08402486],[-10.91914942, -37.08569243],[-10.91982387, -37.08741927],[-10.92017818, -37.08841318],[-10.9202986 , -37.08917997],[-10.92072642, -37.08992047],[-10.92122898, -37.09107839],[-10.92177801, -37.09255782],[-10.92219332, -37.09375387],[-10.92223437, -37.09396239],[-10.92233379, -37.09418302],[-10.92262864, -37.09500575],[-10.92266096, -37.09649988],[-10.92195172, -37.09856732],[-10.92144971, -37.09992324],[-10.92137764, -37.10017009],[-10.92139415, -37.10009944],[-10.92139071, -37.1000928 ]]))
curves.add(fred.Curve([[-10.92235667, -37.05147   ],[-10.92245167, -37.05147   ],[-10.922455  , -37.05147167],[-10.922455  , -37.05147167],[-10.922455  , -37.05147167],[-10.92244   , -37.05151333],[-10.92244167, -37.051515  ],[-10.92236833, -37.05157167],[-10.92232   , -37.05160333],[-10.92224167, -37.05160833],[-10.92227   , -37.05163   ],[-10.92228   , -37.05163667],[-10.92256167, -37.05171667],[-10.92279167, -37.05172167],[-10.92287833, -37.05168667],[-10.922875  , -37.05166833],[-10.922875  , -37.05166833],[-10.922875  , -37.05166833],[-10.92295333, -37.05153333],[-10.92300167, -37.05128167],[-10.92305   , -37.050805  ],[-10.92307833, -37.05046833],[-10.92308167, -37.05028333],[-10.92311833, -37.04986333],[-10.923185  , -37.049385  ],[-10.92318833, -37.049205  ],[-10.92317167, -37.04867167],[-10.92321667, -37.048355  ],[-10.92323   , -37.04823   ],[-10.92325833, -37.04800667],[-10.92331667, -37.04777333],[-10.92334667, -37.04744   ],[-10.923395  , -37.04708833],[-10.923435  , -37.04672333],[-10.92351833, -37.04620167],[-10.92360667, -37.04573833],[-10.923665  , -37.04539833],[-10.923745  , -37.04493833],[-10.92373833, -37.04466667],[-10.92377833, -37.0446    ],[-10.92394833, -37.04455833],[-10.92397167, -37.04454667],[-10.923965  , -37.04453167],[-10.92396   , -37.04453167],[-10.92443   , -37.04439667],[-10.92459333, -37.04432   ],[-10.9246    , -37.04433167],[-10.92460667, -37.04434667],[-10.92475833, -37.04429167],[-10.92513333, -37.04418167],[-10.92552167, -37.04397833],[-10.92571   , -37.043965  ],[-10.92572167, -37.04397167],[-10.92572167, -37.04397167],[-10.92572167, -37.04397167],[-10.92572167, -37.04397167],[-10.92572167, -37.04397167],[-10.9258    , -37.043975  ],[-10.92580167, -37.04398667],[-10.92580167, -37.04398667],[-10.92580167, -37.04398667],[-10.92580167, -37.04398667],[-10.92593333, -37.04392833],[-10.92621167, -37.04386167],[-10.926465  , -37.04372833],[-10.926785  , -37.04363   ],[-10.92716333, -37.04347   ],[-10.92743833, -37.04339167],[-10.92759667, -37.04336667],[-10.92751833, -37.04335   ],[-10.92751833, -37.04335   ],[-10.92743167, -37.04332667],[-10.92742   , -37.04332167],[-10.92742   , -37.04332167],[-10.92817833, -37.04359   ],[-10.92840667, -37.04383667],[-10.92860667, -37.04408833],[-10.92876333, -37.044275  ],[-10.92886333, -37.044375  ],[-10.92888833, -37.04440667],[-10.92889167, -37.04442   ],[-10.92890333, -37.04441833],[-10.92889167, -37.04443167],[-10.92889167, -37.04443167],[-10.92889167, -37.04443167],[-10.92890833, -37.04447167],[-10.92902667, -37.0446    ],[-10.92917833, -37.04482167],[-10.92937333, -37.045035  ],[-10.92968333, -37.04534333],[-10.92988667, -37.04561167],[-10.93001333, -37.04579333],[-10.93005667, -37.04585167],[-10.930055  , -37.04585167],[-10.93012333, -37.045905  ],[-10.93011667, -37.04590833],[-10.93011667, -37.04590833],[-10.93011667, -37.04590833],[-10.93011667, -37.04590833],[-10.93011667, -37.04590833],[-10.93011667, -37.04590833],[-10.93011667, -37.04590833],[-10.930125  , -37.04593667],[-10.930245  , -37.04609833],[-10.93042167, -37.04631667],[-10.93060333, -37.04656667],[-10.93077167, -37.04681833],[-10.93086333, -37.046925  ],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93086333, -37.04692667],[-10.93089667, -37.04699833],[-10.93106333, -37.04724167],[-10.93141667, -37.04764667],[-10.931765  , -37.04799833],[-10.93224333, -37.04842333],[-10.93266   , -37.04882167],[-10.933135  , -37.04924333],[-10.933515  , -37.04955833],[-10.93387167, -37.04981333],[-10.93406167, -37.04995167],[-10.93412833, -37.049985  ],[-10.93412833, -37.04999333],[-10.934125  , -37.049985  ],[-10.934125  , -37.049985  ],[-10.934125  , -37.049985  ],[-10.93411167, -37.04998667],[-10.93411833, -37.05001   ],[-10.93411667, -37.050015  ],[-10.93411667, -37.050015  ],[-10.93412667, -37.050025  ],[-10.93412667, -37.050025  ],[-10.93410167, -37.04999667],[-10.934105  , -37.049995  ],[-10.93415333, -37.050035  ],[-10.93417333, -37.050035  ],[-10.934185  , -37.05002333],[-10.93423167, -37.05004167],[-10.934445  , -37.05017167],[-10.93468667, -37.05030667],[-10.934755  , -37.05034   ],[-10.934765  , -37.05036333],[-10.934945  , -37.05047   ],[-10.935165  , -37.05059667],[-10.93553833, -37.05054833],[-10.93595167, -37.050585  ],[-10.93634   , -37.05065833],[-10.936765  , -37.05062   ],[-10.93713   , -37.05069833],[-10.937205  , -37.05069833],[-10.937185  , -37.05070333],[-10.9372    , -37.05069   ],[-10.93724333, -37.05067833],[-10.93729167, -37.050675  ],[-10.937485  , -37.05078   ],[-10.93782667, -37.05076833],[-10.93817667, -37.05076667],[-10.93863   , -37.05086   ],[-10.939075  , -37.05095833],[-10.93945167, -37.05093833],[-10.93960833, -37.050925  ],[-10.93960167, -37.05091167],[-10.93960167, -37.05091167],[-10.93960167, -37.05091167],[-10.93960167, -37.05091167],[-10.93960167, -37.05091167],[-10.93972833, -37.050865  ],[-10.94003167, -37.05088   ],[-10.94022667, -37.05093333],[-10.94022667, -37.050975  ],[-10.94022   , -37.05097333],[-10.94022   , -37.05097333],[-10.94022   , -37.05097333],[-10.94022   , -37.05097333],[-10.94022   , -37.05097333],[-10.94032333, -37.05095   ],[-10.94040167, -37.05099   ],[-10.94049333, -37.05100833],[-10.94072333, -37.05096167],[-10.940955  , -37.05098333],[-10.94117333, -37.05085167],[-10.94116833, -37.05049333],[-10.94139333, -37.05001167],[-10.94171167, -37.04958667],[-10.94217667, -37.04901333],[-10.94268833, -37.04832333],[-10.94328333, -37.04768   ],[-10.94373   , -37.04728167],[-10.94403   , -37.04705167],[-10.944075  , -37.04698667],[-10.944065  , -37.04698833],[-10.94406   , -37.04698667],[-10.94406   , -37.04698667],[-10.94406   , -37.04698667],[-10.94406   , -37.04698667],[-10.94416167, -37.046895  ],[-10.94442   , -37.04671   ],[-10.94480667, -37.04646333],[-10.94512   , -37.04631   ],[-10.94538333, -37.04619333],[-10.945745  , -37.04602   ],[-10.94621167, -37.04585   ],[-10.94687333, -37.04570333],[-10.94749833, -37.04562833],[-10.94818167, -37.04555833],[-10.94889167, -37.04549333],[-10.94947167, -37.04546833],[-10.95003833, -37.045485  ],[-10.95074667, -37.04557667],[-10.95132833, -37.04566333],[-10.95208333, -37.045675  ],[-10.95271667, -37.04561   ],[-10.95331833, -37.045465  ],[-10.95378667, -37.04532333],[-10.9542    , -37.04525   ],[-10.954435  , -37.04570333],[-10.95462333, -37.04604167],[-10.95494667, -37.04606167],[-10.955405  , -37.04585   ],[-10.95597667, -37.04556833],[-10.9567    , -37.04523833],[-10.957435  , -37.04489   ],[-10.95807   , -37.04461833],[-10.95875   , -37.04432   ],[-10.95936167, -37.04403833],[-10.96001667, -37.04375667],[-10.96067333, -37.043465  ],[-10.961495  , -37.04309333],[-10.96215667, -37.04280333],[-10.96284333, -37.04254   ],[-10.96367667, -37.042345  ],[-10.964405  , -37.04234667],[-10.96523833, -37.04244833],[-10.965885  , -37.04257167],[-10.96649167, -37.04268833],[-10.96696333, -37.04291167],[-10.96724833, -37.04302333],[-10.96727167, -37.04318333],[-10.96722833, -37.04340333],[-10.967165  , -37.04360333],[-10.96710667, -37.0438    ],[-10.96706   , -37.04399667],[-10.96699167, -37.04420667],[-10.96697333, -37.044455  ],[-10.967125  , -37.04454333],[-10.96729167, -37.04457333],[-10.967565  , -37.04461833],[-10.96780167, -37.04471167],[-10.96814833, -37.04478   ],[-10.96845333, -37.04481333],[-10.968825  , -37.04485   ],[-10.96915833, -37.04492833],[-10.969565  , -37.04501667],[-10.96984   , -37.045075  ],[-10.97021167, -37.04514333],[-10.97058333, -37.04526   ],[-10.97099   , -37.04539167],[-10.97121167, -37.04547   ],[-10.971485  , -37.04557333],[-10.97189   , -37.04567   ],[-10.972315  , -37.045765  ],[-10.97257667, -37.045845  ],[-10.97284167, -37.0459    ],[-10.97332333, -37.04598333],[-10.97385667, -37.04610167],[-10.97429833, -37.04618   ],[-10.97471833, -37.046275  ],[-10.97514833, -37.046375  ],[-10.975455  , -37.046445  ],[-10.97551333, -37.04656   ],[-10.975465  , -37.046755  ],[-10.97538833, -37.04700833],[-10.97538167, -37.04715667],[-10.975395  , -37.047165  ],[-10.975395  , -37.04718833],[-10.97546167, -37.047255  ],[-10.97562333, -37.04731833],[-10.97568   , -37.04733667],[-10.97568667, -37.04733333]]))
curves.add(fred.Curve([[-10.9088832 , -37.05223326],[-10.90888295, -37.05223344],[-10.90888293, -37.05223377],[-10.90889716, -37.05228277],[-10.9088838 , -37.05228786],[-10.90888731, -37.0522905 ],[-10.90888763, -37.05229111],[-10.908888  , -37.05229165],[-10.9089084 , -37.05243949],[-10.90962054, -37.05257143],[-10.91017822, -37.05261309],[-10.91024486, -37.05249167],[-10.91020435, -37.05254541],[-10.91020688, -37.05252098],[-10.91039359, -37.0525615 ],[-10.91130085, -37.05267804],[-10.91243579, -37.05275943],[-10.91281912, -37.05275623],[-10.91299845, -37.0527907 ],[-10.9134348 , -37.05281276],[-10.91439671, -37.05283108],[-10.91502386, -37.05244318],[-10.91501958, -37.05163338],[-10.91500529, -37.05097196],[-10.91499287, -37.05073336],[-10.91498954, -37.05071972],[-10.91499182, -37.0506611 ],[-10.9149873 , -37.05005493],[-10.91499228, -37.04907819],[-10.91501937, -37.04823088],[-10.91500077, -37.0482491 ],[-10.91498185, -37.04824279],[-10.91496461, -37.04817991],[-10.91496661, -37.04803323],[-10.91496718, -37.04801393],[-10.91516181, -37.04781039],[-10.91523487, -37.04787634],[-10.91533744, -37.04786168],[-10.91608908, -37.0476793 ],[-10.91683097, -37.04750631],[-10.91684096, -37.04754065],[-10.9168218 , -37.0475271 ],[-10.91713138, -37.0474522 ],[-10.91782558, -37.04702501],[-10.91837927, -37.04666542],[-10.91837243, -37.04770506],[-10.91833151, -37.04808085],[-10.91832686, -37.04858431],[-10.91831371, -37.0485561 ],[-10.91831785, -37.04868209],[-10.91833123, -37.04923471],[-10.91833778, -37.04934597],[-10.91833821, -37.04937453],[-10.9183414 , -37.04945326],[-10.91837705, -37.05024349],[-10.91829526, -37.05077556],[-10.91833569, -37.05084079],[-10.91827778, -37.0512477 ],[-10.91834515, -37.05129777],[-10.91838587, -37.0512946 ],[-10.91838725, -37.05129725],[-10.91838729, -37.05129849],[-10.91838757, -37.05129899],[-10.91833538, -37.05183484],[-10.91837977, -37.05302974],[-10.91836768, -37.05378366],[-10.9183838 , -37.05384531],[-10.91839339, -37.05386023],[-10.91838394, -37.05415973],[-10.91839687, -37.05424387],[-10.91839319, -37.0543386 ],[-10.91842129, -37.05440578],[-10.91841793, -37.05442785],[-10.91842099, -37.05443536],[-10.91841976, -37.05443633],[-10.91841974, -37.05486635],[-10.9184573 , -37.05574085],[-10.91843007, -37.05642844],[-10.91842008, -37.0573422 ],[-10.91848418, -37.05820659],[-10.91845611, -37.05822943],[-10.91847223, -37.05835245],[-10.91844448, -37.05878222],[-10.91843396, -37.05891325],[-10.91847597, -37.05968129],[-10.91843197, -37.06092973],[-10.91846829, -37.06220078],[-10.91849185, -37.06260413],[-10.91851347, -37.06291687],[-10.91851883, -37.06352569],[-10.91849179, -37.06346688],[-10.91855271, -37.06342298],[-10.91837489, -37.06332552],[-10.91842185, -37.06352608],[-10.91848517, -37.06354868],[-10.91851322, -37.06357117],[-10.91849923, -37.06425238],[-10.91851269, -37.06524847],[-10.91854919, -37.06559617],[-10.9186087 , -37.0662168 ],[-10.91865213, -37.06716314],[-10.91869049, -37.06771784],[-10.91869273, -37.06772156],[-10.91870221, -37.06778545],[-10.91871375, -37.06803424],[-10.91870789, -37.06805034],[-10.91870006, -37.06807086],[-10.91865094, -37.06854942],[-10.9186561 , -37.06886971],[-10.91865274, -37.06884879],[-10.91866858, -37.06884557],[-10.91866672, -37.06885515],[-10.91866746, -37.06885616],[-10.91866695, -37.0688564 ],[-10.91867187, -37.06886864],[-10.9186002 , -37.06963588],[-10.91843443, -37.07107976],[-10.91822535, -37.07245827],[-10.91823134, -37.0727501 ],[-10.91821917, -37.07331754],[-10.91827723, -37.07328082],[-10.91828368, -37.07328092],[-10.91828464, -37.0732799 ],[-10.91828491, -37.07328002],[-10.91828525, -37.07327996],[-10.91828515, -37.07327946],[-10.91828575, -37.07328014],[-10.91829734, -37.07332701],[-10.91830273, -37.07397364],[-10.91830768, -37.07410884],[-10.91831768, -37.07426703],[-10.91830898, -37.07426671],[-10.91830983, -37.07427203],[-10.91831097, -37.07427023],[-10.9183095 , -37.07426966],[-10.91831156, -37.07429544],[-10.91833997, -37.07488835],[-10.91839598, -37.07608223],[-10.91848014, -37.0775866 ],[-10.91845126, -37.07872471],[-10.91853722, -37.07971351],[-10.91858856, -37.08101469],[-10.91856093, -37.08130972],[-10.91863132, -37.08159399],[-10.91863097, -37.0826843 ],[-10.91877317, -37.0844055 ],[-10.91923248, -37.08607822],[-10.91991725, -37.08772072],[-10.92021272, -37.0885526 ],[-10.92029294, -37.0893145 ],[-10.92080031, -37.09004848],[-10.92127702, -37.09126236],[-10.92168856, -37.09240736],[-10.92182484, -37.09275357],[-10.921844  , -37.09277982],[-10.9219232 , -37.09298061],[-10.92198168, -37.09314702],[-10.92198211, -37.09314559],[-10.92197872, -37.09313835],[-10.92198474, -37.09315315],[-10.92201721, -37.09321265],[-10.92204373, -37.09327943],[-10.92203945, -37.09331084],[-10.92209226, -37.09346881],[-10.9221319 , -37.09360129],[-10.92213489, -37.09361729],[-10.92220012, -37.09376768],[-10.92224383, -37.09389621],[-10.92229875, -37.09406197],[-10.92231259, -37.09412253],[-10.92232288, -37.09412419],[-10.92238078, -37.09428239],[-10.92239589, -37.09428834],[-10.92246472, -37.09448227],[-10.92249559, -37.0945764 ],[-10.92251675, -37.09464045],[-10.92260502, -37.09488827],[-10.92264693, -37.09504228],[-10.92265818, -37.09509681],[-10.92270123, -37.095324  ],[-10.92271163, -37.09555123],[-10.92273434, -37.0956595 ],[-10.92273551, -37.09580649],[-10.92272208, -37.09613894],[-10.92272138, -37.09624293],[-10.92270692, -37.09635799],[-10.92256523, -37.09683054],[-10.92249965, -37.09699533],[-10.92249912, -37.09700496],[-10.9224854 , -37.09704871],[-10.92245081, -37.09715584],[-10.92239153, -37.0973477 ],[-10.92233273, -37.09750338],[-10.92233884, -37.0975099 ],[-10.92233905, -37.09751035],[-10.9222754 , -37.09767894]]))
curves.add(fred.Curve([[-10.89718252, -37.06153744],[-10.89717928, -37.06154183],[-10.89719427, -37.06162388],[-10.89740362, -37.06170339],[-10.89748508, -37.06158833],[-10.89743247, -37.06147908],[-10.89773025, -37.06136172],[-10.89817615, -37.06121401],[-10.89838822, -37.06119026],[-10.89846547, -37.06133559],[-10.89847857, -37.06135715],[-10.8984647 , -37.06142127],[-10.89839607, -37.06173579],[-10.89846346, -37.06178951],[-10.89862291, -37.06147323],[-10.8987976 , -37.06109532],[-10.89912081, -37.0605383 ],[-10.89952792, -37.06019947],[-10.90019806, -37.0601987 ],[-10.90092503, -37.06019532],[-10.90144101, -37.05998673],[-10.90187237, -37.05955134],[-10.90227431, -37.05964394],[-10.90257628, -37.05978713],[-10.90270141, -37.05984812],[-10.90297523, -37.0599222 ],[-10.90341326, -37.06000922],[-10.903924  , -37.06024892],[-10.90438876, -37.06042871],[-10.90480213, -37.06056641],[-10.90491238, -37.06062265],[-10.90491834, -37.06062478],[-10.90491399, -37.06062826],[-10.90491404, -37.06062837],[-10.90492721, -37.06063118],[-10.90527481, -37.06073831],[-10.9056133 , -37.0608926 ],[-10.90596027, -37.06107534],[-10.90633685, -37.06129554],[-10.90638657, -37.06172159],[-10.90632397, -37.06219955],[-10.90626305, -37.06272622],[-10.90620849, -37.06322452],[-10.90613043, -37.06392946],[-10.9059863 , -37.06466479],[-10.90581971, -37.06542344],[-10.90538927, -37.06614526],[-10.90523179, -37.06684845],[-10.90530215, -37.06775635],[-10.90532315, -37.06854295],[-10.90530999, -37.06898251],[-10.90534177, -37.06945143],[-10.90529468, -37.06988133],[-10.90520903, -37.07045209],[-10.90511702, -37.0710441 ],[-10.90506513, -37.07151597],[-10.90513088, -37.07168569],[-10.90535913, -37.07171172],[-10.905518  , -37.07172697],[-10.9058173 , -37.07175409],[-10.90619432, -37.0718065 ],[-10.90640428, -37.07183294],[-10.90679616, -37.0718898 ],[-10.90714975, -37.07190534],[-10.90731855, -37.07193713],[-10.90747746, -37.07196974],[-10.90768244, -37.07200372],[-10.90796666, -37.07202936],[-10.90835309, -37.0720759 ],[-10.90850061, -37.07228625],[-10.90847072, -37.07253407],[-10.908419  , -37.07285023],[-10.90837452, -37.0731016 ],[-10.90808951, -37.07305897],[-10.90780501, -37.07299979],[-10.90771421, -37.07303244],[-10.90773156, -37.07304193],[-10.90773216, -37.07303937],[-10.9077349 , -37.07304259],[-10.90773444, -37.07304379],[-10.90773465, -37.07304339],[-10.90773488, -37.07304307],[-10.90773492, -37.07304313],[-10.90773493, -37.07304317],[-10.90773493, -37.07304317],[-10.90773491, -37.07304324],[-10.9077349 , -37.07304328],[-10.90773485, -37.07304348],[-10.90773478, -37.07304368],[-10.90773465, -37.07304389],[-10.90769344, -37.07310355],[-10.90772073, -37.07317623],[-10.90775551, -37.07316745],[-10.90765891, -37.07305151],[-10.90766044, -37.07302627],[-10.90762726, -37.07312806],[-10.90766993, -37.07314672],[-10.9075945 , -37.07307496],[-10.90757843, -37.07311326],[-10.90770215, -37.07321334],[-10.90775981, -37.07328701],[-10.90773305, -37.07338943],[-10.90780396, -37.07349019],[-10.90774554, -37.07334824],[-10.90778599, -37.07334086],[-10.90772423, -37.07345392],[-10.90763186, -37.07331128],[-10.90771947, -37.07333281],[-10.90776828, -37.07331933],[-10.90773922, -37.07328206],[-10.90775751, -37.07326267],[-10.9079168 , -37.07324445],[-10.9078881 , -37.07325607],[-10.90794507, -37.07347514],[-10.90796649, -37.07346901],[-10.90790874, -37.0734491 ],[-10.90785067, -37.07350005],[-10.90783778, -37.07342061],[-10.90792676, -37.07337003],[-10.9078536 , -37.07325889],[-10.90786763, -37.07320771],[-10.90776746, -37.0731938 ],[-10.90768463, -37.07312922],[-10.90778169, -37.07319702],[-10.907764  , -37.07329961],[-10.90766187, -37.07325587],[-10.90764612, -37.07318198],[-10.90761881, -37.07315654],[-10.90784353, -37.07341004],[-10.90783896, -37.07341802],[-10.90780446, -37.07331564],[-10.90775324, -37.07321701],[-10.90768318, -37.07321721],[-10.90765836, -37.07319478],[-10.90766332, -37.0731867 ],[-10.9076861 , -37.07317843],[-10.90771507, -37.07319776],[-10.90772807, -37.07322532],[-10.907703  , -37.07324562],[-10.90771112, -37.07324491],[-10.90772459, -37.07324626],[-10.90772836, -37.07324979],[-10.9077341 , -37.07325302],[-10.90773526, -37.07325474],[-10.90773536, -37.07325477],[-10.90773411, -37.0732529 ],[-10.90773286, -37.07325047],[-10.9077335 , -37.07325118],[-10.9077332 , -37.07325138],[-10.9077863 , -37.07326522],[-10.90776883, -37.07326307],[-10.90778074, -37.07327006],[-10.90777981, -37.07327252],[-10.90779339, -37.07328283],[-10.90779073, -37.07328262],[-10.90779103, -37.07328268],[-10.90779144, -37.07328265],[-10.90779399, -37.07328369],[-10.90779412, -37.0732838 ],[-10.90779392, -37.07328378],[-10.90779372, -37.07328371],[-10.90779355, -37.07328362],[-10.90779315, -37.07328342],[-10.9077933 , -37.07328338],[-10.90779535, -37.07328394],[-10.90779754, -37.07328453],[-10.90779809, -37.07328453],[-10.90779863, -37.07328463],[-10.90779888, -37.07328472],[-10.90779837, -37.07328456],[-10.90779823, -37.07328453],[-10.90779813, -37.07328447],[-10.90779871, -37.07328465],[-10.90779818, -37.07328458],[-10.90779536, -37.0732837 ],[-10.90779354, -37.07328315],[-10.90779377, -37.07328328],[-10.90779385, -37.07328332],[-10.90779217, -37.07328266],[-10.90778796, -37.07328124],[-10.90778867, -37.07328162],[-10.90778948, -37.07328197],[-10.90778983, -37.07328208],[-10.90778961, -37.07328202],[-10.90778983, -37.07328209],[-10.90779017, -37.07328225],[-10.90779036, -37.07328234],[-10.90778999, -37.07328225],[-10.90778977, -37.0732822 ],[-10.90778906, -37.0732821 ],[-10.90778846, -37.07328193],[-10.90778859, -37.07328198],[-10.90778818, -37.07328181],[-10.90778771, -37.07328161],[-10.90778745, -37.07328153],[-10.90778749, -37.07328161],[-10.90778761, -37.07328166],[-10.90778761, -37.07328166],[-10.90778771, -37.07328169],[-10.90778771, -37.07328169],[-10.90778766, -37.07328172],[-10.90778768, -37.07328182],[-10.9077877 , -37.07328185],[-10.9077872 , -37.07328168],[-10.90778592, -37.07328124],[-10.90778532, -37.07328103],[-10.90778428, -37.0732805 ]]))
curves.add(fred.Curve([[-10.90772872, -37.0858016 ],[-10.90769264, -37.0858147 ],[-10.90767491, -37.08591119],[-10.90776771, -37.0860094 ],[-10.90804474, -37.08601345],[-10.9081789 , -37.08604445],[-10.90830611, -37.08608333],[-10.90835044, -37.08628076],[-10.90824216, -37.08658882],[-10.907988  , -37.08672575],[-10.90764737, -37.08667506],[-10.90733714, -37.08664967],[-10.90726592, -37.08669627],[-10.90726345, -37.08669419],[-10.9072653 , -37.08669291],[-10.90725668, -37.08671934],[-10.90725482, -37.08672268],[-10.90724792, -37.08672344],[-10.90724662, -37.08672427],[-10.90724695, -37.08674937],[-10.90745213, -37.08680619],[-10.90780427, -37.08683785],[-10.90822772, -37.08697207],[-10.90827724, -37.08736113],[-10.90822139, -37.08787732],[-10.90813868, -37.08846487],[-10.90805041, -37.0890975 ],[-10.90791445, -37.08976604],[-10.90776114, -37.09055038],[-10.90762976, -37.09124773],[-10.9074972 , -37.09174274],[-10.90706666, -37.09184062],[-10.90697036, -37.0913615 ],[-10.90740064, -37.09117438],[-10.90792618, -37.09120379],[-10.90859458, -37.09116682],[-10.90908798, -37.09101294],[-10.90958136, -37.09088523],[-10.91026623, -37.09066052],[-10.91102728, -37.09044706],[-10.91182623, -37.0901804 ],[-10.91256112, -37.08990812],[-10.91332092, -37.08958603],[-10.91392742, -37.08926387],[-10.91451706, -37.0888855 ],[-10.91518566, -37.08836594],[-10.91593118, -37.08771103],[-10.91655531, -37.08714374],[-10.91717904, -37.08657166],[-10.91771477, -37.08609078],[-10.91829564, -37.08555403],[-10.91888463, -37.08504173],[-10.91954058, -37.08444507],[-10.92030723, -37.08378391],[-10.92081163, -37.08329865],[-10.92139016, -37.08277598],[-10.92190766, -37.08227437],[-10.92238033, -37.08188744],[-10.92284275, -37.08149623],[-10.92306874, -37.08135072],[-10.92310144, -37.08134244],[-10.9231371 , -37.08132403],[-10.92321829, -37.08123696],[-10.92353141, -37.08106698],[-10.92397256, -37.08088936],[-10.92439651, -37.08071703],[-10.92494427, -37.08055035],[-10.92560991, -37.0804511 ],[-10.92631605, -37.08045015],[-10.92715145, -37.08057647],[-10.92791779, -37.08079785],[-10.92873752, -37.08112506],[-10.92927668, -37.08134492],[-10.92993042, -37.08158931],[-10.93058405, -37.08183858],[-10.93107363, -37.0820129 ],[-10.93168043, -37.08222898],[-10.93242473, -37.08252733],[-10.93330835, -37.08285164],[-10.93454631, -37.08308465],[-10.93566415, -37.08309616],[-10.93676152, -37.08280934],[-10.93754135, -37.0824557 ],[-10.93812872, -37.08207358],[-10.93839337, -37.08189116],[-10.93838866, -37.08184789],[-10.93838412, -37.08184038],[-10.93837805, -37.08181947],[-10.93837081, -37.08182115],[-10.93839979, -37.08180553],[-10.93862847, -37.08164704],[-10.93895492, -37.08135762],[-10.93939338, -37.08093719],[-10.93980636, -37.08053803],[-10.940309  , -37.08004823],[-10.9407109 , -37.07963531],[-10.94123128, -37.07912858],[-10.94174922, -37.07863201],[-10.94232928, -37.07804908],[-10.94302297, -37.07737554],[-10.94356197, -37.07682755],[-10.94419303, -37.07620503],[-10.94496112, -37.07547244],[-10.9456835 , -37.07476836],[-10.94630675, -37.07417086],[-10.94694075, -37.07354591],[-10.94740689, -37.07312202],[-10.94769489, -37.07282812],[-10.94792436, -37.07295227],[-10.94788824, -37.07313781],[-10.94773945, -37.07322475],[-10.94763357, -37.07317975],[-10.94738632, -37.07300235],[-10.94713075, -37.07270575],[-10.94683891, -37.07229992],[-10.94664963, -37.07205765],[-10.94647365, -37.0718313 ],[-10.94630797, -37.07157862],[-10.94627212, -37.07141242],[-10.94652577, -37.071189  ],[-10.94677912, -37.0709791 ],[-10.94689561, -37.07097204],[-10.94684108, -37.07105775],[-10.94677036, -37.07111451],[-10.94677673, -37.07122847],[-10.94687835, -37.07132281],[-10.94693879, -37.07144081],[-10.94687343, -37.07148984],[-10.9468418 , -37.0715474 ],[-10.94687278, -37.07156909],[-10.94686136, -37.07156657]]))
curves.add(fred.Curve([[-10.92366833, -37.04470333],[-10.92388667, -37.04459   ],[-10.9243    , -37.04443667],[-10.92942   , -37.04500667],[-10.92949833, -37.04511667],[-10.92964167, -37.045355  ],[-10.92979667, -37.04559833],[-10.92994833, -37.04581333],[-10.93014333, -37.04608333],[-10.930215  , -37.04626   ],[-10.930225  , -37.04630833],[-10.93017167, -37.046335  ],[-10.93017667, -37.04637833],[-10.93021167, -37.046435  ],[-10.93022167, -37.046565  ],[-10.930385  , -37.046725  ],[-10.93042333, -37.04682667],[-10.93045333, -37.04683833],[-10.930475  , -37.04681833],[-10.93047167, -37.04683   ],[-10.93051333, -37.04682833],[-10.93051333, -37.04682833],[-10.930535  , -37.04677833],[-10.93051167, -37.04674667],[-10.930545  , -37.04672333],[-10.93058167, -37.046735  ],[-10.93061   , -37.04673167],[-10.930675  , -37.04672   ],[-10.93091167, -37.04697667],[-10.93120167, -37.04733833],[-10.93131   , -37.04729333],[-10.93144833, -37.047365  ],[-10.93161333, -37.04756167],[-10.93188833, -37.047885  ],[-10.93227333, -37.04835333],[-10.93259   , -37.0487    ],[-10.932965  , -37.04894333],[-10.93307667, -37.04904833],[-10.93308   , -37.04904833],[-10.93318667, -37.04907333],[-10.933335  , -37.04922667],[-10.93362   , -37.04955333],[-10.93400333, -37.04988667],[-10.93964   , -37.05037833],[-10.93996   , -37.05036   ],[-10.94004333, -37.05033167],[-10.94005667, -37.05033333],[-10.94004   , -37.05036   ],[-10.94003   , -37.05035667],[-10.94003   , -37.05035667],[-10.94009   , -37.050395  ],[-10.94013333, -37.05041167],[-10.940015  , -37.050435  ],[-10.93998167, -37.05052333],[-10.939915  , -37.05052167],[-10.94105   , -37.05060833],[-10.94120167, -37.05051667],[-10.941275  , -37.050325  ],[-10.94148667, -37.04996667],[-10.94171833, -37.049545  ],[-10.94205167, -37.049065  ],[-10.942365  , -37.04862167],[-10.94272167, -37.04815667],[-10.943265  , -37.04768167],[-10.94363333, -37.04743667],[-10.94393333, -37.04705833],[-10.94407833, -37.04694   ],[-10.944025  , -37.04700667],[-10.94415667, -37.04692   ],[-10.944685  , -37.04639667],[-10.94502   , -37.046285  ],[-10.945335  , -37.04617167],[-10.94551167, -37.04608667],[-10.94556167, -37.04607167],[-10.94560667, -37.04605833],[-10.94565333, -37.04604667],[-10.94564   , -37.04606833],[-10.945655  , -37.04607   ],[-10.94581333, -37.04604   ],[-10.94615833, -37.04593   ],[-10.94657333, -37.04581833],[-10.94716667, -37.04568833],[-10.94778   , -37.04560833],[-10.94858667, -37.04545833],[-10.949345  , -37.04543   ],[-10.95012   , -37.04549833],[-10.95087   , -37.045625  ],[-10.95162167, -37.045695  ],[-10.95236333, -37.045695  ],[-10.95323   , -37.04548   ],[-10.953905  , -37.045265  ],[-10.95445167, -37.04512333],[-10.95491167, -37.04514167],[-10.95536167, -37.04527333],[-10.95569833, -37.04549   ],[-10.956075  , -37.04541667],[-10.95645   , -37.04527833],[-10.9566    , -37.04522167],[-10.956585  , -37.04525   ],[-10.95676167, -37.04518667],[-10.95703667, -37.045075  ],[-10.95743167, -37.04488667],[-10.95790667, -37.04466333],[-10.95842167, -37.04445   ],[-10.959     , -37.04420167],[-10.95958167, -37.04394   ],[-10.96025667, -37.04360167],[-10.96096   , -37.04327667],[-10.96165333, -37.04298667],[-10.96249667, -37.04261833],[-10.96316   , -37.042405  ],[-10.96383667, -37.04225667],[-10.96469   , -37.04225667],[-10.96537333, -37.04236167],[-10.96593   , -37.0425    ],[-10.96629333, -37.04257167],[-10.96652167, -37.04256167],[-10.96662333, -37.04262833],[-10.96689333, -37.04273833],[-10.96724833, -37.04289833],[-10.967395  , -37.04297333],[-10.96738833, -37.04269   ],[-10.967495  , -37.04238833],[-10.96758   , -37.04203333],[-10.96761667, -37.04176333],[-10.96762   , -37.04172333],[-10.96762   , -37.04172   ],[-10.96760667, -37.04174   ],[-10.96763   , -37.04173833],[-10.96764   , -37.04174667],[-10.96767167, -37.04174333],[-10.96767333, -37.04172833],[-10.96771   , -37.04163167],[-10.96771167, -37.04162833],[-10.96771167, -37.04162167],[-10.96769   , -37.04167   ],[-10.96768833, -37.04166   ],[-10.96767667, -37.04166   ],[-10.96767167, -37.04166333],[-10.96770167, -37.04162167],[-10.96783667, -37.04138333],[-10.96813833, -37.04147667],[-10.96852833, -37.04157   ],[-10.96927   , -37.04167833],[-10.96989833, -37.04186   ],[-10.970665  , -37.04203333],[-10.97124167, -37.04213167],[-10.97162   , -37.042135  ],[-10.97151667, -37.04214667],[-10.97125167, -37.04228167],[-10.971275  , -37.04231833],[-10.97151167, -37.04235333],[-10.972     , -37.04243167],[-10.97252   , -37.04254333],[-10.97287   , -37.04260667],[-10.97314   , -37.04266667],[-10.97335667, -37.04272333],[-10.973515  , -37.04261833],[-10.97361167, -37.04260167],[-10.97369   , -37.0426    ],[-10.97369333, -37.04260167],[-10.97385833, -37.042605  ],[-10.97423167, -37.042715  ],[-10.97472333, -37.04288   ],[-10.97553   , -37.04310833],[-10.97617667, -37.04323667],[-10.97693833, -37.04342   ],[-10.97756167, -37.04356   ],[-10.97825667, -37.043745  ],[-10.97886333, -37.0439    ],[-10.97936   , -37.04401   ],[-10.97961333, -37.04409667],[-10.97974833, -37.044205  ],[-10.97982667, -37.044585  ],[-10.97963333, -37.04517   ],[-10.979385  , -37.04588667],[-10.97914833, -37.04651   ],[-10.97898   , -37.04679667],[-10.97893333, -37.04698667],[-10.97909667, -37.04712833],[-10.97912667, -37.047185  ],[-10.97908667, -37.047215  ],[-10.979085  , -37.04724333]]))
curves.add(fred.Curve([[-10.89688234, -37.05339968],[-10.89689515, -37.05332973],[-10.89688042, -37.05333247],[-10.89688029, -37.05333484],[-10.89688058, -37.05333512],[-10.89688073, -37.05333528],[-10.8968805 , -37.05333564],[-10.89688045, -37.0533357 ],[-10.89688034, -37.05333584],[-10.89688022, -37.05333665],[-10.89687998, -37.05333665],[-10.8968798 , -37.05333672],[-10.8968771 , -37.05333475],[-10.89686571, -37.05335033],[-10.89686453, -37.05338245],[-10.89690625, -37.053412  ],[-10.89714799, -37.05345149],[-10.89731552, -37.05353814],[-10.89722363, -37.0537264 ],[-10.89712402, -37.05392281],[-10.89700272, -37.05391454],[-10.89669729, -37.05377559],[-10.89637283, -37.05373788],[-10.89628634, -37.05376364],[-10.89625468, -37.05380163],[-10.89626958, -37.05404301],[-10.89630975, -37.05439376],[-10.8963886 , -37.05509832],[-10.89644765, -37.05538955],[-10.8965611 , -37.05575682],[-10.89665346, -37.05613108],[-10.89677345, -37.05672517],[-10.89689319, -37.05728734],[-10.89692781, -37.05739169],[-10.89695023, -37.05753143],[-10.89702285, -37.05769891],[-10.8971346 , -37.057792  ],[-10.89734674, -37.05797843],[-10.89746565, -37.05809327],[-10.89745873, -37.05809087],[-10.89746601, -37.05809045],[-10.89746603, -37.05809043],[-10.89746598, -37.0580906 ],[-10.89746595, -37.05809065],[-10.89750503, -37.05812029],[-10.89776568, -37.05832533],[-10.89815091, -37.05824039],[-10.89867417, -37.05785587],[-10.89905788, -37.05755751],[-10.89965258, -37.05710931],[-10.90031391, -37.05666742],[-10.90072693, -37.05639989],[-10.90117312, -37.0560867 ],[-10.90167986, -37.05570044],[-10.90223155, -37.05532095],[-10.90267435, -37.05501594],[-10.90281205, -37.05492839],[-10.90281977, -37.05492466],[-10.9028111 , -37.05493056],[-10.90281184, -37.05493098],[-10.90281179, -37.05493125],[-10.90281184, -37.05493146],[-10.9028119 , -37.05493155],[-10.90281195, -37.05493173],[-10.9028117 , -37.05493193],[-10.90281944, -37.05492637],[-10.90318115, -37.05466438],[-10.90361849, -37.05433795],[-10.90395364, -37.05420787],[-10.90395573, -37.05453654],[-10.90394111, -37.05505263],[-10.90394185, -37.05587736],[-10.90392791, -37.05664783],[-10.90389758, -37.05750308],[-10.90388737, -37.05786664],[-10.90403   , -37.05799884],[-10.90431907, -37.05799646],[-10.90463524, -37.05800058],[-10.90492265, -37.0580025 ],[-10.90509223, -37.0580149 ],[-10.90513417, -37.05805725],[-10.90519558, -37.05826165],[-10.90527994, -37.05877507],[-10.90535515, -37.05906472],[-10.90619649, -37.05905899],[-10.90641173, -37.06018562],[-10.90644041, -37.06067098],[-10.90642897, -37.06086107],[-10.90641925, -37.06087587],[-10.90642255, -37.06088143],[-10.90642329, -37.06088235],[-10.90642264, -37.06088388],[-10.90642211, -37.06090121],[-10.90642028, -37.06112732],[-10.90649051, -37.06164138],[-10.90728268, -37.06183192],[-10.90753491, -37.06201037],[-10.90761894, -37.06208487],[-10.90763191, -37.06209038],[-10.90764293, -37.06208956],[-10.90764576, -37.06209575],[-10.90764926, -37.06209673],[-10.90764498, -37.06210186],[-10.90764444, -37.0621018 ],[-10.9076441 , -37.06210176],[-10.90764407, -37.06210173],[-10.90791968, -37.06228879],[-10.90840258, -37.06261833],[-10.90885022, -37.06293506],[-10.90941396, -37.06335066],[-10.91002845, -37.06379331],[-10.91021656, -37.06385412],[-10.91021107, -37.06382905],[-10.91020892, -37.0638263 ],[-10.91020903, -37.06382629],[-10.91020921, -37.06382624],[-10.91020941, -37.06382628],[-10.91020941, -37.06382636],[-10.91020942, -37.0638264 ],[-10.91020943, -37.06382641],[-10.91021067, -37.06382699],[-10.91051354, -37.06396407],[-10.91086507, -37.06407459],[-10.91112765, -37.06418145],[-10.91112402, -37.06418525],[-10.91112739, -37.064179  ],[-10.91113069, -37.06417743],[-10.91113068, -37.06417746],[-10.91113069, -37.06417745],[-10.91113068, -37.06417741],[-10.91113062, -37.06417743],[-10.91113027, -37.06417782],[-10.91113022, -37.06417796],[-10.91113014, -37.06417815],[-10.91113018, -37.06417815],[-10.9111301 , -37.06417819],[-10.91113003, -37.06417818],[-10.91113018, -37.06417821],[-10.91121402, -37.06421624],[-10.91152921, -37.06426605],[-10.91195342, -37.06430617],[-10.91243864, -37.06431631],[-10.91301115, -37.06431624],[-10.91373203, -37.06430726],[-10.9143199 , -37.06424356],[-10.91526178, -37.06407409],[-10.91568718, -37.06396253],[-10.91601655, -37.06391468],[-10.91614016, -37.06400304],[-10.91613907, -37.06426894],[-10.91611837, -37.0645705 ],[-10.91620218, -37.06472138],[-10.91657107, -37.0648288 ],[-10.9170171 , -37.06493571],[-10.91728016, -37.06502682],[-10.91734995, -37.06509569],[-10.91735968, -37.06534346],[-10.91731253, -37.06557038],[-10.91730704, -37.06556575],[-10.91730599, -37.06555167]]))

clustering = fred.discrete_klcenter(6, 8, curves, local_search = 10, fast_simplification=True)
clustering.optimize_centers(curves)

fred.plot_clustering(clustering, curves)
```
