import kms = require('@aws-cdk/aws-kms');
import lambda = require('@aws-cdk/aws-lambda');
import s3 = require('@aws-cdk/aws-s3');
import sns = require('@aws-cdk/aws-sns');
import { CfnReceiptRule } from './ses.generated';
/**
 * Properties for a receipt rule action.
 */
export interface ReceiptRuleActionProps {
    /**
     * Adds a header to the received email.
     */
    readonly addHeaderAction?: CfnReceiptRule.AddHeaderActionProperty;
    /**
     * Rejects the received email by returning a bounce response to the sender and,
     * optionally, publishes a notification to Amazon SNS.
     */
    readonly bounceAction?: CfnReceiptRule.BounceActionProperty;
    /**
     * Calls an AWS Lambda function, and optionally, publishes a notification to
     * Amazon SNS.
     */
    readonly lambdaAction?: CfnReceiptRule.LambdaActionProperty;
    /**
     * Saves the received message to an Amazon S3 bucket and, optionally, publishes
     * a notification to Amazon SNS.
     */
    readonly s3Action?: CfnReceiptRule.S3ActionProperty;
    /**
     * Publishes the email content within a notification to Amazon SNS.
     */
    readonly snsAction?: CfnReceiptRule.SNSActionProperty;
    /**
     * Terminates the evaluation of the receipt rule set and optionally publishes a
     * notification to Amazon SNS.
     */
    readonly stopAction?: CfnReceiptRule.StopActionProperty;
    /**
     * Calls Amazon WorkMail and, optionally, publishes a notification to Amazon SNS.
     */
    readonly workmailAction?: CfnReceiptRule.WorkmailActionProperty;
}
/**
 * An abstract action for a receipt rule.
 */
export interface IReceiptRuleAction {
    /**
     * Renders the action specification
     */
    render(): ReceiptRuleActionProps;
}
/**
 * Construction properties for a ReceiptRuleAddHeaderAction.
 */
export interface ReceiptRuleAddHeaderActionProps {
    /**
     * The name of the header to add. Must be between 1 and 50 characters,
     * inclusive, and consist of alphanumeric (a-z, A-Z, 0-9) characters
     * and dashes only.
     */
    readonly name: string;
    /**
     * The value of the header to add. Must be less than 2048 characters,
     * and must not contain newline characters ("\r" or "\n").
     */
    readonly value: string;
}
/**
 * Adds a header to the received email
 */
export declare class ReceiptRuleAddHeaderAction implements IReceiptRuleAction {
    private readonly name;
    private readonly value;
    constructor(props: ReceiptRuleAddHeaderActionProps);
    render(): ReceiptRuleActionProps;
}
/**
 * Construction properties for a ReceiptRuleBounceActionTemplate.
 */
export interface ReceiptRuleBounceActionTemplateProps {
    /**
     * Human-readable text to include in the bounce message.
     */
    readonly message: string;
    /**
     * The SMTP reply code, as defined by RFC 5321.
     *
     * @see https://tools.ietf.org/html/rfc5321
     */
    readonly smtpReplyCode: string;
    /**
     * The SMTP enhanced status code, as defined by RFC 3463.
     *
     * @see https://tools.ietf.org/html/rfc3463
     */
    readonly statusCode?: string;
}
/**
 * A bounce action template.
 */
export declare class ReceiptRuleBounceActionTemplate {
    static readonly MailboxDoesNotExist: ReceiptRuleBounceActionTemplate;
    static readonly MessageTooLarge: ReceiptRuleBounceActionTemplate;
    static readonly MailboxFull: ReceiptRuleBounceActionTemplate;
    static readonly MessageContentRejected: ReceiptRuleBounceActionTemplate;
    static readonly TemporaryFailure: ReceiptRuleBounceActionTemplate;
    readonly message: string;
    readonly smtpReplyCode: string;
    readonly statusCode?: string;
    constructor(props: ReceiptRuleBounceActionTemplateProps);
}
/**
 * Construction properties for a ReceiptRuleBounceAction.
 */
export interface ReceiptRuleBounceActionProps {
    /**
     * The template containing the message, reply code and status code.
     */
    readonly template: ReceiptRuleBounceActionTemplate;
    /**
     * The email address of the sender of the bounced email. This is the address
     * from which the bounce message will be sent.
     */
    readonly sender: string;
    /**
     * The SNS topic to notify when the bounce action is taken.
     *
     * @default no notification
     */
    readonly topic?: sns.ITopic;
}
/**
 * Rejects the received email by returning a bounce response to the sender and,
 * optionally, publishes a notification to Amazon SNS.
 */
export declare class ReceiptRuleBounceAction implements IReceiptRuleAction {
    private readonly props;
    constructor(props: ReceiptRuleBounceActionProps);
    render(): ReceiptRuleActionProps;
}
/**
 * The type of invocation to use for a Lambda Action.
 */
export declare enum LambdaInvocationType {
    /**
     * The function will be invoked asynchronously.
     */
    Event = "Event",
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    RequestResponse = "RequestResponse"
}
/**
 * Construction properties for a ReceiptRuleLambdaAction.
 */
export interface ReceiptRuleLambdaActionProps {
    /**
     * The Lambda function to invoke.
     */
    readonly function: lambda.IFunction;
    /**
     * The invocation type of the Lambda function.
     *
     * @default Event
     */
    readonly invocationType?: LambdaInvocationType;
    /**
     * The SNS topic to notify when the Lambda action is taken.
     *
     * @default no notification
     */
    readonly topic?: sns.ITopic;
}
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
export declare class ReceiptRuleLambdaAction implements IReceiptRuleAction {
    private readonly props;
    constructor(props: ReceiptRuleLambdaActionProps);
    render(): ReceiptRuleActionProps;
}
/**
 * Construction properties for a ReceiptRuleS3Action.
 */
export interface ReceiptRuleS3ActionProps {
    /**
     * The S3 bucket that incoming email will be saved to.
     */
    readonly bucket: s3.IBucket;
    /**
     * The master key that SES should use to encrypt your emails before saving
     * them to the S3 bucket.
     *
     * @default no encryption
     */
    readonly kmsKey?: kms.IKey;
    /**
     * The key prefix of the S3 bucket.
     *
     * @default no prefix
     */
    readonly objectKeyPrefix?: string;
    /**
     * The SNS topic to notify when the S3 action is taken.
     *
     * @default no notification
     */
    readonly topic?: sns.ITopic;
}
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
export declare class ReceiptRuleS3Action implements IReceiptRuleAction {
    private readonly props;
    constructor(props: ReceiptRuleS3ActionProps);
    render(): ReceiptRuleActionProps;
}
/**
 * The type of email encoding to use for a SNS action.
 */
export declare enum EmailEncoding {
    /**
     * Base 64
     */
    Base64 = "Base64",
    /**
     * UTF-8
     */
    UTF8 = "UTF-8"
}
/**
 * Construction properties for a ReceiptRuleSnsAction.
 */
export interface ReceiptRuleSnsActionProps {
    /**
     * The encoding to use for the email within the Amazon SNS notification.
     *
     * @default UTF-8
     */
    readonly encoding?: EmailEncoding;
    /**
     * The SNS topic to notify.
     */
    readonly topic: sns.ITopic;
}
/**
 * Publishes the email content within a notification to Amazon SNS.
 */
export declare class ReceiptRuleSnsAction implements IReceiptRuleAction {
    private readonly props;
    constructor(props: ReceiptRuleSnsActionProps);
    render(): ReceiptRuleActionProps;
}
/**
 * Construction properties for a ReceiptRuleStopAction.
 */
export interface ReceiptRuleStopActionProps {
    /**
     * The SNS topic to notify when the stop action is taken.
     */
    readonly topic?: sns.ITopic;
}
/**
 * Terminates the evaluation of the receipt rule set and optionally publishes a
 * notification to Amazon SNS.
 */
export declare class ReceiptRuleStopAction implements IReceiptRuleAction {
    private readonly props?;
    constructor(props?: ReceiptRuleStopActionProps | undefined);
    render(): ReceiptRuleActionProps;
}
