"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const receipt_rule_1 = require("./receipt-rule");
const ses_generated_1 = require("./ses.generated");
/**
 * A new or imported receipt rule set.
 */
class ReceiptRuleSetBase extends cdk_1.Resource {
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id, options) {
        this.lastAddedRule = new receipt_rule_1.ReceiptRule(this, id, {
            after: this.lastAddedRule ? this.lastAddedRule : undefined,
            ruleSet: this,
            ...options
        });
        return this.lastAddedRule;
    }
    /**
     * Adds a drop spam rule
     */
    addDropSpamRule() {
        const dropSpam = new receipt_rule_1.DropSpamReceiptRule(this, 'DropSpam', {
            ruleSet: this
        });
        this.lastAddedRule = dropSpam.rule;
    }
}
/**
 * A new receipt rule set.
 */
class ReceiptRuleSet extends ReceiptRuleSetBase {
    /**
     * Import an exported receipt rule set.
     */
    static fromReceiptRuleSetName(scope, id, receiptRuleSetName) {
        class Import extends ReceiptRuleSetBase {
            constructor() {
                super(...arguments);
                this.receiptRuleSetName = receiptRuleSetName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new ses_generated_1.CfnReceiptRuleSet(this, 'Resource', {
            ruleSetName: props ? props.name : undefined
        });
        this.receiptRuleSetName = resource.receiptRuleSetName;
        if (props) {
            const rules = props.rules || [];
            rules.forEach((ruleOption, idx) => this.addRule(`Rule${idx}`, ruleOption));
            if (props.dropSpam) {
                this.addDropSpamRule();
            }
        }
    }
}
exports.ReceiptRuleSet = ReceiptRuleSet;
//# sourceMappingURL=data:application/json;base64,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