"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const cdk_1 = require("@aws-cdk/cdk");
const receipt_rule_action_1 = require("./receipt-rule-action");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["Optional"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["Require"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends cdk_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.renderedActions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: new cdk_1.Token(() => this.getRenderedActions()),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: props.name,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy
            },
            ruleSetName: props.ruleSet.receiptRuleSetName
        });
        this.receiptRuleName = resource.receiptRuleName;
        if (props.actions) {
            props.actions.forEach(action => this.addAction(action));
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends cdk_1.Construct {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        const renderedAction = action.render();
        this.renderedActions.push(renderedAction);
    }
    getRenderedActions() {
        if (this.renderedActions.length === 0) {
            return undefined;
        }
        return this.renderedActions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NodeJS810,
            handler: 'index.handler',
            code: lambda.Code.inline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16'
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                new receipt_rule_action_1.ReceiptRuleLambdaAction({
                    function: fn,
                    invocationType: receipt_rule_action_1.LambdaInvocationType.RequestResponse
                })
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log("SES Notification:\n", JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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