"use strict";
const assert_1 = require("@aws-cdk/assert");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const cdk_1 = require("@aws-cdk/cdk");
// tslint:disable:max-line-length
const lib_1 = require("../lib");
module.exports = {
    'can add an add header action'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleAddHeaderAction({
                            name: 'X-My-Header',
                            value: 'value'
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        AddHeaderAction: {
                            HeaderName: 'X-My-Header',
                            HeaderValue: 'value'
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    },
    'fails when header name is invalid'(test) {
        const stack = new cdk_1.Stack();
        test.throws(() => new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleAddHeaderAction({
                            name: 'He@der',
                            value: 'value'
                        })
                    ]
                }
            ]
        }), /`name`/);
        test.done();
    },
    'fails when header value is invalid'(test) {
        const stack = new cdk_1.Stack();
        const ruleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        test.throws(() => ruleSet.addRule('Rule', {
            actions: [
                new lib_1.ReceiptRuleAddHeaderAction({
                    name: 'Header',
                    value: `va
          lu`
                })
            ]
        }), /`value`/);
        test.done();
    },
    'can add a bounce action'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleBounceAction({
                            sender: 'noreply@aws.com',
                            template: lib_1.ReceiptRuleBounceActionTemplate.MessageContentRejected,
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        BounceAction: {
                            Message: 'Message content rejected',
                            Sender: 'noreply@aws.com',
                            SmtpReplyCode: '500',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            },
                            StatusCode: '5.6.1',
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    },
    'can add a lambda action'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline(''),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810
        });
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleLambdaAction({
                            function: fn,
                            invocationType: lib_1.LambdaInvocationType.RequestResponse,
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        LambdaAction: {
                            FunctionArn: {
                                'Fn::GetAtt': [
                                    'Function76856677',
                                    'Arn'
                                ]
                            },
                            InvocationType: 'RequestResponse',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            }
                        }
                    },
                ],
                Enabled: true
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'ses.amazonaws.com',
            SourceAccount: {
                Ref: 'AWS::AccountId'
            }
        }));
        test.done();
    },
    'can add a s3 action'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const kmsKey = new kms.Key(stack, 'Key');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleS3Action({
                            bucket,
                            kmsKey,
                            objectKeyPrefix: 'emails/',
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        S3Action: {
                            BucketName: {
                                Ref: 'Bucket83908E77'
                            },
                            KmsKeyArn: {
                                'Fn::GetAtt': [
                                    'Key961B73FD',
                                    'Arn'
                                ]
                            },
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            },
                            ObjectKeyPrefix: 'emails/'
                        }
                    }
                ],
                Enabled: true
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
            Bucket: {
                Ref: 'Bucket83908E77'
            },
            PolicyDocument: {
                Statement: [
                    {
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                'aws:Referer': {
                                    Ref: 'AWS::AccountId'
                                }
                            }
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'ses.',
                                        {
                                            Ref: 'AWS::URLSuffix'
                                        }
                                    ]
                                ]
                            }
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'Arn'
                                        ]
                                    },
                                    '/emails/*'
                                ]
                            ]
                        }
                    }
                ],
                Version: '2012-10-17'
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: [
                            'kms:Create*',
                            'kms:Describe*',
                            'kms:Enable*',
                            'kms:List*',
                            'kms:Put*',
                            'kms:Update*',
                            'kms:Revoke*',
                            'kms:Disable*',
                            'kms:Get*',
                            'kms:Delete*',
                            'kms:ScheduleKeyDeletion',
                            'kms:CancelKeyDeletion'
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':root'
                                    ]
                                ]
                            }
                        },
                        Resource: '*'
                    },
                    {
                        Action: [
                            'km:Encrypt',
                            'kms:GenerateDataKey'
                        ],
                        Condition: {
                            Null: {
                                'kms:EncryptionContext:aws:ses:rule-name': 'false',
                                'kms:EncryptionContext:aws:ses:message-id': 'false'
                            },
                            StringEquals: {
                                'kms:EncryptionContext:aws:ses:source-account': {
                                    Ref: 'AWS::AccountId'
                                }
                            }
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'ses.',
                                        {
                                            Ref: 'AWS::URLSuffix'
                                        }
                                    ]
                                ]
                            }
                        },
                        Resource: '*'
                    }
                ],
                Version: '2012-10-17'
            }
        }));
        test.done();
    },
    'can add a sns action'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleSnsAction({
                            encoding: lib_1.EmailEncoding.Base64,
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        SNSAction: {
                            Encoding: 'Base64',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            }
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    },
    'can add a stop action'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleStopAction({
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        StopAction: {
                            Scope: 'RuleSet',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            }
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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