"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certbot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const r53 = require("@aws-cdk/aws-route53");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const subscriptions = require("@aws-cdk/aws-sns-subscriptions");
const cdk = require("@aws-cdk/core");
const oneTimeEvents = require("@renovosolutions/cdk-library-one-time-event");
class Certbot extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        // Create a bucket if one is not provided
        if (props.bucket === undefined) {
            bucket = new s3.Bucket(this, 'bucket', {
                objectOwnership: s3.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                removalPolicy: props.removalPolicy || cdk.RemovalPolicy.RETAIN,
                autoDeleteObjects: props.enableObjectDeletion ?? false,
                versioned: true,
                lifecycleRules: [{
                        enabled: true,
                        abortIncompleteMultipartUploadAfter: cdk.Duration.days(1),
                    }],
                encryption: s3.BucketEncryption.S3_MANAGED,
                enforceSSL: true,
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            });
        }
        else {
            bucket = props.bucket;
        }
        const functionDir = path.join(__dirname, '../function');
        // Create an SNS topic if one is not provided and add the defined email to it
        let snsTopic = props.snsTopic ?? new sns.Topic(this, 'topic');
        if (props.snsTopic === undefined) {
            snsTopic.addSubscription(new subscriptions.EmailSubscription(props.letsencryptEmail));
        }
        // Set property defaults
        let layers = props.layers ?? [];
        let runOnDeploy = props.runOnDeploy ?? true;
        let functionDescription = props.functionDescription ?? 'Certbot Renewal Lambda for domain ' + props.letsencryptDomains.split(',')[0];
        let enableInsights = props.enableInsights ?? false;
        let insightsARN = props.insightsARN ?? 'arn:aws:lambda:' + cdk.Stack.of(this).region + ':580247275435:layer:LambdaInsightsExtension:14';
        // Set up role policies
        let managedPolicies = [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')];
        if (enableInsights) {
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
            layers.push(lambda.LayerVersion.fromLayerVersionArn(this, 'insightsLayer', insightsARN));
        }
        const snsPolicy = new iam.ManagedPolicy(this, 'snsPolicy', {
            description: 'Allow the Certbot function to notify an SNS topic upon completion.',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['sns:Publish'],
                    resources: [snsTopic.topicArn],
                }),
            ],
        });
        let hostedZones = [];
        props.hostedZoneNames.forEach((domainName) => {
            hostedZones.push(r53.HostedZone.fromLookup(this, 'zone' + domainName, {
                domainName,
                privateZone: false,
            }).hostedZoneArn);
        });
        const r53Policy = new iam.ManagedPolicy(this, 'r53Policy', {
            description: 'Allow the Certbot function to perform DNS verification.',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['route53:ListHostedZones'],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'route53:GetChange',
                        'route53:ChangeResourceRecordSets',
                    ],
                    resources: ['arn:aws:route53:::change/*'].concat(hostedZones),
                }),
            ],
        });
        const acmPolicy = new iam.ManagedPolicy(this, 'acmPolicy', {
            description: 'Allow the Certbot function to import and list certificates.',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: [
                        'acm:ListCertificates',
                        'acm:ImportCertificate',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['acm:DescribeCertificate'],
                    resources: ['arn:aws:acm:' + cdk.Stack.of(this).region + ':' + cdk.Stack.of(this).account + ':certificate/*'],
                }),
            ],
        });
        managedPolicies.push(snsPolicy);
        managedPolicies.push(r53Policy);
        managedPolicies.push(acmPolicy);
        const role = new iam.Role(this, 'role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        bucket.grantWrite(role);
        // Create the Lambda function
        this.handler = new lambda.Function(this, 'handler', {
            runtime: lambda.Runtime.PYTHON_3_8,
            role,
            code: lambda.Code.fromAsset(functionDir + '/function.zip'),
            handler: 'index.handler',
            functionName: props.functionName,
            description: functionDescription,
            environment: {
                LETSENCRYPT_DOMAINS: props.letsencryptDomains,
                LETSENCRYPT_EMAIL: props.letsencryptEmail,
                CERTIFICATE_BUCKET: bucket.bucketName,
                OBJECT_PREFIX: props.objectPrefix || '',
                REISSUE_DAYS: (props.reIssueDays === undefined) ? '30' : String(props.reIssueDays),
                PREFERRED_CHAIN: props.preferredChain || 'None',
                NOTIFICATION_SNS_ARN: snsTopic.topicArn,
            },
            layers,
            timeout: props.timeout || cdk.Duration.seconds(180),
        });
        // Add function triggers
        new events.Rule(this, 'trigger', {
            schedule: props.schedule || events.Schedule.cron({ minute: '0', hour: '0', weekDay: '1' }),
            targets: [new targets.LambdaFunction(this.handler)],
        });
        if (runOnDeploy) {
            new events.Rule(this, 'triggerImmediate', {
                schedule: new oneTimeEvents.OnDeploy(this, 'schedule', {
                    offsetMinutes: props.runOnDeployWaitMinutes || 10,
                }).schedule,
                targets: [new targets.LambdaFunction(this.handler)],
            });
        }
    }
}
exports.Certbot = Certbot;
_a = JSII_RTTI_SYMBOL_1;
Certbot[_a] = { fqn: "@renovosolutions/cdk-library-certbot.Certbot", version: "1.1.195" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFDN0IsOENBQThDO0FBQzlDLHVEQUF1RDtBQUN2RCx3Q0FBd0M7QUFDeEMsOENBQThDO0FBQzlDLDRDQUE0QztBQUM1QyxzQ0FBc0M7QUFDdEMsd0NBQXdDO0FBQ3hDLGdFQUFnRTtBQUNoRSxxQ0FBcUM7QUFDckMsNkVBQTZFO0FBdUc3RSxNQUFhLE9BQVEsU0FBUSxHQUFHLENBQUMsU0FBUztJQUl4QyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQW1CO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxNQUFpQixDQUFDO1FBRXRCLHlDQUF5QztRQUN6QyxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUyxFQUFFO1lBQzlCLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtnQkFDckMsZUFBZSxFQUFFLEVBQUUsQ0FBQyxlQUFlLENBQUMsc0JBQXNCO2dCQUMxRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWEsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLE1BQU07Z0JBQzlELGlCQUFpQixFQUFFLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxLQUFLO2dCQUN0RCxTQUFTLEVBQUUsSUFBSTtnQkFDZixjQUFjLEVBQUUsQ0FBQzt3QkFDZixPQUFPLEVBQUUsSUFBSTt3QkFDYixtQ0FBbUMsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7cUJBQzFELENBQUM7Z0JBQ0YsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO2dCQUMxQyxVQUFVLEVBQUUsSUFBSTtnQkFDaEIsaUJBQWlCLEVBQUUsRUFBRSxDQUFDLGlCQUFpQixDQUFDLFNBQVM7YUFDbEQsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1NBQ3ZCO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFeEQsNkVBQTZFO1FBQzdFLElBQUksUUFBUSxHQUFjLEtBQUssQ0FBQyxRQUFRLElBQUksSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN6RSxJQUFJLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxFQUFFO1lBQ2hDLFFBQVEsQ0FBQyxlQUFlLENBQUMsSUFBSSxhQUFhLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztTQUN2RjtRQUVELHdCQUF3QjtRQUN4QixJQUFJLE1BQU0sR0FBMkIsS0FBSyxDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUM7UUFDeEQsSUFBSSxXQUFXLEdBQVksS0FBSyxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUM7UUFDckQsSUFBSSxtQkFBbUIsR0FBVyxLQUFLLENBQUMsbUJBQW1CLElBQUksb0NBQW9DLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3SSxJQUFJLGNBQWMsR0FBWSxLQUFLLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQztRQUM1RCxJQUFJLFdBQVcsR0FBVyxLQUFLLENBQUMsV0FBVyxJQUFJLGlCQUFpQixHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxnREFBZ0QsQ0FBQztRQUVoSix1QkFBdUI7UUFDdkIsSUFBSSxlQUFlLEdBQUcsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDBDQUEwQyxDQUFDLENBQUMsQ0FBQztRQUMvRyxJQUFJLGNBQWMsRUFBRTtZQUNsQixlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsNkNBQTZDLENBQUMsQ0FBQyxDQUFDO1lBQ2hILE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUM7U0FDMUY7UUFFRCxNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUN6RCxXQUFXLEVBQUUsb0VBQW9FO1lBQ2pGLFVBQVUsRUFBRTtnQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLE9BQU8sRUFBRSxDQUFDLGFBQWEsQ0FBQztvQkFDeEIsU0FBUyxFQUFFLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQztpQkFDL0IsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxXQUFXLEdBQVksRUFBRSxDQUFDO1FBQzlCLEtBQUssQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFFLENBQUMsVUFBVSxFQUFFLEVBQUU7WUFDNUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsTUFBTSxHQUFHLFVBQVUsRUFBRTtnQkFDcEUsVUFBVTtnQkFDVixXQUFXLEVBQUUsS0FBSzthQUNuQixDQUFDLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDcEIsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUN6RCxXQUFXLEVBQUUseURBQXlEO1lBQ3RFLFVBQVUsRUFBRTtnQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLE9BQU8sRUFBRSxDQUFDLHlCQUF5QixDQUFDO29CQUNwQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2pCLENBQUM7Z0JBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO29CQUN4QixPQUFPLEVBQUU7d0JBQ1AsbUJBQW1CO3dCQUNuQixrQ0FBa0M7cUJBQ25DO29CQUNELFNBQVMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQztpQkFDOUQsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDekQsV0FBVyxFQUFFLDZEQUE2RDtZQUMxRSxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO29CQUN4QixPQUFPLEVBQUU7d0JBQ1Asc0JBQXNCO3dCQUN0Qix1QkFBdUI7cUJBQ3hCO29CQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztpQkFDakIsQ0FBQztnQkFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLE9BQU8sRUFBRSxDQUFDLHlCQUF5QixDQUFDO29CQUNwQyxTQUFTLEVBQUUsQ0FBQyxjQUFjLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxHQUFHLEdBQUcsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLEdBQUcsZ0JBQWdCLENBQUM7aUJBQzlHLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILGVBQWUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDaEMsZUFBZSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoQyxlQUFlLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRWhDLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQ3RDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxlQUFlO1NBQ2hCLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFeEIsNkJBQTZCO1FBQzdCLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxJQUFJO1lBQ0osSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsR0FBRyxlQUFlLENBQUM7WUFDMUQsT0FBTyxFQUFFLGVBQWU7WUFDeEIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLFdBQVcsRUFBRSxtQkFBbUI7WUFDaEMsV0FBVyxFQUFFO2dCQUNYLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7Z0JBQzdDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7Z0JBQ3pDLGtCQUFrQixFQUFFLE1BQU0sQ0FBQyxVQUFVO2dCQUNyQyxhQUFhLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxFQUFFO2dCQUN2QyxZQUFZLEVBQUUsQ0FBQyxLQUFLLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDO2dCQUNsRixlQUFlLEVBQUUsS0FBSyxDQUFDLGNBQWMsSUFBSSxNQUFNO2dCQUMvQyxvQkFBb0IsRUFBRSxRQUFRLENBQUMsUUFBUTthQUN4QztZQUNELE1BQU07WUFDTixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7U0FDcEQsQ0FBQyxDQUFDO1FBRUgsd0JBQXdCO1FBQ3hCLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQy9CLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUSxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQztZQUMxRixPQUFPLEVBQUUsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ3BELENBQUMsQ0FBQztRQUVILElBQUksV0FBVyxFQUFFO1lBQ2YsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDeEMsUUFBUSxFQUFFLElBQUksYUFBYSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO29CQUNyRCxhQUFhLEVBQUUsS0FBSyxDQUFDLHNCQUFzQixJQUFJLEVBQUU7aUJBQ2xELENBQUMsQ0FBQyxRQUFRO2dCQUNYLE9BQU8sRUFBRSxDQUFDLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7YUFDcEQsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDOztBQXpKSCwwQkEwSkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgdGFyZ2V0cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzLXRhcmdldHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgcjUzIGZyb20gJ0Bhd3MtY2RrL2F3cy1yb3V0ZTUzJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBzbnMgZnJvbSAnQGF3cy1jZGsvYXdzLXNucyc7XG5pbXBvcnQgKiBhcyBzdWJzY3JpcHRpb25zIGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBvbmVUaW1lRXZlbnRzIGZyb20gJ0ByZW5vdm9zb2x1dGlvbnMvY2RrLWxpYnJhcnktb25lLXRpbWUtZXZlbnQnO1xuXG5leHBvcnQgaW50ZXJmYWNlIENlcnRib3RQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgY29tbWEgZGVsaW1pdGVkIGxpc3Qgb2YgZG9tYWlucyBmb3Igd2hpY2ggdGhlIExldCdzIEVuY3J5cHQgY2VydGlmaWNhdGUgd2lsbCBiZSB2YWxpZC4gUHJpbWFyeSBkb21haW4gc2hvdWxkIGJlIGZpcnN0LlxuICAgKi9cbiAgcmVhZG9ubHkgbGV0c2VuY3J5cHREb21haW5zOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZW1haWwgdG8gYXNzb2NpYXRlIHdpdGggdGhlIExldCdzIEVuY3J5cHQgY2VydGlmaWNhdGUgcmVxdWVzdC5cbiAgICovXG4gIHJlYWRvbmx5IGxldHNlbmNyeXB0RW1haWw6IHN0cmluZztcbiAgLyoqXG4gICAqIEFueSBhZGRpdGlvbmFsIExhbWJkYSBsYXllcnMgdG8gdXNlIHdpdGggdGhlIGNyZWF0ZWQgZnVuY3Rpb24uIEZvciBleGFtcGxlIExhbWJkYSBFeHRlbnNpb25zXG4gICAqL1xuICByZWFkb25seSBsYXllcnM/OiBsYW1iZGEuSUxheWVyVmVyc2lvbltdO1xuICAvKipcbiAgICogSG9zdGVkIHpvbmUgbmFtZXMgdGhhdCB3aWxsIGJlIHJlcXVpcmVkIGZvciBETlMgdmVyaWZpY2F0aW9uIHdpdGggY2VydGJvdFxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdGVkWm9uZU5hbWVzOiBzdHJpbmdbXTtcbiAgLyoqXG4gICAqIFRoZSBTMyBidWNrZXQgdG8gcGxhY2UgdGhlIHJlc3VsdGluZyBjZXJ0aWZpY2F0ZXMgaW4uIElmIG5vIGJ1Y2tldCBpcyBnaXZlbiBvbmUgd2lsbCBiZSBjcmVhdGVkIGF1dG9tYXRpY2FsbHkuXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ/OiBzMy5CdWNrZXQ7XG4gIC8qKlxuICAgKiBUaGUgcHJlZml4IHRvIGFwcGx5IHRvIHRoZSBmaW5hbCBTMyBrZXkgbmFtZSBmb3IgdGhlIGNlcnRpZmljYXRlcy4gRGVmYXVsdCBpcyBubyBwcmVmaXguXG4gICAqL1xuICByZWFkb25seSBvYmplY3RQcmVmaXg/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbnVtYmVycyBvZiBkYXlzIGxlZnQgdW50aWwgdGhlIHByaW9yIGNlcnQgZXhwaXJlcyBiZWZvcmUgaXNzdWluZyBhIG5ldyBvbmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IDMwXG4gICAqL1xuICByZWFkb25seSByZUlzc3VlRGF5cz86IG51bWJlcjtcbiAgLyoqXG4gICAqIFNldCB0aGUgcHJlZmVycmVkIGNlcnRpZmljYXRlIGNoYWluLlxuICAgKlxuICAgKiBAZGVmYXVsdCAnTm9uZSdcbiAgICovXG4gIHJlYWRvbmx5IHByZWZlcnJlZENoYWluPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIFNOUyB0b3BpYyB0byBub3RpZnkgd2hlbiBhIG5ldyBjZXJ0IGlzIGlzc3VlZC4gSWYgbm8gdG9waWMgaXMgZ2l2ZW4gb25lIHdpbGwgYmUgY3JlYXRlZCBhdXRvbWF0aWNhbGx5LlxuICAgKi9cbiAgcmVhZG9ubHkgc25zVG9waWM/OiBzbnMuVG9waWM7XG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBlbmFibGUgTGFtYmRhIEluc2lnaHRzXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVJbnNpZ2h0cz86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBJbnNpZ2h0cyBsYXllciBBUk4gZm9yIHlvdXIgcmVnaW9uLiBEZWZhdWx0cyB0byBsYXllciBmb3IgVVMtRUFTVC0xXG4gICAqL1xuICByZWFkb25seSBpbnNpZ2h0c0FSTj86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSB0aW1lb3V0IGR1cmF0aW9uIGZvciBMYW1iZGEgZnVuY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgY2RrLkR1cmFjdGlvbi5zZWNvbmRzKDE4MClcbiAgICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBjZGsuRHVyYXRpb247XG4gIC8qKlxuICAgKiBUaGUgc2NoZWR1bGUgZm9yIHRoZSBjZXJ0aWZpY2F0ZSBjaGVjayB0cmlnZ2VyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBldmVudHMuU2NoZWR1bGUuY3Jvbih7IG1pbnV0ZTogJzAnLCBob3VyOiAnMCcsIHdlZWtEYXk6ICcxJyB9KVxuICAgKi9cbiAgcmVhZG9ubHkgc2NoZWR1bGU/OiBldmVudHMuU2NoZWR1bGU7XG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBzY2hlZHVsZSBhIHRyaWdnZXIgdG8gcnVuIHRoZSBmdW5jdGlvbiBhZnRlciBlYWNoIGRlcGxveW1lbnRcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcnVuT25EZXBsb3k/OiBib29sZWFuO1xuICAvKipcbiAgICogSG93IG1hbnkgbWludXRlcyB0byB3YWl0IGJlZm9yZSBydW5uaW5nIHRoZSBwb3N0IGRlcGxveW1lbnQgTGFtYmRhIHRyaWdnZXJcbiAgICpcbiAgICogQGRlZmF1bHQgMTBcbiAgICovXG4gIHJlYWRvbmx5IHJ1bk9uRGVwbG95V2FpdE1pbnV0ZXM/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgZGVzY3JpcHRpb24gZm9yIHRoZSByZXN1bHRpbmcgTGFtYmRhIGZ1bmN0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgZnVuY3Rpb25EZXNjcmlwdGlvbj86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSByZXN1bHRpbmcgTGFtYmRhIGZ1bmN0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgZnVuY3Rpb25OYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIHJlbW92YWwgcG9saWN5IGZvciB0aGUgUzMgYnVja2V0IHRoYXQgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkLlxuICAgKlxuICAgKiBIYXMgbm8gZWZmZWN0IGlmIGEgYnVja2V0IGlzIGdpdmVuIGFzIGEgcHJvcGVydHlcbiAgICpcbiAgICogQGRlZmF1bHQgUmVtb3ZhbFBvbGljeS5SRVRBSU5cbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBjZGsuUmVtb3ZhbFBvbGljeTtcbiAgLyoqXG4gICAgKiBXaGV0aGVyIG9yIG5vdCB0byBlbmFibGUgYXV0b21hdGljIG9iamVjdCBkZWxldGlvbiBpZiB0aGUgcHJvdmlkZWQgYnVja2V0IGlzIGRlbGV0ZWQuXG4gICAgKlxuICAgICogSGFzIG5vIGVmZmVjdCBpZiBhIGJ1Y2tldCBpcyBnaXZlbiBhcyBhIHByb3BlcnR5XG4gICAgKlxuICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAqL1xuICByZWFkb25seSBlbmFibGVPYmplY3REZWxldGlvbj86IGJvb2xlYW47XG59XG5cbmV4cG9ydCBjbGFzcyBDZXJ0Ym90IGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG5cbiAgcHVibGljIHJlYWRvbmx5IGhhbmRsZXI6IGxhbWJkYS5GdW5jdGlvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENlcnRib3RQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBsZXQgYnVja2V0OiBzMy5CdWNrZXQ7XG5cbiAgICAvLyBDcmVhdGUgYSBidWNrZXQgaWYgb25lIGlzIG5vdCBwcm92aWRlZFxuICAgIGlmIChwcm9wcy5idWNrZXQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnYnVja2V0Jywge1xuICAgICAgICBvYmplY3RPd25lcnNoaXA6IHMzLk9iamVjdE93bmVyc2hpcC5CVUNLRVRfT1dORVJfUFJFRkVSUkVELFxuICAgICAgICByZW1vdmFsUG9saWN5OiBwcm9wcy5yZW1vdmFsUG9saWN5IHx8IGNkay5SZW1vdmFsUG9saWN5LlJFVEFJTixcbiAgICAgICAgYXV0b0RlbGV0ZU9iamVjdHM6IHByb3BzLmVuYWJsZU9iamVjdERlbGV0aW9uID8/IGZhbHNlLFxuICAgICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICAgIGxpZmVjeWNsZVJ1bGVzOiBbe1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgYWJvcnRJbmNvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkQWZ0ZXI6IGNkay5EdXJhdGlvbi5kYXlzKDEpLFxuICAgICAgICB9XSxcbiAgICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgICBibG9ja1B1YmxpY0FjY2VzczogczMuQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGJ1Y2tldCA9IHByb3BzLmJ1Y2tldDtcbiAgICB9XG5cbiAgICBjb25zdCBmdW5jdGlvbkRpciA9IHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9mdW5jdGlvbicpO1xuXG4gICAgLy8gQ3JlYXRlIGFuIFNOUyB0b3BpYyBpZiBvbmUgaXMgbm90IHByb3ZpZGVkIGFuZCBhZGQgdGhlIGRlZmluZWQgZW1haWwgdG8gaXRcbiAgICBsZXQgc25zVG9waWM6IHNucy5Ub3BpYyA9IHByb3BzLnNuc1RvcGljID8/IG5ldyBzbnMuVG9waWModGhpcywgJ3RvcGljJyk7XG4gICAgaWYgKHByb3BzLnNuc1RvcGljID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHNuc1RvcGljLmFkZFN1YnNjcmlwdGlvbihuZXcgc3Vic2NyaXB0aW9ucy5FbWFpbFN1YnNjcmlwdGlvbihwcm9wcy5sZXRzZW5jcnlwdEVtYWlsKSk7XG4gICAgfVxuXG4gICAgLy8gU2V0IHByb3BlcnR5IGRlZmF1bHRzXG4gICAgbGV0IGxheWVyczogbGFtYmRhLklMYXllclZlcnNpb25bXSA9IHByb3BzLmxheWVycyA/PyBbXTtcbiAgICBsZXQgcnVuT25EZXBsb3k6IGJvb2xlYW4gPSBwcm9wcy5ydW5PbkRlcGxveSA/PyB0cnVlO1xuICAgIGxldCBmdW5jdGlvbkRlc2NyaXB0aW9uOiBzdHJpbmcgPSBwcm9wcy5mdW5jdGlvbkRlc2NyaXB0aW9uID8/ICdDZXJ0Ym90IFJlbmV3YWwgTGFtYmRhIGZvciBkb21haW4gJyArIHByb3BzLmxldHNlbmNyeXB0RG9tYWlucy5zcGxpdCgnLCcpWzBdO1xuICAgIGxldCBlbmFibGVJbnNpZ2h0czogYm9vbGVhbiA9IHByb3BzLmVuYWJsZUluc2lnaHRzID8/IGZhbHNlO1xuICAgIGxldCBpbnNpZ2h0c0FSTjogc3RyaW5nID0gcHJvcHMuaW5zaWdodHNBUk4gPz8gJ2Fybjphd3M6bGFtYmRhOicgKyBjZGsuU3RhY2sub2YodGhpcykucmVnaW9uICsgJzo1ODAyNDcyNzU0MzU6bGF5ZXI6TGFtYmRhSW5zaWdodHNFeHRlbnNpb246MTQnO1xuXG4gICAgLy8gU2V0IHVwIHJvbGUgcG9saWNpZXNcbiAgICBsZXQgbWFuYWdlZFBvbGljaWVzID0gW2lhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZScpXTtcbiAgICBpZiAoZW5hYmxlSW5zaWdodHMpIHtcbiAgICAgIG1hbmFnZWRQb2xpY2llcy5wdXNoKGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQ2xvdWRXYXRjaExhbWJkYUluc2lnaHRzRXhlY3V0aW9uUm9sZVBvbGljeScpKTtcbiAgICAgIGxheWVycy5wdXNoKGxhbWJkYS5MYXllclZlcnNpb24uZnJvbUxheWVyVmVyc2lvbkFybih0aGlzLCAnaW5zaWdodHNMYXllcicsIGluc2lnaHRzQVJOKSk7XG4gICAgfVxuXG4gICAgY29uc3Qgc25zUG9saWN5ID0gbmV3IGlhbS5NYW5hZ2VkUG9saWN5KHRoaXMsICdzbnNQb2xpY3knLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ0FsbG93IHRoZSBDZXJ0Ym90IGZ1bmN0aW9uIHRvIG5vdGlmeSBhbiBTTlMgdG9waWMgdXBvbiBjb21wbGV0aW9uLicsXG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogWydzbnM6UHVibGlzaCddLFxuICAgICAgICAgIHJlc291cmNlczogW3Nuc1RvcGljLnRvcGljQXJuXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgbGV0IGhvc3RlZFpvbmVzOnN0cmluZ1tdID0gW107XG4gICAgcHJvcHMuaG9zdGVkWm9uZU5hbWVzLmZvckVhY2goIChkb21haW5OYW1lKSA9PiB7XG4gICAgICBob3N0ZWRab25lcy5wdXNoKHI1My5Ib3N0ZWRab25lLmZyb21Mb29rdXAodGhpcywgJ3pvbmUnICsgZG9tYWluTmFtZSwge1xuICAgICAgICBkb21haW5OYW1lLFxuICAgICAgICBwcml2YXRlWm9uZTogZmFsc2UsXG4gICAgICB9KS5ob3N0ZWRab25lQXJuKTtcbiAgICB9KTtcblxuICAgIGNvbnN0IHI1M1BvbGljeSA9IG5ldyBpYW0uTWFuYWdlZFBvbGljeSh0aGlzLCAncjUzUG9saWN5Jywge1xuICAgICAgZGVzY3JpcHRpb246ICdBbGxvdyB0aGUgQ2VydGJvdCBmdW5jdGlvbiB0byBwZXJmb3JtIEROUyB2ZXJpZmljYXRpb24uJyxcbiAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3JvdXRlNTM6TGlzdEhvc3RlZFpvbmVzJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgJ3JvdXRlNTM6R2V0Q2hhbmdlJyxcbiAgICAgICAgICAgICdyb3V0ZTUzOkNoYW5nZVJlc291cmNlUmVjb3JkU2V0cycsXG4gICAgICAgICAgXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnYXJuOmF3czpyb3V0ZTUzOjo6Y2hhbmdlLyonXS5jb25jYXQoaG9zdGVkWm9uZXMpLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBhY21Qb2xpY3kgPSBuZXcgaWFtLk1hbmFnZWRQb2xpY3kodGhpcywgJ2FjbVBvbGljeScsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiAnQWxsb3cgdGhlIENlcnRib3QgZnVuY3Rpb24gdG8gaW1wb3J0IGFuZCBsaXN0IGNlcnRpZmljYXRlcy4nLFxuICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICdhY206TGlzdENlcnRpZmljYXRlcycsXG4gICAgICAgICAgICAnYWNtOkltcG9ydENlcnRpZmljYXRlJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFsnYWNtOkRlc2NyaWJlQ2VydGlmaWNhdGUnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnYXJuOmF3czphY206JyArIGNkay5TdGFjay5vZih0aGlzKS5yZWdpb24gKyAnOicgKyBjZGsuU3RhY2sub2YodGhpcykuYWNjb3VudCArICc6Y2VydGlmaWNhdGUvKiddLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBtYW5hZ2VkUG9saWNpZXMucHVzaChzbnNQb2xpY3kpO1xuICAgIG1hbmFnZWRQb2xpY2llcy5wdXNoKHI1M1BvbGljeSk7XG4gICAgbWFuYWdlZFBvbGljaWVzLnB1c2goYWNtUG9saWN5KTtcblxuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ3JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llcyxcbiAgICB9KTtcblxuICAgIGJ1Y2tldC5ncmFudFdyaXRlKHJvbGUpO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICB0aGlzLmhhbmRsZXIgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdoYW5kbGVyJywge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfOCxcbiAgICAgIHJvbGUsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoZnVuY3Rpb25EaXIgKyAnL2Z1bmN0aW9uLnppcCcpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgZnVuY3Rpb25OYW1lOiBwcm9wcy5mdW5jdGlvbk5hbWUsXG4gICAgICBkZXNjcmlwdGlvbjogZnVuY3Rpb25EZXNjcmlwdGlvbixcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIExFVFNFTkNSWVBUX0RPTUFJTlM6IHByb3BzLmxldHNlbmNyeXB0RG9tYWlucyxcbiAgICAgICAgTEVUU0VOQ1JZUFRfRU1BSUw6IHByb3BzLmxldHNlbmNyeXB0RW1haWwsXG4gICAgICAgIENFUlRJRklDQVRFX0JVQ0tFVDogYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIE9CSkVDVF9QUkVGSVg6IHByb3BzLm9iamVjdFByZWZpeCB8fCAnJyxcbiAgICAgICAgUkVJU1NVRV9EQVlTOiAocHJvcHMucmVJc3N1ZURheXMgPT09IHVuZGVmaW5lZCkgPyAnMzAnIDogU3RyaW5nKHByb3BzLnJlSXNzdWVEYXlzKSxcbiAgICAgICAgUFJFRkVSUkVEX0NIQUlOOiBwcm9wcy5wcmVmZXJyZWRDaGFpbiB8fCAnTm9uZScsXG4gICAgICAgIE5PVElGSUNBVElPTl9TTlNfQVJOOiBzbnNUb3BpYy50b3BpY0FybixcbiAgICAgIH0sXG4gICAgICBsYXllcnMsXG4gICAgICB0aW1lb3V0OiBwcm9wcy50aW1lb3V0IHx8IGNkay5EdXJhdGlvbi5zZWNvbmRzKDE4MCksXG4gICAgfSk7XG5cbiAgICAvLyBBZGQgZnVuY3Rpb24gdHJpZ2dlcnNcbiAgICBuZXcgZXZlbnRzLlJ1bGUodGhpcywgJ3RyaWdnZXInLCB7XG4gICAgICBzY2hlZHVsZTogcHJvcHMuc2NoZWR1bGUgfHwgZXZlbnRzLlNjaGVkdWxlLmNyb24oeyBtaW51dGU6ICcwJywgaG91cjogJzAnLCB3ZWVrRGF5OiAnMScgfSksXG4gICAgICB0YXJnZXRzOiBbbmV3IHRhcmdldHMuTGFtYmRhRnVuY3Rpb24odGhpcy5oYW5kbGVyKV0sXG4gICAgfSk7XG5cbiAgICBpZiAocnVuT25EZXBsb3kpIHtcbiAgICAgIG5ldyBldmVudHMuUnVsZSh0aGlzLCAndHJpZ2dlckltbWVkaWF0ZScsIHtcbiAgICAgICAgc2NoZWR1bGU6IG5ldyBvbmVUaW1lRXZlbnRzLk9uRGVwbG95KHRoaXMsICdzY2hlZHVsZScsIHtcbiAgICAgICAgICBvZmZzZXRNaW51dGVzOiBwcm9wcy5ydW5PbkRlcGxveVdhaXRNaW51dGVzIHx8IDEwLFxuICAgICAgICB9KS5zY2hlZHVsZSxcbiAgICAgICAgdGFyZ2V0czogW25ldyB0YXJnZXRzLkxhbWJkYUZ1bmN0aW9uKHRoaXMuaGFuZGxlcildLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG59XG4iXX0=