"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const receipt_rule_action_1 = require("./receipt-rule-action");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.renderedActions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.getRenderedActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy
            },
            ruleSetName: props.ruleSet.receiptRuleSetName
        });
        this.receiptRuleName = resource.ref;
        if (props.actions) {
            props.actions.forEach(action => this.addAction(action));
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        const renderedAction = action.render();
        this.renderedActions.push(renderedAction);
    }
    getRenderedActions() {
        if (this.renderedActions.length === 0) {
            return undefined;
        }
        return this.renderedActions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_8_10,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16'
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                new receipt_rule_action_1.ReceiptRuleLambdaAction({
                    function: fn,
                    invocationType: receipt_rule_action_1.LambdaInvocationType.REQUEST_RESPONSE
                })
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log("SES Notification:\n", JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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