"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const vpc_link_1 = require("./vpc-link");
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IntegrationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const options = this.props.options || {};
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.timeout && !options.timeout.isUnresolved() && (options.timeout.toMilliseconds() < 50 || options.timeout.toMilliseconds() > 29000)) {
            throw new Error('Integration timeout must be between 50 milliseconds and 29 seconds.');
        }
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_Method(_method);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        let uri = this.props.uri;
        const options = this.props.options;
        if ((options === null || options === void 0 ? void 0 : options.connectionType) === ConnectionType.VPC_LINK && uri === undefined) {
            uri = core_1.Lazy.string({
                // needs to be a lazy since the targets can be added to the VpcLink construct after initialization.
                produce: () => {
                    const vpcLink = options.vpcLink;
                    if (vpcLink instanceof vpc_link_1.VpcLink) {
                        const targets = vpcLink._targetDnsNames;
                        if (targets.length > 1) {
                            throw new Error("'uri' is required when there are more than one NLBs in the VPC Link");
                        }
                        else {
                            return `http://${targets[0]}`;
                        }
                    }
                    else {
                        throw new Error("'uri' is required when the 'connectionType' is VPC_LINK");
                    }
                },
            });
        }
        return {
            options: {
                ...options,
                connectionType: (options === null || options === void 0 ? void 0 : options.vpcLink) ? ConnectionType.VPC_LINK : options === null || options === void 0 ? void 0 : options.connectionType,
            },
            type: this.props.type,
            uri,
            integrationHttpMethod: this.props.integrationHttpMethod,
        };
    }
}
exports.Integration = Integration;
_a = JSII_RTTI_SYMBOL_1;
Integration[_a] = { fqn: "@aws-cdk/aws-apigateway.Integration", version: "1.154.0" };
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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