"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvedTypeHint = exports.RememberingTokenResolver = exports.findTokens = exports.resolve = exports.ResolutionTypeHint = exports.INTRINSIC_KEY_PREFIX = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolved complex values will have a type hint applied.
 *
 * The type hint will be based on the type of the input value that was resolved.
 *
 * If the value was encoded, the type hint will be the type of the encoded value. In case
 * of a plain `IResolvable`, a type hint of 'string' will be assumed.
 */
const RESOLUTION_TYPEHINT_SYM = Symbol.for('@aws-cdk/core.resolvedTypeHint');
/**
 * Prefix used for intrinsic keys
 *
 * If a key with this prefix is found in an object, the actual value of the
 * key doesn't matter. The value of this key will be an `[ actualKey, actualValue ]`
 * tuple, and the `actualKey` will be a value which otherwise couldn't be represented
 * in the types of `string | number | symbol`, which are the only possible JavaScript
 * object keys.
 */
exports.INTRINSIC_KEY_PREFIX = '$IntrinsicKey$';
/**
 * Type hints for resolved values
 */
var ResolutionTypeHint;
(function (ResolutionTypeHint) {
    ResolutionTypeHint["STRING"] = "string";
    ResolutionTypeHint["NUMBER"] = "number";
    ResolutionTypeHint["LIST"] = "list";
})(ResolutionTypeHint = exports.ResolutionTypeHint || (exports.ResolutionTypeHint = {}));
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x, changeOptions) { return resolve(x, { ...options, ...changeOptions, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        // If this is a "list element" Token, it should never occur by itself in string context
        if (encoding_1.TokenString.forListToken(obj).test()) {
            throw new Error('Found an encoded list token string in a scalar string context. Use \'Fn.select(0, list)\' (not \'list[0]\') to extract elements from token lists.');
        }
        // Otherwise look for a stringified Token in this object
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return tagResolvedValue(options.resolver.resolveString(fragments, makeContext()[0]), ResolutionTypeHint.STRING);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return tagResolvedValue(resolveNumberToken(obj, makeContext()[0]), ResolutionTypeHint.NUMBER);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return tagResolvedValue(options.resolver.resolveList(obj, makeContext()[0]), ResolutionTypeHint.LIST);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        const ret = tagResolvedValue(options.resolver.resolveToken(obj, context, postProcessor), ResolutionTypeHint.STRING);
        return ret;
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    let intrinsicKeyCtr = 0;
    for (const key of Object.keys(obj)) {
        const value = makeContext(String(key))[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        // Simple case -- not an unresolved key
        if (!encoding_1.unresolved(key)) {
            result[key] = value;
            continue;
        }
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) === 'string') {
            result[resolvedKey] = value;
        }
        else {
            if (!options.allowIntrinsicKeys) {
                // eslint-disable-next-line max-len
                throw new Error(`"${String(key)}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}. Consider using "CfnJson" to delay resolution to deployment-time`);
            }
            // Can't represent this object in a JavaScript key position, but we can store it
            // in value position. Use a unique symbol as the key.
            result[`${exports.INTRINSIC_KEY_PREFIX}${intrinsicKeyCtr++}`] = [resolvedKey, value];
        }
    }
    // Because we may be called to recurse on already resolved values (that already have type hints applied)
    // and we just copied those values into a fresh object, be sure to retain any type hints.
    const previousTypeHint = resolvedTypeHint(obj);
    return previousTypeHint ? tagResolvedValue(result, previousTypeHint) : result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
/**
 * Apply a type hint to a resolved value
 *
 * The type hint will only be applied to objects.
 *
 * These type hints are used for correct JSON-ification of intrinsic values.
 */
function tagResolvedValue(value, typeHint) {
    if (typeof value !== 'object' || value == null) {
        return value;
    }
    Object.defineProperty(value, RESOLUTION_TYPEHINT_SYM, {
        value: typeHint,
        configurable: true,
    });
    return value;
}
/**
 * Return the type hint from the given value
 *
 * If the value is not a resolved value (i.e, the result of resolving a token),
 * `undefined` will be returned.
 *
 * These type hints are used for correct JSON-ification of intrinsic values.
 */
function resolvedTypeHint(value) {
    if (typeof value !== 'object' || value == null) {
        return undefined;
    }
    return value[RESOLUTION_TYPEHINT_SYM];
}
exports.resolvedTypeHint = resolvedTypeHint;
//# sourceMappingURL=data:application/json;base64,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