'''
``homelette.routines``
======================

The :mod:`homelette.routines` submodule contains classes for model generation.
Routines are the building blocks that are used to generate homology models.

Currently, a number of pre-implemented routines based on `MODELLER`, `altMOD`
and `ProMod3` are available. It is possible to implement custom routines for
model generation and use them in the `homelette` framework.

Tutorials
---------

The basics of generating homology models with pre-implemented modelling
routines are presented in :ref:`Tutorial 2</Tutorial2_Modelling.ipynb>`.
Complex modelling with `homelette` is introduced in
:ref:`Tutorial 6</Tutorial6_ComplexModelling.ipynb>`.
Implementing custom modelling routines is discussed in
:ref:`Tutorial 4</Tutorial4_ExtendingHomelette.ipynb>`.
Assembling custom pipelines is discussed in :ref:`Tutorial
7</Tutorial7_AssemblingPipelines.ipynb>`.

Classes
-------

The following modelling routines are implemented:

    :class:`Routine_automodel_default`
    :class:`Routine_automodel_slow`
    :class:`Routine_altmod_default`
    :class:`Routine_altmod_slow`
    :class:`Routine_promod3`

Specifically for the modelling of complex structures, the following routines
are implemented:

    :class:`Routine_complex_automodel_default`
    :class:`Routine_complex_automodel_slow`
    :class:`Routine_complex_altmod_default`
    :class:`Routine_complex_altmod_slow`

-----

'''

__all__ = [
        'Routine_automodel_default', 'Routine_automodel_slow',
        'Routine_altmod_default', 'Routine_altmod_slow', 'Routine_promod3',
        'Routine_complex_automodel_default', 'Routine_complex_automodel_slow',
        'Routine_complex_altmod_default', 'Routine_complex_altmod_slow']

# Standard library imports
import contextlib
import copy
import glob
import os
# import shutil
import string
import typing
# import warnings

# Third party imports

# Local application imports
from .organization import Model  # noqa: E402

# Optional dependencies
_IMPORTS = dict()
try:
    import altmod
    _IMPORTS['altmod'] = True
except ImportError:
    _IMPORTS['altmod'] = False

try:
    import modeller
    import modeller.automodel
    import modeller.parallel
    _IMPORTS['modeller'] = True
except ImportError:
    _IMPORTS['modeller'] = False

try:
    import ost
    _IMPORTS['ost'] = True
except ImportError:
    _IMPORTS['ost'] = False

try:
    import promod3
    _IMPORTS['promod3'] = True
except ImportError:
    _IMPORTS['promod3'] = False

# Local imports for type checking
# taken from https://stackoverflow.com/a/39757388/7912251
if typing.TYPE_CHECKING:
    from .alignment import Alignment


class Routine():
    '''
    Parent class to all modelling routines.

    Not supposed to be used by user, used for inheritance. Implements a few
    common attributes and methods shared by all Routine objects.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used
        for the modelling
    tag : str
        The identifier associated with a specific execution of the routine

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine
    '''

    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str) -> None:
        self.alignment = copy.deepcopy(alignment)
        self.target = target
        self.templates = templates
        self.tag = tag

        self.models = []
        self.routine = None  # will be set by children classes

    def _rename_models(self) -> None:
        '''
        Rename generated models using the tag given for this execution of a
        routine.

        Called by Children classes during model generation.

        Returns
        -------
        None
        '''
        i = 1
        for model in self.models:
            model.rename('{}_{}.pdb'.format(self.tag, i))
            i += 1

    @staticmethod
    def _remove_files(*args: str) -> None:
        '''
        Remove files given after modelling procedure.

        Called by Children classes after model generation in order to clean up
        temporary files or unneccessary output files.
        Uses glob.glob internally and can therefore handle wildcards.

        Parameters
        ----------
        *args : str
            Filenames or queries for filenames that will be deleted.

        Returns
        -------
        None
        '''
        for query in args:
            for file in glob.glob(query):
                os.remove(file)

    @staticmethod
    def _check_dependencies(dependencies: typing.Iterable) -> None:
        '''
        Checks if all dependencies could be loaded. Helper function for
        children classes.

        Parameters
        ----------
        dependencies : Iterable
            Iterable of dependencies

        Returns
        -------
        None

        Raises
        ------
        ImportError
            One or more dependencies were not imported
        '''
        for dependency in dependencies:
            if not _IMPORTS[dependency]:
                raise ImportError(
                    '"{}" is required for this functionality, but could '
                    'not be imported.'.format(dependency))


class Routine_modeller(Routine):
    '''
    Parent class to all MODELLER routines.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    routine : str
        The identifier associated with a specific routine
    models : list
        List of generated models

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    Not supposed to be used by users. Implements the model generation procedure
    that is shared between all different children objects. Children objects
    only specify different settings for the various modelling and refinement
    parameters.
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str) -> None:
        # check dependencies
        dependencies = ['modeller']
        self._check_dependencies(dependencies)
        # continue init
        Routine.__init__(self, alignment, target, templates, tag)

    def _generate_models(
            self, model_class: typing.Type['modeller.automodel.automodel'],
            n_models: int, library_schedule:
            typing.Type['modeller.schedule.schedule'], max_var_iterations: int,
            md_level: typing.Callable, repeat_optimization: int,
            n_threads: int, use_hetatms: bool = False) -> None:
        '''
        Generate models using modeller

        For more information about the different modelling settings, please
        check the documentation of modeller.

        Parameters
        ----------
        model_class : modeller.automodel.automodel
            The modeller/altmod class used for modelling
        n_models : int
            Number of models generated
        library_schedule : modeller.schedule.schedule
            The refinement schedule used for modelling
        max_var_iterations : int
            The length of the refinement procedure
        md_level : Callable
            The level of molecular dynamics simulations used to refine the
            model
        repeat_optimization : int
            The number of iterations for the optimization procedure
        n_threads : int
            Number of threads used for model generation
        use_hetatms : bool
            Read heteroatoms from PDB templates

        Returns
        -------
        None
        '''
        # process alignment for modeller, expects correct annotation
        self.alignment.select_sequences([self.target] + self.templates)
        self.alignment.remove_redundant_gaps()
        self.alignment.write_pir('.tmp.pir')

        # modelling
        with contextlib.redirect_stdout(None):  # suppress modellers output
            # prepare inputs
            env = modeller.environ()
            if use_hetatms is True:
                env.io.hetatm = True
            m = model_class(env, alnfile='.tmp.pir', knowns=self.templates,
                            sequence=self.target)
            # set output parameters
            m.blank_single_chain = False
            # set modelling parameters
            m.starting_model = 1
            m.ending_model = n_models
            m.library_schedule = library_schedule
            m.max_var_iterations = max_var_iterations
            m.md_level = md_level
            m.repeat_optimization = repeat_optimization
            # set up parallelization if multiple threads requested
            if n_threads > 1:
                j = modeller.parallel.job()
                for i in range(n_threads):
                    j.append(modeller.parallel.local_slave())
                m.use_parallel_job(j)
            # perform modelling
            m.make()

        # capturing output
        for pdb in glob.glob('{}.B99*.pdb'.format(self.target)):
            self.models.append(
                    Model(os.path.realpath(os.path.expanduser(pdb)),
                          self.tag, self.routine))

        # cleaning up
        self._rename_models()
        self._remove_files(
            '{}.D00*'.format(self.target),
            '{}.V99*'.format(self.target),
            '{}.rsr'.format(self.target),
            '{}.sch'.format(self.target),
            '{}.ini'.format(self.target),
            '.tmp*')


class Routine_automodel_default(Routine_modeller):
    '''
    Class for performing homology modelling using the automodel class from
    modeller with a default parameter set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used
        for the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation (default 1)
    n_models : int
        Number of models generated (default 1)

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+---------------------------------------+
    | modelling             | value                                 |
    | parameter             |                                       |
    +=======================+=======================================+
    | model_class           | modeller.automodel.automodel          |
    +-----------------------+---------------------------------------+
    | library_schedule      | modeller.automodel.autosched.normal   |
    +-----------------------+---------------------------------------+
    | md_level              | modeller.automodel.refine.very_fast   |
    +-----------------------+---------------------------------------+
    | max_var_iterations    | 200                                   |
    +-----------------------+---------------------------------------+
    | repeat_optmization    | 1                                     |
    +-----------------------+---------------------------------------+
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # init parameters
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'automodel_default'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate models with the parameter set automodel_default.

        Returns
        -------
        None
        '''
        # set fixed parameters
        model_class = modeller.automodel.automodel
        library_schedule = modeller.automodel.autosched.normal
        max_var_iterations = 200
        md_level = modeller.automodel.refine.very_fast
        repeat_optimization = 1
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads)


class Routine_automodel_slow(Routine_modeller):
    '''
    Class for performing homology modelling using the automodel class from
    modeller with a slow parameter set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used
        for the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation
    n_models : int
        Number of models generated

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+---------------------------------------+
    | modelling             | value                                 |
    | parameter             |                                       |
    +=======================+=======================================+
    | model_class           | modeller.automodel.automodel          |
    +-----------------------+---------------------------------------+
    | library_schedule      | modeller.automodel.autosched.slow     |
    +-----------------------+---------------------------------------+
    | md_level              | modeller.automodel.refine.very_slow   |
    +-----------------------+---------------------------------------+
    | max_var_iterations    | 400                                   |
    +-----------------------+---------------------------------------+
    | repeat_optmization    | 3                                     |
    +-----------------------+---------------------------------------+
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # init parameters
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'automodel_slow'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate models with the parameter set automodel_slow.

        Returns
        -------
        None
        '''
        # set fixed parameters
        model_class = modeller.automodel.automodel
        library_schedule = modeller.automodel.autosched.slow
        max_var_iterations = 400
        md_level = modeller.automodel.refine.very_slow
        repeat_optimization = 3
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads)


class Routine_altmod_default(Routine_modeller):
    '''
    Class for performing homology modelling using the
    Automodel_statistical_potential class from altmod with a default parameter
    set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation
    n_models : int
        Number of models generated

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : list
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+-------------------------------------------+
    | modelling             | value                                     |
    | parameter             |                                           |
    +=======================+===========================================+
    | model_class           | altmod.Automodel_statistical_potential    |
    +-----------------------+-------------------------------------------+
    | library_schedule      | modeller.automodel.autosched.normal       |
    +-----------------------+-------------------------------------------+
    | md_level              | modeller.automodel.refine.very_fast       |
    +-----------------------+-------------------------------------------+
    | max_var_iterations    | 200                                       |
    +-----------------------+-------------------------------------------+
    | repeat_optmization    | 1                                         |
    +-----------------------+-------------------------------------------+

    Autmodel_statistical_potential uses the DOPE potential for model
    refinement.
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: [list, tuple], tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # check dependencies
        # (modeller is already checked in Routine_modeller.__init__)
        dependencies = ['altmod']
        self._check_dependencies(dependencies)
        # init parameters
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'altmod_default'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate models with the parameter set altmod_default.

        Returns
        -------
        None
        '''
        # set fixed parameters
        model_class = altmod.Automodel_statistical_potential
        library_schedule = modeller.automodel.autosched.normal
        max_var_iterations = 200
        md_level = modeller.automodel.refine.very_fast
        repeat_optimization = 1
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads)


class Routine_altmod_slow(Routine_modeller):
    '''
    Class for performing homology modelling using the
    Automodel_statistical_potential class from altmod with a slow parameter
    set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation
    n_models : int
        Number of models generated

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : list
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+-------------------------------------------+
    | modelling             | value                                     |
    | parameter             |                                           |
    +=======================+===========================================+
    | model_class           | altmod.Automodel_statistical_potential    |
    +-----------------------+-------------------------------------------+
    | library_schedule      | modeller.automodel.autosched.slow         |
    +-----------------------+-------------------------------------------+
    | md_level              | modeller.automodel.refine.very_slow       |
    +-----------------------+-------------------------------------------+
    | max_var_iterations    | 400                                       |
    +-----------------------+-------------------------------------------+
    | repeat_optmization    | 3                                         |
    +-----------------------+-------------------------------------------+

    Autmodel_statistical_potential uses the DOPE potential for model
    refinement.
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: [list, tuple], tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # check dependencies
        # (modeller is already checked in Routine_modeller.__init__)
        dependencies = ['altmod']
        self._check_dependencies(dependencies)
        # init
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'altmod_slow'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate models with the parameter set altmod_slow.

        Returns
        -------
        None
        '''
        # set fixed parameters
        model_class = altmod.Automodel_statistical_potential
        library_schedule = modeller.automodel.autosched.slow
        max_var_iterations = 400
        md_level = modeller.automodel.refine.very_slow
        repeat_optimization = 3
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads)


class Routine_promod3(Routine):
    '''
    Class for performing homology modelling using the ProMod3 engine with
    default parameters.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : iterable
        The iterable containing the identifier of the template used for the
        modelling

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : iterable
        The iterable containing the identifier of the template used for the
        modelling
    tag : str
        The identifier associated with a specific execution of the routine
    routine : str
        The identifier associated with this specific routine: promod3
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies
    ValueError
        Number of given templates is not 1
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str) -> None:
        # check dependencies
        dependencies = ['ost', 'promod3']
        self._check_dependencies(dependencies)
        # check number of templates
        if len(templates) != 1:
            raise ValueError('ProMod3 only accepts one template structure, '
                             '{} were given.'.format(len(templates)))
        # init
        Routine.__init__(self, alignment, target, templates, tag)
        self.routine = 'promod3'

    def generate_models(self) -> None:
        '''
        Generate models with the ProMod3 engine with default parameters.

        Returns
        -------
        None
        '''
        # process alignment for promod3
        self.alignment.rename_sequence(self.target, 'Target')
        self.alignment.select_sequences(['Target'] + self.templates)
        self.alignment.remove_redundant_gaps()
        self.alignment.write_fasta('.tmp.fasta')

        # modelling
        template = ost.io.LoadPDB(self.templates[0] + '.pdb')
        aln = ost.io.LoadAlignment('.tmp.fasta')
        aln.AttachView(1, template.CreateFullView())
        mhandle = promod3.modelling.BuildRawModel(aln)
        final_model = promod3.modelling.BuildFromRawModel(mhandle)
        ost.io.SavePDB(final_model, 'model.pdb')

        # capturing output
        self.models.append(
            Model(os.path.realpath(os.path.expanduser(('model.pdb'))),
                  self.tag, self.routine))

        # cleaning up
        self._rename_models()
        self._remove_files('.tmp*')


###############################
# Complex Modelling

# define custom modelling classes
if _IMPORTS['modeller'] == True:
    class _AutomodelComplex(modeller.automodel.automodel):
        '''
        Modelling class adapted for complex modelling.

        Based on the automodel class from modeller.

        Multi-chain models are assigned different chain identifiers for the
        individual chains, with each chain starting at residue 1.
        '''
        def special_patches(self, aln):
            '''
            Enforce custom renaming of chains and renumbering of residues.
            '''
            # see https://salilab.org/modeller/manual/node30.html
            segment_ids = list(string.ascii_uppercase)
            renumber_residues = [1] * len(segment_ids)
            self.rename_segments(segment_ids=segment_ids,
                                 renumber_residues=renumber_residues)


if _IMPORTS['altmod'] == True:
    class _AltmodComplex(altmod.Automodel_statistical_potential):
        '''
        Modelling class adapted for complex modelling.

        Based on the Automodel_statistical_potential class from altmod.

        Multi-chain models are assigned different chain identifiers for the
        individual chains, with each chain starting at residue 1.
        '''
        def special_patches(self, aln):
            '''
            Enforce custom renaming of chains and renumbering of residues.
            '''
            # see https://salilab.org/modeller/manual/node30.html
            segment_ids = list(string.ascii_uppercase)
            renumber_residues = [1] * len(segment_ids)
            self.rename_segments(segment_ids=segment_ids,
                                 renumber_residues=renumber_residues)


class Routine_complex_automodel_default(Routine_modeller):
    '''
    Class for performing homology modelling of complexes using the automodel
    class from modeller with a default parameter set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used
        for the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation (default 1)
    n_models : int
        Number of models generated (default 1)

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+-------------------------------------------+
    | modelling             | value                                     |
    | parameter             |                                           |
    +=======================+===========================================+
    | model_class           | modeller.automodel.autmodel               |
    +-----------------------+-------------------------------------------+
    | library_schedule      | modeller.automodel.autosched.normal       |
    +-----------------------+-------------------------------------------+
    | md_level              | modeller.automodel.refine.very_fast       |
    +-----------------------+-------------------------------------------+
    | max_var_iterations    | 200                                       |
    +-----------------------+-------------------------------------------+
    | repeat_optmization    | 1                                         |
    +-----------------------+-------------------------------------------+
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: [list, tuple], tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # init parameters
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'complex_automodel_default'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate complex models with the parameter set automodel_default.

        Returns
        -------
        None
        '''
        # set fixed parameters
        model_class = _AutomodelComplex
        library_schedule = modeller.automodel.autosched.normal
        max_var_iterations = 200
        md_level = modeller.automodel.refine.very_fast
        repeat_optimization = 1
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads, True)


class Routine_complex_automodel_slow(Routine_modeller):
    '''
    Class for performing homology modelling of complexes using the automodel
    class from modeller with a slow parameter set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used
        for the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation (default 1)
    n_models : int
        Number of models generated (default 1)

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : Iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+---------------------------------------+
    | modelling             | value                                 |
    | parameter             |                                       |
    +=======================+=======================================+
    | model_class           | modeller.automodel.automodel          |
    +-----------------------+---------------------------------------+
    | library_schedule      | modeller.automodel.autosched.slow     |
    +-----------------------+---------------------------------------+
    | md_level              | modeller.autmodel.refine.very_slow    |
    +-----------------------+---------------------------------------+
    | max_var_iterations    | 400                                   |
    +-----------------------+---------------------------------------+
    | repeat_optmization    | 3                                     |
    +-----------------------+---------------------------------------+
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # init parameters
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'complex_automodel_default'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate complex models with the parameters set automodel_slow.

        Returns
        -------
        None
        '''
        # set fixed parameters
        model_class = _AutomodelComplex
        library_schedule = modeller.automodel.autosched.slow
        max_var_iterations = 400
        md_level = modeller.automodel.refine.very_slow
        repeat_optimization = 3
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads, True)


class Routine_complex_altmod_default(Routine_modeller):
    '''
    Class for performing homology modelling of complexes using the
    Automodel_statistical_potential class from altmod with a default parameter
    set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation
    n_models : int
        Number of models generated

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : list
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+-------------------------------------------+
    | modelling             | value                                     |
    | parameter             |                                           |
    +=======================+===========================================+
    | model_class           | altmod.Automodel_statistical_potential    |
    +-----------------------+-------------------------------------------+
    | library_schedule      | modeller.automodel.autosched.normal       |
    +-----------------------+-------------------------------------------+
    | md_level              | modeller.automodel.refine.very_fast       |
    +-----------------------+-------------------------------------------+
    | max_var_iterations    | 200                                       |
    +-----------------------+-------------------------------------------+
    | repeat_optmization    | 1                                         |
    +-----------------------+-------------------------------------------+

    Autmodel_statistical_potential uses the DOPE potential for model
    refinement.
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # check dependencies
        # (modeller is already checked in Routine_modeller.__init__)
        dependencies = ['altmod']
        self._check_dependencies(dependencies)
        # init
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'complex_altmod_default'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate complex models with the parameter set altmod_default.

        Returns
        -------
        None
        '''
        model_class = _AltmodComplex
        library_schedule = modeller.automodel.autosched.normal
        max_var_iterations = 200
        md_level = modeller.automodel.refine.very_fast
        repeat_optimization = 1
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads, True)


class Routine_complex_altmod_slow(Routine_modeller):
    '''
    Class for performing homology modelling of complexes using the
    Automodel_statistical_potential class from altmod with a slow parameter
    set.

    Parameters
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : iterable
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used in model generation
    n_models : int
        Number of models generated

    Attributes
    ----------
    alignment : Alignment
        The alignment object that will be used for modelling
    target : str
        The identifier of the protein to model
    templates : list
        The iterable containing the identifier(s) of the template(s) used for
        the modelling
    tag : str
        The identifier associated with a specific execution of the routine
    n_threads : int
        Number of threads used for model generation
    n_models : int
        Number of models generated
    routine : str
        The identifier associated with a specific routine
    models : list
        List of models generated by the execution of this routine

    Raises
    ------
    ImportError
        Unable to import dependencies

    Notes
    -----
    The following modelling parameters can be set when initializing this
    Routine object:

    * n_models
    * n_threads

    The following modelling parameters are set for this class:

    +-----------------------+-------------------------------------------+
    | modelling             | value                                     |
    | parameter             |                                           |
    +=======================+===========================================+
    | model_class           | altmod.Automodel_statistical_potential    |
    +-----------------------+-------------------------------------------+
    | library_schedule      | modeller.automodel.autosched.slow         |
    +-----------------------+-------------------------------------------+
    | md_level              | modeller.automodel.refine.very_slow       |
    +-----------------------+-------------------------------------------+
    | max_var_iterations    | 400                                       |
    +-----------------------+-------------------------------------------+
    | repeat_optmization    | 3                                         |
    +-----------------------+-------------------------------------------+

    Autmodel_statistical_potential uses the DOPE potential for model
    refinement.
    '''
    def __init__(self, alignment: typing.Type['Alignment'], target: str,
                 templates: typing.Iterable, tag: str, n_threads: int = 1,
                 n_models: int = 1) -> None:
        # check dependencies
        # (modeller is already checked in Routine_modeller.__init__)
        dependencies = ['altmod']
        self._check_dependencies(dependencies)
        # init
        Routine_modeller.__init__(self, alignment, target, templates, tag)
        self.routine = 'complex_altmod_default'
        # modelling parameters
        self.n_threads = n_threads
        self.n_models = n_models

    def generate_models(self) -> None:
        '''
        Generate complex models with the parameter set altmod_slow.

        Returns
        -------
        None
        '''
        model_class = _AltmodComplex
        library_schedule = modeller.automodel.autosched.slow
        max_var_iterations = 400
        md_level = modeller.automodel.refine.very_slow
        repeat_optimization = 3
        # run model generation
        self._generate_models(
            model_class, self.n_models, library_schedule, max_var_iterations,
            md_level, repeat_optimization, self.n_threads, True)
