"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: () => ({ bucket, zipObjectKey })
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path) {
        return {
            bind(context) {
                let id = 1;
                while (context.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(context, `Asset${id}`, { path });
                if (!asset.isZipArchive) {
                    throw new Error(`Asset path must be either a .zip file or a directory`);
                }
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey
                };
            }
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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