"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const path = require("path");
const s3deploy = require("../lib");
const lib_1 = require("../lib");
module.exports = {
    'deploy from local directory asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536",
                    "Arn"
                ]
            },
            "SourceBucketNames": [{
                    "Ref": "AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A"
                }],
            "SourceObjectKeys": [{
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::Select": [
                                    0,
                                    {
                                        "Fn::Split": [
                                            "||",
                                            {
                                                "Ref": "AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C"
                                            }
                                        ]
                                    }
                                ]
                            },
                            {
                                "Fn::Select": [
                                    1,
                                    {
                                        "Fn::Split": [
                                            "||",
                                            {
                                                "Ref": "AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C"
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    ]
                }],
            "DestinationBucketName": {
                "Ref": "DestC383B82A"
            }
        }));
        test.done();
    },
    'deploy from local directory assets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [
                s3deploy.Source.asset(path.join(__dirname, 'my-website')),
                s3deploy.Source.asset(path.join(__dirname, 'my-website-second'))
            ],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536",
                    "Arn"
                ]
            },
            "SourceBucketNames": [
                {
                    "Ref": "AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A"
                },
                {
                    "Ref": "AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3Bucket99793559"
                }
            ],
            "SourceObjectKeys": [
                {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::Select": [
                                    0,
                                    {
                                        "Fn::Split": [
                                            "||",
                                            {
                                                "Ref": "AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C"
                                            }
                                        ]
                                    }
                                ]
                            },
                            {
                                "Fn::Select": [
                                    1,
                                    {
                                        "Fn::Split": [
                                            "||",
                                            {
                                                "Ref": "AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C"
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    ]
                },
                {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::Select": [
                                    0,
                                    {
                                        "Fn::Split": [
                                            "||",
                                            {
                                                "Ref": "AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665"
                                            }
                                        ]
                                    }
                                ]
                            },
                            {
                                "Fn::Select": [
                                    1,
                                    {
                                        "Fn::Split": [
                                            "||",
                                            {
                                                "Ref": "AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665"
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    ]
                }
            ],
            "DestinationBucketName": {
                "Ref": "DestC383B82A"
            }
        }));
        test.done();
    },
    'fails if local asset is a non-zip file'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // THEN
        test.throws(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
            destinationBucket: bucket,
        }), /Asset path must be either a \.zip file or a directory/);
        test.done();
    },
    'deploy from a local .zip file'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
        });
        test.done();
    },
    'retainOnDelete can be used to retain files when resource is deleted'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            retainOnDelete: true,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            RetainOnDelete: true
        }));
        test.done();
    },
    'object metadata can be given'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            metadata: { "A": "1", "b": "2" },
            contentType: "text/html",
            contentLanguage: "en",
            storageClass: lib_1.StorageClass.INTELLIGENT_TIERING,
            contentDisposition: "inline",
            serverSideEncryption: lib_1.ServerSideEncryption.AES_256,
            cacheControl: [lib_1.CacheControl.setPublic(), lib_1.CacheControl.maxAge(cdk.Duration.hours(1))],
            expires: lib_1.Expires.after(cdk.Duration.hours(12))
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            UserMetadata: { 'x-amzn-meta-a': '1', 'x-amzn-meta-b': '2' },
            SystemMetadata: {
                'content-type': 'text/html',
                'content-language': 'en',
                'content-disposition': 'inline',
                'storage-class': 'INTELLIGENT_TIERING',
                'server-side-encryption': 'AES256',
                'cache-control': 'public, max-age=3600',
                'expires': lib_1.Expires.after(cdk.Duration.hours(12)).value
            }
        }));
        test.done();
    },
    'distribution can be used to provide a CloudFront distribution for invalidation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: bucket
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ]
        });
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            distribution,
            distributionPaths: ['/images/*']
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            DistributionId: {
                "Ref": "DistributionCFDistribution882A7313"
            },
            DistributionPaths: ['/images/*']
        }));
        test.done();
    },
    'invalidation can happen without distributionPaths provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: bucket
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ]
        });
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            distribution,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            DistributionId: {
                "Ref": "DistributionCFDistribution882A7313"
            },
        }));
        test.done();
    },
    'fails if distribution paths provided but not distribution ID'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // THEN
        test.throws(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
            destinationBucket: bucket,
            distributionPaths: ['/images/*']
        }), /Distribution must be specified if distribution paths are specified/);
        test.done();
    },
    'lambda execution role gets permissions to read from the source bucket and read/write in destination'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const source = new s3.Bucket(stack, 'Source');
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.bucket(source, 'file.zip')],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "s3:GetObject*",
                            "s3:GetBucket*",
                            "s3:List*"
                        ],
                        "Effect": "Allow",
                        "Resource": [
                            {
                                "Fn::GetAtt": [
                                    "Source71E471F1",
                                    "Arn"
                                ]
                            },
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        {
                                            "Fn::GetAtt": [
                                                "Source71E471F1",
                                                "Arn"
                                            ]
                                        },
                                        "/*"
                                    ]
                                ]
                            }
                        ]
                    },
                    {
                        "Action": [
                            "s3:GetObject*",
                            "s3:GetBucket*",
                            "s3:List*",
                            "s3:DeleteObject*",
                            "s3:PutObject*",
                            "s3:Abort*"
                        ],
                        "Effect": "Allow",
                        "Resource": [
                            {
                                "Fn::GetAtt": [
                                    "DestC383B82A",
                                    "Arn"
                                ]
                            },
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        {
                                            "Fn::GetAtt": [
                                                "DestC383B82A",
                                                "Arn"
                                            ]
                                        },
                                        "/*"
                                    ]
                                ]
                            }
                        ]
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRoleDefaultPolicy88902FDF",
            "Roles": [
                {
                    "Ref": "CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRole89A01265"
                }
            ]
        }));
        test.done();
    },
    'memoryLimit can be used to specify the memory limit for the deployment resource handler'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        // we define 3 deployments with 2 different memory configurations
        new s3deploy.BucketDeployment(stack, 'Deploy256-1', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            memoryLimit: 256
        });
        new s3deploy.BucketDeployment(stack, 'Deploy256-2', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            memoryLimit: 256
        });
        new s3deploy.BucketDeployment(stack, 'Deploy1024', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            memoryLimit: 1024
        });
        // THEN
        // we expect to find only two handlers, one for each configuration
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { MemorySize: 256 }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { MemorySize: 1024 }));
        test.done();
    },
    'deployment allows custom role to be supplied'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        const existingRole = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazon.com')
        });
        // WHEN
        new s3deploy.BucketDeployment(stack, 'DeployWithRole', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            role: existingRole
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 1));
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 1));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Role": {
                "Fn::GetAtt": [
                    "Role1ABCC5F0",
                    "Arn"
                ]
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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