import yaml
from pyapollo.apollo_client import ApolloClient
from typing import Any


class ConfigBase:
    def _get_config_field(self, field: str) -> Any:
        # 获取field指
        raise NotImplementedError

    def _get_field_type(self, field: str) -> type:
        # 获取field的类型
        if field in self.__annotations__:
            return self.__annotations__[field]

    # def __getitem__(self, item):
    #     if item in self.__annotations__.keys():
    #         return self._get_config_field(item)

    @classmethod
    def _parse_field_type(cls, _type, value):
        func = getattr(cls, f"_parse_{_type.__name__}", cls._parse_str)
        return func(value)

    @staticmethod
    def _parse_bool(value: str) -> bool:
        return value.lower() not in frozenset(("0", "", "false"))

    @staticmethod
    def _parse_str(value) -> str:
        return str(value)

    @staticmethod
    def _parse_int(value: str) -> int:
        try:
            return int(value)
        except ValueError:
            return 0

    @staticmethod
    def _parse_float(value: str) -> float:
        try:
            return float(value)
        except ValueError:
            return 0

    @staticmethod
    def _parse_list(value: str) -> list:
        return value.split(",")

    def fetch(self):
        for field in self.__annotations__.keys():
            setattr(self, field, self._get_config_field(field))


class YamlConfig(ConfigBase):
    def __init__(self, yaml_path):
        with open(yaml_path) as fin:
            self.config_dict = yaml.load(fin.read(), Loader=yaml.FullLoader)

    def _get_config_field(self, field: str):
        if field in self.config_dict:
            return self.config_dict[field]


class ApolloConfig(ConfigBase):
    def __init__(self, app_id: str, server_url: str, cluster: str = "default"):
        self.apollo_client = ApolloClient(
            app_id=app_id, config_server_url=server_url, cluster=cluster
        )

    def _get_config_field(self, field: str):
        value = self.apollo_client.get_value(field)
        if value is None:
            value = getattr(self, field)
        return self._parse_field_type(self._get_field_type(field), value)


# class Config(ApolloConfig):
#     ES_URI: list = None
#     MONGO_URI: str = "mongo_uri"
#
#
# config = Config("insnail-qa-robot", "http://10.0.0.176:8080", "default")
# config.fetch()
