"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackSet = exports.RegionConcurrencyType = exports.Capability = exports.DeploymentType = exports.StackSetTarget = exports.StackSetTemplate = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Represents a StackSet CloudFormation template
 */
class StackSetTemplate {
    /**
     * @param stack the stack to use as the base for the stackset template
     * @returns StackSetTemplate
     */
    static fromStackSetStack(stack) {
        return { templateUrl: stack._getTemplateUrl() };
    }
}
exports.StackSetTemplate = StackSetTemplate;
_a = JSII_RTTI_SYMBOL_1;
StackSetTemplate[_a] = { fqn: "cdk-stacksets.StackSetTemplate", version: "0.0.58" };
/**
 * The type of StackSet account filter
 */
var AccountFilterType;
(function (AccountFilterType) {
    /**
     * _Only_ deploys to specified accounts
     */
    AccountFilterType["INTERSECTION"] = "INTERSECTION";
    /**
     * _Does not_ deploy to the specified accounts
     */
    AccountFilterType["DIFFERENCE"] = "DIFFERENCE";
    /**
     * (default value) include both OUs and Accounts
     * Allows you to deploy to an OU _and_ specific accounts in a different OU
     */
    AccountFilterType["UNION"] = "UNION";
    /**
     * Only deploy to accounts in an OU
     */
    AccountFilterType["NONE"] = "NONE";
})(AccountFilterType || (AccountFilterType = {}));
/**
 * Which organizational units and/or accounts the stack set
 * should be deployed to.
 *
 * `fromAccounts` can be used to deploy the stack set to specific AWS accounts
 *
 * `fromOrganizationalUnits` can be used to deploy the stack set to specific organizational units
 * and optionally include additional accounts from other OUs, or exclude accounts from the specified
 * OUs
 *
 * @example
 * // deploy to specific accounts
 * StackSetTarget.fromAccounts({
 *   accounts: ['11111111111', '22222222222'],
 *   regions: ['us-east-1', 'us-east-2'],
 * });
 *
 * // deploy to OUs and 1 additional account
 * StackSetTarget.fromOrganizationalUnits({
 *   regions: ['us-east-1', 'us-east-2'],
 *   organizationalUnits: ['ou-1111111', 'ou-2222222'],
 *   additionalAccounts: ['33333333333'],
 * });
 *
 * // deploy to OUs but exclude 1 account
 * StackSetTarget.fromOrganizationalUnits({
 *   regions: ['us-east-1', 'us-east-2'],
 *   organizationalUnits: ['ou-1111111', 'ou-2222222'],
 *   excludeAccounts: ['11111111111'],
 * });
 */
class StackSetTarget {
    /**
     * Deploy the StackSet to a list of accounts
     *
     * @example
     * StackSetTarget.fromAccounts({
     *   accounts: ['11111111111', '22222222222'],
     *   regions: ['us-east-1', 'us-east-2'],
     * });
     */
    static fromAccounts(options) {
        return new AccountsTarget(options);
    }
    ;
    /**
     * Deploy the StackSet to a list of AWS Organizations organizational units.
     *
     * You can optionally include/exclude individual AWS accounts.
     *
     * @example
     * StackSetTarget.fromOrganizationalUnits({
     *   regions: ['us-east-1', 'us-east-2'],
     *   organizationalUnits: ['ou-1111111', 'ou-2222222'],
     * });
     */
    static fromOrganizationalUnits(options) {
        return new OrganizationsTarget(options);
    }
    /**
     * @internal
     */
    _renderParameters(parameters) {
        if (!parameters)
            return undefined;
        const overrides = [];
        for (const [key, value] of Object.entries(parameters ?? {})) {
            overrides.push({
                parameterKey: key,
                parameterValue: value,
            });
        }
        return overrides;
    }
}
exports.StackSetTarget = StackSetTarget;
_b = JSII_RTTI_SYMBOL_1;
StackSetTarget[_b] = { fqn: "cdk-stacksets.StackSetTarget", version: "0.0.58" };
class AccountsTarget extends StackSetTarget {
    constructor(options) {
        super();
        this.options = options;
    }
    _bind(_scope, _options = {}) {
        return {
            regions: this.options.regions,
            parameterOverrides: this._renderParameters(this.options.parameterOverrides),
            accountFilterType: AccountFilterType.INTERSECTION,
            accounts: this.options.accounts,
        };
    }
}
class OrganizationsTarget extends StackSetTarget {
    constructor(options) {
        super();
        this.options = options;
    }
    _bind(_scope, _options = {}) {
        if (this.options.excludeAccounts && this.options.additionalAccounts) {
            throw new Error("cannot specify both 'excludeAccounts' and 'additionalAccounts'");
        }
        const filterType = this.options.additionalAccounts ? AccountFilterType.UNION
            : this.options.excludeAccounts ? AccountFilterType.DIFFERENCE
                : AccountFilterType.NONE;
        return {
            regions: this.options.regions,
            parameterOverrides: this._renderParameters(this.options.parameterOverrides),
            accountFilterType: filterType,
            organizationalUnits: this.options.organizationalUnits,
            accounts: this.options.additionalAccounts ?? this.options.excludeAccounts,
        };
    }
}
var PermissionModel;
(function (PermissionModel) {
    PermissionModel["SERVICE_MANAGED"] = "SERVICE_MANAGED";
    PermissionModel["SELF_MANAGED"] = "SELF_MANAGED";
})(PermissionModel || (PermissionModel = {}));
class DeploymentType {
    /**
     * StackSets deployed using service managed permissions allow you to deploy
     * StackSet instances to accounts within an AWS Organization. Using this module
     * AWS Organizations will handle creating the necessary IAM roles and setting up the
     * required permissions.
     *
     * This model also allows you to enable automated deployments which allows the StackSet
     * to be automatically deployed to new accounts that are added to your organization in the future.
     *
     * This model requires you to be operating in either the AWS Organizations management account
     * or the delegated administrator account
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stacksets-concepts-stackset-permission-models
     */
    static serviceManaged(options = {}) {
        return new ManagedDeploymentType(options);
    }
    /**
     * StackSets deployed using the self managed model require you to create the necessary
     * IAM roles in the source and target AWS accounts and to setup the required IAM permissions.
     *
     * Using this model you can only deploy to AWS accounts that have the necessary IAM roles/permissions
     * pre-created.
     */
    static selfManaged(options = {}) {
        return new SelfDeploymentType(options);
    }
}
exports.DeploymentType = DeploymentType;
_c = JSII_RTTI_SYMBOL_1;
DeploymentType[_c] = { fqn: "cdk-stacksets.DeploymentType", version: "0.0.58" };
class ManagedDeploymentType extends DeploymentType {
    constructor(options) {
        super();
        this.options = options;
    }
    _bind(_scope, _options = {}) {
        const autoDeployEnabled = this.options?.autoDeployEnabled ?? true;
        if (!autoDeployEnabled && this.options?.autoDeployRetainStacks !== undefined) {
            throw new Error('autoDeployRetainStacks only applies if autoDeploy is enabled');
        }
        return {
            permissionsModel: PermissionModel.SERVICE_MANAGED,
            autoDeployEnabled,
            callAs: (this.options?.delegatedAdmin ?? true) ? CallAs.DELEGATED_ADMIN : CallAs.SELF,
            autoDeployRetainStacks: autoDeployEnabled ? (this.options?.autoDeployRetainStacks ?? true)
                : undefined,
        };
    }
}
class SelfDeploymentType extends DeploymentType {
    constructor(options) {
        super();
        this.options = options;
    }
    _bind(_scope, _options = {}) {
        return {
            permissionsModel: PermissionModel.SELF_MANAGED,
            adminRole: this.options?.adminRole,
            executionRoleName: this.options?.executionRoleName,
        };
    }
}
/**
 * Indicates whether a service managed stackset is deployed from the
 * AWS Organizations management account or the delegated admin account
 */
var CallAs;
(function (CallAs) {
    /**
     * The StackSet is deployed from the Delegated Admin account
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html
     */
    CallAs["DELEGATED_ADMIN"] = "DELEGATED_ADMIN";
    /**
     * The StackSet is deployed from the Management account
     */
    CallAs["SELF"] = "SELF";
})(CallAs || (CallAs = {}));
/**
 * StackSets that contains certain functionality require an explicit acknowledgement
 * that the stack contains these capabilities.
 */
var Capability;
(function (Capability) {
    /**
     * Required if the stack contains IAM resources with custom names
     */
    Capability["NAMED_IAM"] = "CAPABILITY_NAMED_IAM";
    /**
     * Required if the stack contains IAM resources. If the IAM resources
     * also have custom names then specify {@link Capability.NAMED_IAM} instead.
     */
    Capability["IAM"] = "CAPABILITY_IAM";
    /**
     * Required if the stack contains macros. Not supported if deploying
     * a service managed stackset.
     */
    Capability["AUTO_EXPAND"] = "CAPABILITY_AUTO_EXPAND";
})(Capability = exports.Capability || (exports.Capability = {}));
var RegionConcurrencyType;
(function (RegionConcurrencyType) {
    RegionConcurrencyType["SEQUENTIAL"] = "SEQUENCIAL";
    RegionConcurrencyType["PARALLEL"] = "PARALLEL";
})(RegionConcurrencyType = exports.RegionConcurrencyType || (exports.RegionConcurrencyType = {}));
/**
 * AWS Regions introduced after March 20, 2019, such as Asia Pacific (Hong Kong), are disabled by default.
 * Be aware that to deploy stack instances into a target account that resides in a Region that's disabled by default,
 * you will also need to include the regional service principal for that Region.
 * Each Region that's disabled by default will have its own regional service principal.
 */
const ENABLED_REGIONS = [
    'us-east-1',
    'eu-west-1',
    'us-west-1',
    'ap-southeast-1',
    'ap-northeast-1',
    'us-gov-west-1',
    'us-west-2',
    'sa-east-1',
    'ap-southeast-2',
    'cn-north-1',
    'eu-central-1',
    'ap-northeast-2',
    'ap-south-1',
    'us-east-2',
    'ca-central-1',
    'eu-west-2',
    'cn-northwest-1',
    'eu-west-3',
    'ap-northeast-3',
    'us-gov-east-1',
    'eu-north-1',
    'eu-south-2',
];
// disabled regions
// 'af-south-1', // Africa (Cape Town)
// 'ap-southeast-3', // Asia Pacific (Jakarta)
// 'ap-east-1', // Asia Pacific (Hong Kong)
// 'eu-south-1', // Europe (Milan)
// 'me-south-1', // Middle East (Bahrain)
class StackSet extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stackSetName ?? aws_cdk_lib_1.Lazy.string({ produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, {}) }),
        });
        this.stackInstances = [];
        const deploymentTypeConfig = (props.deploymentType ?? DeploymentType.selfManaged())._bind(this);
        if (deploymentTypeConfig.permissionsModel === PermissionModel.SELF_MANAGED) {
            this._role = deploymentTypeConfig.adminRole ?? new aws_cdk_lib_1.aws_iam.Role(scope, 'AdminRole', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('cloudformation'),
            });
            this._role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
                effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                actions: ['sts:AssumeRole'],
                resources: [
                    `arn:aws:iam::*:role/${deploymentTypeConfig.executionRoleName ?? 'AWSCloudFormationStackSetExecutionRole'}`,
                ],
            }));
        }
        if (props.capabilities?.includes(Capability.AUTO_EXPAND) && deploymentTypeConfig.permissionsModel === PermissionModel.SERVICE_MANAGED) {
            throw new Error('service managed stacksets do not current support the "AUTO_EXPAND" capability');
        }
        this.permissionModel = deploymentTypeConfig.permissionsModel;
        this.addTarget(props.target);
        new aws_cdk_lib_1.aws_cloudformation.CfnStackSet(this, 'Resource', {
            autoDeployment: undefinedIfNoKeys({
                enabled: deploymentTypeConfig.autoDeployEnabled,
                retainStacksOnAccountRemoval: deploymentTypeConfig.autoDeployRetainStacks,
            }),
            executionRoleName: deploymentTypeConfig.executionRoleName,
            administrationRoleArn: this._role?.roleArn,
            description: props.description,
            managedExecution: {
                Active: props.managedExecution ?? true,
            },
            operationPreferences: undefinedIfNoKeys({
                regionConcurrencyType: props.operationPreferences?.regionConcurrencyType,
                maxConcurrentPercentage: props.operationPreferences?.maxConcurrentPercentage,
                failureTolerancePercentage: props.operationPreferences?.failureTolerancePercentage,
            }),
            stackSetName: this.physicalName,
            capabilities: props.capabilities,
            permissionModel: deploymentTypeConfig.permissionsModel,
            callAs: deploymentTypeConfig.callAs,
            templateUrl: props.template.templateUrl,
            stackInstancesGroup: aws_cdk_lib_1.Lazy.any({ produce: () => { return this.stackInstances; } }),
        });
    }
    get role() {
        if (!this._role) {
            throw new Error('service managed StackSets do not have an associated role');
        }
        return this._role;
    }
    addTarget(target) {
        const targetConfig = target._bind(this);
        if (this._role && this._role instanceof aws_cdk_lib_1.aws_iam.Role) {
            const disabledPrincipals = [];
            targetConfig.regions.forEach(region => {
                if (!ENABLED_REGIONS.includes(region)) {
                    disabledPrincipals.push(new aws_cdk_lib_1.aws_iam.ServicePrincipal(`cloudformation.${region}.amazonaws.com`));
                }
            });
            if (disabledPrincipals.length > 0) {
                this._role.assumeRolePolicy?.addStatements(new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ['sts:AssumeRole'],
                    principals: disabledPrincipals,
                }));
            }
        }
        this.stackInstances.push({
            regions: targetConfig.regions,
            parameterOverrides: targetConfig.parameterOverrides,
            deploymentTargets: {
                accounts: targetConfig.accounts,
                accountFilterType: this.permissionModel === PermissionModel.SERVICE_MANAGED
                    ? targetConfig.accountFilterType
                    : undefined,
                organizationalUnitIds: targetConfig.organizationalUnits,
            },
        });
    }
}
exports.StackSet = StackSet;
_d = JSII_RTTI_SYMBOL_1;
StackSet[_d] = { fqn: "cdk-stacksets.StackSet", version: "0.0.58" };
function undefinedIfNoKeys(obj) {
    const allUndefined = Object.values(obj).every(val => val === undefined);
    return allUndefined ? undefined : obj;
}
//# sourceMappingURL=data:application/json;base64,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