"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appstream = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appstream](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonappstream2.0.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appstream extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appstream](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonappstream2.0.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appstream';
        this.accessLevelList = {
            "Write": [
                "AssociateFleet",
                "BatchAssociateUserStack",
                "BatchDisassociateUserStack",
                "CopyImage",
                "CreateDirectoryConfig",
                "CreateFleet",
                "CreateImageBuilder",
                "CreateImageBuilderStreamingURL",
                "CreateStack",
                "CreateStreamingURL",
                "CreateUsageReportSubscription",
                "CreateUser",
                "DeleteDirectoryConfig",
                "DeleteFleet",
                "DeleteImage",
                "DeleteImageBuilder",
                "DeleteImagePermissions",
                "DeleteStack",
                "DeleteUsageReportSubscription",
                "DeleteUser",
                "DisableUser",
                "DisassociateFleet",
                "EnableUser",
                "ExpireSession",
                "GetParametersForThemeAssetUpload",
                "StartFleet",
                "StartImageBuilder",
                "StopFleet",
                "StopImageBuilder",
                "Stream",
                "UpdateDirectoryConfig",
                "UpdateFleet",
                "UpdateImagePermissions",
                "UpdateStack"
            ],
            "Read": [
                "DescribeDirectoryConfigs",
                "DescribeFleets",
                "DescribeImageBuilders",
                "DescribeImagePermissions",
                "DescribeImages",
                "DescribeSessions",
                "DescribeStacks",
                "DescribeUsageReportSubscriptions",
                "DescribeUserStackAssociations",
                "DescribeUsers",
                "GetImageBuilders",
                "ListAssociatedFleets",
                "ListAssociatedStacks",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate the specified fleet with the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_AssociateFleet.html
     */
    toAssociateFleet() {
        this.to('appstream:AssociateFleet');
        return this;
    }
    /**
     * Grants permission to associate the specified users with the specified stacks. Users in a user pool cannot be assigned to stacks with fleets that are joined to an Active Directory domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchAssociateUserStack.html
     */
    toBatchAssociateUserStack() {
        this.to('appstream:BatchAssociateUserStack');
        return this;
    }
    /**
     * Grants permission to disassociate the specified users from the specified stacks
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_BatchDisassociateUserStack.html
     */
    toBatchDisassociateUserStack() {
        this.to('appstream:BatchDisassociateUserStack');
        return this;
    }
    /**
     * Grants permission to copy the specified image within the same Region or to a new Region within the same AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CopyImage.html
     */
    toCopyImage() {
        this.to('appstream:CopyImage');
        return this;
    }
    /**
     * Grants permission to create a Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateDirectoryConfig.html
     */
    toCreateDirectoryConfig() {
        this.to('appstream:CreateDirectoryConfig');
        return this;
    }
    /**
     * Grants permission to create a fleet. A fleet is a group of streaming instances from which applications are launched and streamed to users
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        this.to('appstream:CreateFleet');
        return this;
    }
    /**
     * Grants permission to create an image builder. An image builder is a virtual machine that is used to create an image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilder.html
     */
    toCreateImageBuilder() {
        this.to('appstream:CreateImageBuilder');
        return this;
    }
    /**
     * Grants permission to create a URL to start an image builder streaming session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateImageBuilderStreamingURL.html
     */
    toCreateImageBuilderStreamingURL() {
        this.to('appstream:CreateImageBuilderStreamingURL');
        return this;
    }
    /**
     * Grants permission to create a stack to start streaming applications to users. A stack consists of an associated fleet, user access policies, and storage configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        this.to('appstream:CreateStack');
        return this;
    }
    /**
     * Grants permission to create a temporary URL to start an AppStream 2.0 streaming session for the specified user. A streaming URL enables application streaming to be tested without user setup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateStreamingURL.html
     */
    toCreateStreamingURL() {
        this.to('appstream:CreateStreamingURL');
        return this;
    }
    /**
     * Grants permission to create a usage report subscription. Usage reports are generated daily
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUsageReportSubscription.html
     */
    toCreateUsageReportSubscription() {
        this.to('appstream:CreateUsageReportSubscription');
        return this;
    }
    /**
     * Grants permission to create a new user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('appstream:CreateUser');
        return this;
    }
    /**
     * Grants permission to delete the specified Directory Config object from AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteDirectoryConfig.html
     */
    toDeleteDirectoryConfig() {
        this.to('appstream:DeleteDirectoryConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        this.to('appstream:DeleteFleet');
        return this;
    }
    /**
     * Grants permission to delete the specified image. An image cannot be deleted when it is in use
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImage.html
     */
    toDeleteImage() {
        this.to('appstream:DeleteImage');
        return this;
    }
    /**
     * Grants permission to delete the specified image builder and release capacity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImageBuilder.html
     */
    toDeleteImageBuilder() {
        this.to('appstream:DeleteImageBuilder');
        return this;
    }
    /**
     * Grants permission to delete permissions for the specified private image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteImagePermissions.html
     */
    toDeleteImagePermissions() {
        this.to('appstream:DeleteImagePermissions');
        return this;
    }
    /**
     * Grants permission to delete the specified stack. After the stack is deleted, the application streaming environment provided by the stack is no longer available to users. Also, any reservations made for application streaming sessions for the stack are released
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        this.to('appstream:DeleteStack');
        return this;
    }
    /**
     * Grants permission to disable usage report generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUsageReportSubscription.html
     */
    toDeleteUsageReportSubscription() {
        this.to('appstream:DeleteUsageReportSubscription');
        return this;
    }
    /**
     * Grants permission to delete a user from the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('appstream:DeleteUser');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified Directory Config objects for AppStream 2.0, if the names for these objects are provided. Otherwise, all Directory Config objects in the account are described. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeDirectoryConfigs.html
     */
    toDescribeDirectoryConfigs() {
        this.to('appstream:DescribeDirectoryConfigs');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified fleets, if the fleet names are provided. Otherwise, all fleets in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeFleets.html
     */
    toDescribeFleets() {
        this.to('appstream:DescribeFleets');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImageBuilders.html
     */
    toDescribeImageBuilders() {
        this.to('appstream:DescribeImageBuilders');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes the permissions for shared AWS account IDs on a private image that you own
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImagePermissions.html
     */
    toDescribeImagePermissions() {
        this.to('appstream:DescribeImagePermissions');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified images, if the image names or image ARNs are provided. Otherwise, all images in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeImages.html
     */
    toDescribeImages() {
        this.to('appstream:DescribeImages');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes the streaming sessions for the specified stack and fleet. If a user ID is provided for the stack and fleet, only the streaming sessions for that user are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeSessions.html
     */
    toDescribeSessions() {
        this.to('appstream:DescribeSessions');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified stacks, if the stack names are provided. Otherwise, all stacks in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        this.to('appstream:DescribeStacks');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more usage report subscriptions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsageReportSubscriptions.html
     */
    toDescribeUsageReportSubscriptions() {
        this.to('appstream:DescribeUsageReportSubscriptions');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes the UserStackAssociation objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUserStackAssociations.html
     */
    toDescribeUserStackAssociations() {
        this.to('appstream:DescribeUserStackAssociations');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes users in the user pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        this.to('appstream:DescribeUsers');
        return this;
    }
    /**
     * Grants permission to disable the specified user in the user pool. This action does not delete the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisableUser.html
     */
    toDisableUser() {
        this.to('appstream:DisableUser');
        return this;
    }
    /**
     * Grants permission to disassociate the specified fleet from the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_DisassociateFleet.html
     */
    toDisassociateFleet() {
        this.to('appstream:DisassociateFleet');
        return this;
    }
    /**
     * Grants permission to enable a user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_EnableUser.html
     */
    toEnableUser() {
        this.to('appstream:EnableUser');
        return this;
    }
    /**
     * Grants permission to immediately stop the specified streaming session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ExpireSession.html
     */
    toExpireSession() {
        this.to('appstream:ExpireSession');
        return this;
    }
    /**
     * Grants permission to retrieve a list that describes one or more specified image builders, if the image builder names are provided. Otherwise, all image builders in the account are described
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/controlling-administrator-access-with-policies-service-linked-roles.html#managed-policies-required-to-access-appstream-resources
     */
    toGetImageBuilders() {
        this.to('appstream:GetImageBuilders');
        return this;
    }
    /**
     * Grants permission to upload theme assets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/controlling-administrator-access-with-policies-service-linked-roles.html#managed-policies-required-to-access-appstream-resources
     */
    toGetParametersForThemeAssetUpload() {
        this.to('appstream:GetParametersForThemeAssetUpload');
        return this;
    }
    /**
     * Grants permission to retrieve the name of the fleet that is associated with the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedFleets.html
     */
    toListAssociatedFleets() {
        this.to('appstream:ListAssociatedFleets');
        return this;
    }
    /**
     * Grants permission to retrieve the name of the stack with which the specified fleet is associated
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListAssociatedStacks.html
     */
    toListAssociatedStacks() {
        this.to('appstream:ListAssociatedStacks');
        return this;
    }
    /**
     * Grants permission to retrieve a list of all tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('appstream:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to start the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartFleet.html
     */
    toStartFleet() {
        this.to('appstream:StartFleet');
        return this;
    }
    /**
     * Grants permission to start the specified image builder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StartImageBuilder.html
     */
    toStartImageBuilder() {
        this.to('appstream:StartImageBuilder');
        return this;
    }
    /**
     * Grants permission to stop the specified fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopFleet.html
     */
    toStopFleet() {
        this.to('appstream:StopFleet');
        return this;
    }
    /**
     * Grants permission to stop the specified image builder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_StopImageBuilder.html
     */
    toStopImageBuilder() {
        this.to('appstream:StopImageBuilder');
        return this;
    }
    /**
     * Grants permission to federated users to sign in by using their existing credentials and stream applications from the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserId()
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     */
    toStream() {
        this.to('appstream:Stream');
        return this;
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified AppStream 2.0 resource. The following resources can be tagged: Image builders, images, fleets, and stacks
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('appstream:TagResource');
        return this;
    }
    /**
     * Grants permission to disassociate one or more tags from the specified AppStream 2.0 resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('appstream:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the specified Directory Config object in AppStream 2.0. This object includes the configuration information required to join fleets and image builders to Microsoft Active Directory domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateDirectoryConfig.html
     */
    toUpdateDirectoryConfig() {
        this.to('appstream:UpdateDirectoryConfig');
        return this;
    }
    /**
     * Grants permission to update the specified fleet. All attributes except the fleet name can be updated when the fleet is in the STOPPED state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        this.to('appstream:UpdateFleet');
        return this;
    }
    /**
     * Grants permission to add or update permissions for the specified private image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateImagePermissions.html
     */
    toUpdateImagePermissions() {
        this.to('appstream:UpdateImagePermissions');
        return this;
    }
    /**
     * Grants permission to update the specified fields for the specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appstream2/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        this.to('appstream:UpdateStack');
        return this;
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param fleetName - Identifier for the fleetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:fleet/${FleetName}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param imageName - Identifier for the imageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImage(imageName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:image/${ImageName}';
        arn = arn.replace('${ImageName}', imageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image-builder to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param imageBuilderName - Identifier for the imageBuilderName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImageBuilder(imageBuilderName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:image-builder/${ImageBuilderName}';
        arn = arn.replace('${ImageBuilderName}', imageBuilderName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/what-is-appstream.html#what-is-concepts
     *
     * @param stackName - Identifier for the stackName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStack(stackName, account, region, partition) {
        var arn = 'arn:${Partition}:appstream:${Region}:${Account}:stack/${StackName}';
        arn = arn.replace('${StackName}', stackName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the ID of the AppStream 2.0 user
     *
     * https://docs.aws.amazon.com/appstream2/latest/developerguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     *
     * Applies to actions:
     * - .toStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`appstream:userId`, value, operator || 'StringLike');
    }
}
exports.Appstream = Appstream;
//# sourceMappingURL=data:application/json;base64,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