"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudsearch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudsearch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudsearch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudsearch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudsearch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudsearch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudsearch';
        this.accessLevelList = {
            "Tagging": [
                "AddTags",
                "RemoveTags"
            ],
            "Write": [
                "BuildSuggesters",
                "CreateDomain",
                "DefineAnalysisScheme",
                "DefineExpression",
                "DefineIndexField",
                "DefineSuggester",
                "DeleteAnalysisScheme",
                "DeleteDomain",
                "DeleteExpression",
                "DeleteIndexField",
                "DeleteSuggester",
                "IndexDocuments",
                "UpdateAvailabilityOptions",
                "UpdateDomainEndpointOptions",
                "UpdateScalingParameters",
                "document"
            ],
            "Read": [
                "DescribeAnalysisSchemes",
                "DescribeAvailabilityOptions",
                "DescribeDomainEndpointOptions",
                "DescribeExpressions",
                "DescribeIndexFields",
                "DescribeScalingParameters",
                "DescribeServiceAccessPolicies",
                "DescribeSuggesters",
                "ListTags",
                "search",
                "suggest"
            ],
            "List": [
                "DescribeDomains",
                "ListDomainNames"
            ],
            "Permissions management": [
                "UpdateServiceAccessPolicies"
            ]
        };
    }
    /**
     * Attaches resource tags to an Amazon CloudSearch domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_AddTags.html
     */
    toAddTags() {
        this.to('cloudsearch:AddTags');
        return this;
    }
    /**
     * Indexes the search suggestions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_BuildSuggesters.html
     */
    toBuildSuggesters() {
        this.to('cloudsearch:BuildSuggesters');
        return this;
    }
    /**
     * Creates a new search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_CreateDomain.html
     */
    toCreateDomain() {
        this.to('cloudsearch:CreateDomain');
        return this;
    }
    /**
     * Configures an analysis scheme that can be applied to a text or text-array field to define language-specific text processing options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineAnalysisScheme.html
     */
    toDefineAnalysisScheme() {
        this.to('cloudsearch:DefineAnalysisScheme');
        return this;
    }
    /**
     * Configures an Expression for the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineExpression.html
     */
    toDefineExpression() {
        this.to('cloudsearch:DefineExpression');
        return this;
    }
    /**
     * Configures an IndexField for the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineIndexField.html
     */
    toDefineIndexField() {
        this.to('cloudsearch:DefineIndexField');
        return this;
    }
    /**
     * Configures a suggester for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineSuggester.html
     */
    toDefineSuggester() {
        this.to('cloudsearch:DefineSuggester');
        return this;
    }
    /**
     * Deletes an analysis scheme.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteAnalysisScheme.html
     */
    toDeleteAnalysisScheme() {
        this.to('cloudsearch:DeleteAnalysisScheme');
        return this;
    }
    /**
     * Permanently deletes a search domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.to('cloudsearch:DeleteDomain');
        return this;
    }
    /**
     * Removes an Expression from the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteExpression.html
     */
    toDeleteExpression() {
        this.to('cloudsearch:DeleteExpression');
        return this;
    }
    /**
     * Removes an IndexField from the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteIndexField.html
     */
    toDeleteIndexField() {
        this.to('cloudsearch:DeleteIndexField');
        return this;
    }
    /**
     * Deletes a suggester.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteSuggester.html
     */
    toDeleteSuggester() {
        this.to('cloudsearch:DeleteSuggester');
        return this;
    }
    /**
     * Gets the analysis schemes configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAnalysisSchemes.html
     */
    toDescribeAnalysisSchemes() {
        this.to('cloudsearch:DescribeAnalysisSchemes');
        return this;
    }
    /**
     * Gets the availability options configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAvailabilityOptions.html
     */
    toDescribeAvailabilityOptions() {
        this.to('cloudsearch:DescribeAvailabilityOptions');
        return this;
    }
    /**
     * Gets the domain endpoint options configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomainEndpointOptions.html
     */
    toDescribeDomainEndpointOptions() {
        this.to('cloudsearch:DescribeDomainEndpointOptions');
        return this;
    }
    /**
     * Gets information about the search domains owned by this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomains.html
     */
    toDescribeDomains() {
        this.to('cloudsearch:DescribeDomains');
        return this;
    }
    /**
     * Gets the expressions configured for the search domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeExpressions.html
     */
    toDescribeExpressions() {
        this.to('cloudsearch:DescribeExpressions');
        return this;
    }
    /**
     * Gets information about the index fields configured for the search domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeIndexFields.html
     */
    toDescribeIndexFields() {
        this.to('cloudsearch:DescribeIndexFields');
        return this;
    }
    /**
     * Gets the scaling parameters configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeScalingParameters.html
     */
    toDescribeScalingParameters() {
        this.to('cloudsearch:DescribeScalingParameters');
        return this;
    }
    /**
     * Gets information about the access policies that control access to the domain's document and search endpoints.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeServiceAccessPolicies.html
     */
    toDescribeServiceAccessPolicies() {
        this.to('cloudsearch:DescribeServiceAccessPolicies');
        return this;
    }
    /**
     * Gets the suggesters configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeSuggesters.html
     */
    toDescribeSuggesters() {
        this.to('cloudsearch:DescribeSuggesters');
        return this;
    }
    /**
     * Tells the search domain to start indexing its documents using the latest indexing options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_IndexDocuments.html
     */
    toIndexDocuments() {
        this.to('cloudsearch:IndexDocuments');
        return this;
    }
    /**
     * Lists all search domains owned by an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListDomainNames.html
     */
    toListDomainNames() {
        this.to('cloudsearch:ListDomainNames');
        return this;
    }
    /**
     * Displays all of the resource tags for an Amazon CloudSearch domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListTags.html
     */
    toListTags() {
        this.to('cloudsearch:ListTags');
        return this;
    }
    /**
     * Removes the specified resource tags from an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_RemoveTags.html
     */
    toRemoveTags() {
        this.to('cloudsearch:RemoveTags');
        return this;
    }
    /**
     * Configures the availability options for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateAvailabilityOptions.html
     */
    toUpdateAvailabilityOptions() {
        this.to('cloudsearch:UpdateAvailabilityOptions');
        return this;
    }
    /**
     * Configures the domain endpoint options for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateDomainEndpointOptions.html
     */
    toUpdateDomainEndpointOptions() {
        this.to('cloudsearch:UpdateDomainEndpointOptions');
        return this;
    }
    /**
     * Configures scaling parameters for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateScalingParameters.html
     */
    toUpdateScalingParameters() {
        this.to('cloudsearch:UpdateScalingParameters');
        return this;
    }
    /**
     * Configures the access rules that control access to the domain's document and search endpoints.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateServiceAccessPolicies.html
     */
    toUpdateServiceAccessPolicies() {
        this.to('cloudsearch:UpdateServiceAccessPolicies');
        return this;
    }
    /**
     * Allows access to the document service operations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    todocument() {
        this.to('cloudsearch:document');
        return this;
    }
    /**
     * Allows access to the search operations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    tosearch() {
        this.to('cloudsearch:search');
        return this;
    }
    /**
     * Allows access to the suggest operations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    tosuggest() {
        this.to('cloudsearch:suggest');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudsearch:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudsearch = Cloudsearch;
//# sourceMappingURL=data:application/json;base64,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