"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoSync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoSync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cognito-sync';
        this.accessLevelList = {
            "Write": [
                "BulkPublish",
                "DeleteDataset",
                "RegisterDevice",
                "SetCognitoEvents",
                "SetDatasetConfiguration",
                "SetIdentityPoolConfiguration",
                "SubscribeToDataset",
                "UnsubscribeFromDataset",
                "UpdateRecords"
            ],
            "Read": [
                "DescribeDataset",
                "DescribeIdentityPoolUsage",
                "DescribeIdentityUsage",
                "GetBulkPublishDetails",
                "GetCognitoEvents",
                "GetIdentityPoolConfiguration",
                "ListIdentityPoolUsage",
                "ListRecords",
                "QueryRecords"
            ],
            "List": [
                "ListDatasets"
            ]
        };
    }
    /**
     * Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html
     */
    toBulkPublish() {
        this.to('cognito-sync:BulkPublish');
        return this;
    }
    /**
     * Deletes the specific dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        this.to('cognito-sync:DeleteDataset');
        return this;
    }
    /**
     * Gets meta data about a dataset by identity and dataset name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        this.to('cognito-sync:DescribeDataset');
        return this;
    }
    /**
     * Gets usage details (for example, data storage) about a particular identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html
     */
    toDescribeIdentityPoolUsage() {
        this.to('cognito-sync:DescribeIdentityPoolUsage');
        return this;
    }
    /**
     * Gets usage information for an identity, including number of datasets and data usage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html
     */
    toDescribeIdentityUsage() {
        this.to('cognito-sync:DescribeIdentityUsage');
        return this;
    }
    /**
     * Get the status of the last BulkPublish operation for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html
     */
    toGetBulkPublishDetails() {
        this.to('cognito-sync:GetBulkPublishDetails');
        return this;
    }
    /**
     * Gets the events and the corresponding Lambda functions associated with an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html
     */
    toGetCognitoEvents() {
        this.to('cognito-sync:GetCognitoEvents');
        return this;
    }
    /**
     * Gets the configuration settings of an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html
     */
    toGetIdentityPoolConfiguration() {
        this.to('cognito-sync:GetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Lists datasets for an identity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        this.to('cognito-sync:ListDatasets');
        return this;
    }
    /**
     * Gets a list of identity pools registered with Cognito.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html
     */
    toListIdentityPoolUsage() {
        this.to('cognito-sync:ListIdentityPoolUsage');
        return this;
    }
    /**
     * Gets paginated records, optionally changed after a particular sync count for a dataset and identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html
     */
    toListRecords() {
        this.to('cognito-sync:ListRecords');
        return this;
    }
    /**
     * A permission that grants the ability to query records.
     *
     * Access Level: Read
     */
    toQueryRecords() {
        this.to('cognito-sync:QueryRecords');
        return this;
    }
    /**
     * Registers a device to receive push sync notifications.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html
     */
    toRegisterDevice() {
        this.to('cognito-sync:RegisterDevice');
        return this;
    }
    /**
     * Sets the AWS Lambda function for a given event type for an identity pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html
     */
    toSetCognitoEvents() {
        this.to('cognito-sync:SetCognitoEvents');
        return this;
    }
    /**
     * A permission that grants ability to configure datasets.
     *
     * Access Level: Write
     */
    toSetDatasetConfiguration() {
        this.to('cognito-sync:SetDatasetConfiguration');
        return this;
    }
    /**
     * Sets the necessary configuration for push sync.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html
     */
    toSetIdentityPoolConfiguration() {
        this.to('cognito-sync:SetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Subscribes to receive notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html
     */
    toSubscribeToDataset() {
        this.to('cognito-sync:SubscribeToDataset');
        return this;
    }
    /**
     * Unsubscribes from receiving notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html
     */
    toUnsubscribeFromDataset() {
        this.to('cognito-sync:UnsubscribeFromDataset');
        return this;
    }
    /**
     * Posts updates to records and adds and deletes records for a dataset and user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html
     */
    toUpdateRecords() {
        this.to('cognito-sync:UpdateRecords');
        return this;
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDataset(identityPoolId, identityId, datasetName, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${DatasetName}', datasetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityPoolId, identityId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoSync = CognitoSync;
//# sourceMappingURL=data:application/json;base64,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