"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Es = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [es](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Es extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [es](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticsearchservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'es';
        this.accessLevelList = {
            "Write": [
                "AcceptInboundCrossClusterSearchConnection",
                "CreateElasticsearchDomain",
                "CreateElasticsearchServiceRole",
                "CreateOutboundCrossClusterSearchConnection",
                "DeleteElasticsearchDomain",
                "DeleteElasticsearchServiceRole",
                "DeleteInboundCrossClusterSearchConnection",
                "DeleteOutboundCrossClusterSearchConnection",
                "ESHttpDelete",
                "ESHttpPatch",
                "ESHttpPost",
                "ESHttpPut",
                "PurchaseReservedElasticsearchInstanceOffering",
                "RejectInboundCrossClusterSearchConnection",
                "UpdateElasticsearchDomainConfig",
                "UpgradeElasticsearchDomain"
            ],
            "Tagging": [
                "AddTags",
                "RemoveTags"
            ],
            "Read": [
                "DescribeElasticsearchDomain",
                "DescribeElasticsearchDomainConfig",
                "ESCrossClusterGet",
                "ESHttpGet",
                "ESHttpHead",
                "GetUpgradeHistory",
                "GetUpgradeStatus",
                "ListTags"
            ],
            "List": [
                "DescribeElasticsearchDomains",
                "DescribeElasticsearchInstanceTypeLimits",
                "DescribeInboundCrossClusterSearchConnections",
                "DescribeOutboundCrossClusterSearchConnections",
                "DescribeReservedElasticsearchInstanceOfferings",
                "DescribeReservedElasticsearchInstances",
                "GetCompatibleElasticsearchVersions",
                "ListDomainNames",
                "ListElasticsearchInstanceTypeDetails",
                "ListElasticsearchInstanceTypes",
                "ListElasticsearchVersions"
            ]
        };
    }
    /**
     * Allows the destination domain owner to accept an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-acceptinboundcrossclustersearchconnection
     */
    toAcceptInboundCrossClusterSearchConnection() {
        this.to('es:AcceptInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to attach resource tags to an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-addtags
     */
    toAddTags() {
        this.to('es:AddTags');
        return this;
    }
    /**
     * Grants permission to create an Amazon ES domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchdomain
     */
    toCreateElasticsearchDomain() {
        this.to('es:CreateElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to create the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createelasticsearchservicerole
     */
    toCreateElasticsearchServiceRole() {
        this.to('es:CreateElasticsearchServiceRole');
        return this;
    }
    /**
     * Creates a new cross-cluster search connection from a source domain to a destination domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-createoutboundcrossclustersearchconnection
     */
    toCreateOutboundCrossClusterSearchConnection() {
        this.to('es:CreateOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to delete an Amazon ES domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchdomain
     */
    toDeleteElasticsearchDomain() {
        this.to('es:DeleteElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to delete the service-linked role required for Amazon ES domains that use VPC access.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteelasticsearchservicerole
     */
    toDeleteElasticsearchServiceRole() {
        this.to('es:DeleteElasticsearchServiceRole');
        return this;
    }
    /**
     * Allows the destination domain owner to delete an existing inbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteinboundcrossclustersearchconnection
     */
    toDeleteInboundCrossClusterSearchConnection() {
        this.to('es:DeleteInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Allows the source domain owner to delete an existing outbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-deleteoutboundcrossclustersearchconnection
     */
    toDeleteOutboundCrossClusterSearchConnection() {
        this.to('es:DeleteOutboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for the specified Amazon ES domain, including the domain ID, domain service endpoint, and domain ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    toDescribeElasticsearchDomain() {
        this.to('es:DescribeElasticsearchDomain');
        return this;
    }
    /**
     * Grants permission to view a description of the configuration options and status of an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomainconfig
     */
    toDescribeElasticsearchDomainConfig() {
        this.to('es:DescribeElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to view a description of the domain configuration for up to five specified Amazon ES domains.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeelasticsearchdomain
     */
    toDescribeElasticsearchDomains() {
        this.to('es:DescribeElasticsearchDomains');
        return this;
    }
    /**
     * Grants permission to view the instance count, storage, and master node limits for a given Elasticsearch version and instance type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinstancetypelimits
     */
    toDescribeElasticsearchInstanceTypeLimits() {
        this.to('es:DescribeElasticsearchInstanceTypeLimits');
        return this;
    }
    /**
     * Lists all the inbound cross-cluster search connections for a destination domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeinboundcrossclustersearchconnections
     */
    toDescribeInboundCrossClusterSearchConnections() {
        this.to('es:DescribeInboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Lists all the outbound cross-cluster search connections for a source domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describeoutboundcrossclustersearchconnections
     */
    toDescribeOutboundCrossClusterSearchConnections() {
        this.to('es:DescribeOutboundCrossClusterSearchConnections');
        return this;
    }
    /**
     * Grants permission to fetch reserved instance offerings for ES
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstanceofferings
     */
    toDescribeReservedElasticsearchInstanceOfferings() {
        this.to('es:DescribeReservedElasticsearchInstanceOfferings');
        return this;
    }
    /**
     * Grants permission to fetch ES reserved instances already purchased by customer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-describereservedelasticsearchinstances
     */
    toDescribeReservedElasticsearchInstances() {
        this.to('es:DescribeReservedElasticsearchInstances');
        return this;
    }
    /**
     * Grants permission to send cross-cluster requests to a destination domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESCrossClusterGet() {
        this.to('es:ESCrossClusterGet');
        return this;
    }
    /**
     * Grants permission to send HTTP DELETE requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpDelete() {
        this.to('es:ESHttpDelete');
        return this;
    }
    /**
     * Grants permission to send HTTP GET requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpGet() {
        this.to('es:ESHttpGet');
        return this;
    }
    /**
     * Grants permission to send HTTP HEAD requests to the Elasticsearch APIs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpHead() {
        this.to('es:ESHttpHead');
        return this;
    }
    /**
     * Grants permission to send HTTP PATCH requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpPatch() {
        this.to('es:ESHttpPatch');
        return this;
    }
    /**
     * Grants permission to send HTTP POST requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpPost() {
        this.to('es:ESHttpPost');
        return this;
    }
    /**
     * Grants permission to send HTTP PUT requests to the Elasticsearch APIs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html#es-ac-reference
     */
    toESHttpPut() {
        this.to('es:ESHttpPut');
        return this;
    }
    /**
     * Grants permission to fetch list of compatible elastic search versions to which Amazon ES domain can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getcompatibleelasticsearchversions
     */
    toGetCompatibleElasticsearchVersions() {
        this.to('es:GetCompatibleElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to fetch upgrade history for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradehistory
     */
    toGetUpgradeHistory() {
        this.to('es:GetUpgradeHistory');
        return this;
    }
    /**
     * Grants permission to fetch upgrade status for given ES domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-getupgradestatus
     */
    toGetUpgradeStatus() {
        this.to('es:GetUpgradeStatus');
        return this;
    }
    /**
     * Grants permission to display the names of all Amazon ES domains that the current user owns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listdomainnames
     */
    toListDomainNames() {
        this.to('es:ListDomainNames');
        return this;
    }
    /**
     * Grants permission to list all instance types and available features for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypedetails
     */
    toListElasticsearchInstanceTypeDetails() {
        this.to('es:ListElasticsearchInstanceTypeDetails');
        return this;
    }
    /**
     * Grants permission to list all Elasticsearch instance types that are supported for a given Elasticsearch version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchinstancetypes
     */
    toListElasticsearchInstanceTypes() {
        this.to('es:ListElasticsearchInstanceTypes');
        return this;
    }
    /**
     * Grants permission to list all supported Elasticsearch versions on Amazon ES.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listelasticsearchversions
     */
    toListElasticsearchVersions() {
        this.to('es:ListElasticsearchVersions');
        return this;
    }
    /**
     * Grants permission to display all of the tags for an Amazon ES domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    toListTags() {
        this.to('es:ListTags');
        return this;
    }
    /**
     * Grants permission to purchase ES reserved instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-purchasereservedelasticsearchinstance
     */
    toPurchaseReservedElasticsearchInstanceOffering() {
        this.to('es:PurchaseReservedElasticsearchInstanceOffering');
        return this;
    }
    /**
     * Allows the destination domain owner to reject an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-rejectinboundcrossclustersearchconnection
     */
    toRejectInboundCrossClusterSearchConnection() {
        this.to('es:RejectInboundCrossClusterSearchConnection');
        return this;
    }
    /**
     * Grants permission to remove tags from Amazon ES domains.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-listtags
     */
    toRemoveTags() {
        this.to('es:RemoveTags');
        return this;
    }
    /**
     * Grants permission to modify the configuration of an Amazon ES domain, such as the instance type or number of instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-updateelasticsearchdomainconfig
     */
    toUpdateElasticsearchDomainConfig() {
        this.to('es:UpdateElasticsearchDomainConfig');
        return this;
    }
    /**
     * Grants permission to initiate upgrade of elastic search domain to given version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-configuration-api.html#es-configuration-api-actions-upgradeelasticsearchdomain
     */
    toUpgradeElasticsearchDomain() {
        this.to('es:UpgradeElasticsearchDomain');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-ac.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:es:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Es = Es;
//# sourceMappingURL=data:application/json;base64,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