"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schemas = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [schemas](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridgeschemas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Schemas extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [schemas](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridgeschemas.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'schemas';
        this.accessLevelList = {
            "Write": [
                "CreateDiscoverer",
                "CreateRegistry",
                "CreateSchema",
                "DeleteDiscoverer",
                "DeleteRegistry",
                "DeleteResourcePolicy",
                "DeleteSchema",
                "DeleteSchemaVersion",
                "PutCodeBinding",
                "PutResourcePolicy",
                "StartDiscoverer",
                "StopDiscoverer",
                "UpdateDiscoverer",
                "UpdateRegistry",
                "UpdateSchema"
            ],
            "Read": [
                "DescribeCodeBinding",
                "DescribeDiscoverer",
                "DescribeRegistry",
                "DescribeSchema",
                "GetCodeBindingSource",
                "GetDiscoveredSchema",
                "GetResourcePolicy"
            ],
            "List": [
                "ListDiscoverers",
                "ListRegistries",
                "ListSchemaVersions",
                "ListSchemas",
                "ListTagsForResource",
                "SearchSchemas"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates an event schema discoverer. Once created, your events will be automatically map into corresponding schema documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#CreateDiscoverer
     */
    toCreateDiscoverer() {
        this.to('schemas:CreateDiscoverer');
        return this;
    }
    /**
     * Create a new schema registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#CreateRegistry
     */
    toCreateRegistry() {
        this.to('schemas:CreateRegistry');
        return this;
    }
    /**
     * Create a new schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#CreateSchema
     */
    toCreateSchema() {
        this.to('schemas:CreateSchema');
        return this;
    }
    /**
     * Deletes discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DeleteDiscoverer
     */
    toDeleteDiscoverer() {
        this.to('schemas:DeleteDiscoverer');
        return this;
    }
    /**
     * Deletes an existing registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DeleteRegistry
     */
    toDeleteRegistry() {
        this.to('schemas:DeleteRegistry');
        return this;
    }
    /**
     * Delete the resource-based policy attached to a given registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#DeleteResourcePolicy
     */
    toDeleteResourcePolicy() {
        this.to('schemas:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DeleteSchema
     */
    toDeleteSchema() {
        this.to('schemas:DeleteSchema');
        return this;
    }
    /**
     * Deletes a specific version of schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-version-schemaversion.html#DeleteSchemaVersion
     */
    toDeleteSchemaVersion() {
        this.to('schemas:DeleteSchemaVersion');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#DescribeCodeBinding
     */
    toDescribeCodeBinding() {
        this.to('schemas:DescribeCodeBinding');
        return this;
    }
    /**
     * Retrieves discoverer metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DescribeDiscoverer
     */
    toDescribeDiscoverer() {
        this.to('schemas:DescribeDiscoverer');
        return this;
    }
    /**
     * Describes an existing registry metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DescribeRegistry
     */
    toDescribeRegistry() {
        this.to('schemas:DescribeRegistry');
        return this;
    }
    /**
     * Retrieves an existing schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DescribeSchema
     */
    toDescribeSchema() {
        this.to('schemas:DescribeSchema');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language-source.html#GetCodeBindingSource
     */
    toGetCodeBindingSource() {
        this.to('schemas:GetCodeBindingSource');
        return this;
    }
    /**
     * Retrieves schema for the provided list of sample events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discover.html#GetDiscoveredSchema
     */
    toGetDiscoveredSchema() {
        this.to('schemas:GetDiscoveredSchema');
        return this;
    }
    /**
     * Retrieves the resource-based policy attached to a given registry.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#GetResourcePolicy
     */
    toGetResourcePolicy() {
        this.to('schemas:GetResourcePolicy');
        return this;
    }
    /**
     * Lists all the discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers.html#ListDiscoverers
     */
    toListDiscoverers() {
        this.to('schemas:ListDiscoverers');
        return this;
    }
    /**
     * List all discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries.html#ListRegistries
     */
    toListRegistries() {
        this.to('schemas:ListRegistries');
        return this;
    }
    /**
     * List all versions of a schema.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-versions.html#ListSchemaVersions
     */
    toListSchemaVersions() {
        this.to('schemas:ListSchemaVersions');
        return this;
    }
    /**
     * List all schemas.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas.html#ListSchemas
     */
    toListSchemas() {
        this.to('schemas:ListSchemas');
        return this;
    }
    /**
     * This action lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#ListTagsForResource
     */
    toListTagsForResource() {
        this.to('schemas:ListTagsForResource');
        return this;
    }
    /**
     * Generates code for specific schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#PutCodeBinding
     */
    toPutCodeBinding() {
        this.to('schemas:PutCodeBinding');
        return this;
    }
    /**
     * Attach resource-based policy to the specific registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#PutResourcePolicy
     */
    toPutResourcePolicy() {
        this.to('schemas:PutResourcePolicy');
        return this;
    }
    /**
     * Searches schemas based on specified keywords in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-search.html#SearchSchemas
     */
    toSearchSchemas() {
        this.to('schemas:SearchSchemas');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StartDiscoverer
     */
    toStartDiscoverer() {
        this.to('schemas:StartDiscoverer');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StopDiscoverer
     */
    toStopDiscoverer() {
        this.to('schemas:StopDiscoverer');
        return this;
    }
    /**
     * This action tags an resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#TagResource
     */
    toTagResource() {
        this.to('schemas:TagResource');
        return this;
    }
    /**
     * This action removes a tag from on a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#UntagResource
     */
    toUntagResource() {
        this.to('schemas:UntagResource');
        return this;
    }
    /**
     * Updates an existing discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#UpdateDiscoverer
     */
    toUpdateDiscoverer() {
        this.to('schemas:UpdateDiscoverer');
        return this;
    }
    /**
     * Updates an existing registry metadata in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#UpdateRegistry
     */
    toUpdateRegistry() {
        this.to('schemas:UpdateRegistry');
        return this;
    }
    /**
     * Updates an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#UpdateSchema
     */
    toUpdateSchema() {
        this.to('schemas:UpdateSchema');
        return this;
    }
    /**
     * Adds a resource of type discoverer to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param discovererId - Identifier for the discovererId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDiscoverer(discovererId, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:discoverer/${DiscovererId}';
        arn = arn.replace('${DiscovererId}', discovererId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type registry to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegistry(registryName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:registry/${RegistryName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type schema to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSchema(registryName, schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:schema/${RegistryName}/${SchemaName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Schemas = Schemas;
//# sourceMappingURL=data:application/json;base64,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