"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Freertos = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [freertos](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Freertos extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [freertos](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfreertos.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'freertos';
        this.accessLevelList = {
            "Write": [
                "CreateSoftwareConfiguration",
                "DeleteSoftwareConfiguration",
                "UpdateSoftwareConfiguration"
            ],
            "Read": [
                "DescribeHardwarePlatform",
                "DescribeSoftwareConfiguration",
                "GetSoftwareURL",
                "GetSoftwareURLForConfiguration"
            ],
            "List": [
                "ListFreeRTOSVersions",
                "ListHardwarePlatforms",
                "ListHardwareVendors",
                "ListSoftwareConfigurations"
            ]
        };
    }
    /**
     * Creates a software configuration.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toCreateSoftwareConfiguration() {
        this.to('freertos:CreateSoftwareConfiguration');
        return this;
    }
    /**
     * Deletes the software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDeleteSoftwareConfiguration() {
        this.to('freertos:DeleteSoftwareConfiguration');
        return this;
    }
    /**
     * Describes the hardware platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeHardwarePlatform() {
        this.to('freertos:DescribeHardwarePlatform');
        return this;
    }
    /**
     * Describes the software configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeSoftwareConfiguration() {
        this.to('freertos:DescribeSoftwareConfiguration');
        return this;
    }
    /**
     * Get the URL for Amazon FreeRTOS software download.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURL() {
        this.to('freertos:GetSoftwareURL');
        return this;
    }
    /**
     * Get the URL for Amazon FreeRTOS software download based on the configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURLForConfiguration() {
        this.to('freertos:GetSoftwareURLForConfiguration');
        return this;
    }
    /**
     * Lists versions of AmazonFreeRTOS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListFreeRTOSVersions() {
        this.to('freertos:ListFreeRTOSVersions');
        return this;
    }
    /**
     * Lists the hardware platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwarePlatforms() {
        this.to('freertos:ListHardwarePlatforms');
        return this;
    }
    /**
     * Lists the hardware vendors.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwareVendors() {
        this.to('freertos:ListHardwareVendors');
        return this;
    }
    /**
     * Lists the software configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListSoftwareConfigurations() {
        this.to('freertos:ListSoftwareConfigurations');
        return this;
    }
    /**
     * Updates the software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toUpdateSoftwareConfiguration() {
        this.to('freertos:UpdateSoftwareConfiguration');
        return this;
    }
    /**
     * Adds a resource of type configuration to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     *
     * @param configurationName - Identifier for the configurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguration(configurationName, account, region, partition) {
        var arn = 'arn:${Partition}:freertos:${Region}:${Account}:configuration/${ConfigurationName}';
        arn = arn.replace('${ConfigurationName}', configurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Freertos = Freertos;
//# sourceMappingURL=data:application/json;base64,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