"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gamelift = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [gamelift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazongamelift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Gamelift extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [gamelift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazongamelift.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'gamelift';
        this.accessLevelList = {
            "Write": [
                "AcceptMatch",
                "ClaimGameServer",
                "CreateAlias",
                "CreateBuild",
                "CreateFleet",
                "CreateGameServerGroup",
                "CreateGameSession",
                "CreateGameSessionQueue",
                "CreateMatchmakingConfiguration",
                "CreateMatchmakingRuleSet",
                "CreatePlayerSession",
                "CreatePlayerSessions",
                "CreateScript",
                "CreateVpcPeeringAuthorization",
                "CreateVpcPeeringConnection",
                "DeleteAlias",
                "DeleteBuild",
                "DeleteFleet",
                "DeleteGameServerGroup",
                "DeleteGameSessionQueue",
                "DeleteMatchmakingConfiguration",
                "DeleteMatchmakingRuleSet",
                "DeleteScalingPolicy",
                "DeleteScript",
                "DeleteVpcPeeringAuthorization",
                "DeleteVpcPeeringConnection",
                "DeregisterGameServer",
                "PutScalingPolicy",
                "RegisterGameServer",
                "ResumeGameServerGroup",
                "StartFleetActions",
                "StartGameSessionPlacement",
                "StartMatchBackfill",
                "StartMatchmaking",
                "StopFleetActions",
                "StopGameSessionPlacement",
                "StopMatchmaking",
                "SuspendGameServerGroup",
                "UpdateAlias",
                "UpdateBuild",
                "UpdateFleetAttributes",
                "UpdateFleetCapacity",
                "UpdateFleetPortSettings",
                "UpdateGameServer",
                "UpdateGameServerGroup",
                "UpdateGameSession",
                "UpdateGameSessionQueue",
                "UpdateMatchmakingConfiguration",
                "UpdateRuntimeConfiguration",
                "UpdateScript"
            ],
            "Read": [
                "DescribeAlias",
                "DescribeBuild",
                "DescribeEC2InstanceLimits",
                "DescribeFleetAttributes",
                "DescribeFleetCapacity",
                "DescribeFleetEvents",
                "DescribeFleetPortSettings",
                "DescribeFleetUtilization",
                "DescribeGameServer",
                "DescribeGameServerGroup",
                "DescribeGameServerInstances",
                "DescribeGameSessionDetails",
                "DescribeGameSessionPlacement",
                "DescribeGameSessionQueues",
                "DescribeGameSessions",
                "DescribeInstances",
                "DescribeMatchmaking",
                "DescribeMatchmakingConfigurations",
                "DescribeMatchmakingRuleSets",
                "DescribePlayerSessions",
                "DescribeRuntimeConfiguration",
                "DescribeScalingPolicies",
                "DescribeScript",
                "DescribeVpcPeeringAuthorizations",
                "DescribeVpcPeeringConnections",
                "GetGameSessionLogUrl",
                "GetInstanceAccess",
                "RequestUploadCredentials",
                "ResolveAlias",
                "SearchGameSessions",
                "ValidateMatchmakingRuleSet"
            ],
            "List": [
                "ListAliases",
                "ListBuilds",
                "ListFleets",
                "ListGameServerGroups",
                "ListGameServers",
                "ListScripts",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Registers player acceptance or rejection of a proposed FlexMatch match.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_AcceptMatch.html
     */
    toAcceptMatch() {
        this.to('gamelift:AcceptMatch');
        return this;
    }
    /**
     * Locates and reserves a game server to host a new game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ClaimGameServer.html
     */
    toClaimGameServer() {
        this.to('gamelift:ClaimGameServer');
        return this;
    }
    /**
     * Defines a new alias for a fleet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateAlias.html
     */
    toCreateAlias() {
        this.to('gamelift:CreateAlias');
        return this;
    }
    /**
     * Creates a new game build using files stored in an Amazon S3 bucket.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html
     */
    toCreateBuild() {
        this.to('gamelift:CreateBuild');
        return this;
    }
    /**
     * Creates a new fleet of computing resources to run your game servers.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateFleet.html
     */
    toCreateFleet() {
        this.to('gamelift:CreateFleet');
        return this;
    }
    /**
     * Creates a new game server group, sets up a corresponding Auto Scaling group, and launches instances to host game servers.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameServerGroup.html
     */
    toCreateGameServerGroup() {
        this.to('gamelift:CreateGameServerGroup');
        return this;
    }
    /**
     * Starts a new game session on a specified fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSession.html
     */
    toCreateGameSession() {
        this.to('gamelift:CreateGameSession');
        return this;
    }
    /**
     * Sets up a new queue for processing new game session placement requests.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateGameSessionQueue.html
     */
    toCreateGameSessionQueue() {
        this.to('gamelift:CreateGameSessionQueue');
        return this;
    }
    /**
     * Creates a new FlexMatch matchmaker.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingConfiguration.html
     */
    toCreateMatchmakingConfiguration() {
        this.to('gamelift:CreateMatchmakingConfiguration');
        return this;
    }
    /**
     * Creates a new matchmaking rule set for FlexMatch.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateMatchmakingRuleSet.html
     */
    toCreateMatchmakingRuleSet() {
        this.to('gamelift:CreateMatchmakingRuleSet');
        return this;
    }
    /**
     * Reserves an available game session slot for a player.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSession.html
     */
    toCreatePlayerSession() {
        this.to('gamelift:CreatePlayerSession');
        return this;
    }
    /**
     * Reserves available game session slots for multiple players.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreatePlayerSessions.html
     */
    toCreatePlayerSessions() {
        this.to('gamelift:CreatePlayerSessions');
        return this;
    }
    /**
     * Creates a new Realtime Servers script.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateScript.html
     */
    toCreateScript() {
        this.to('gamelift:CreateScript');
        return this;
    }
    /**
     * Allows GameLift to create or delete a peering connection between a GameLift fleet VPC and a VPC on another AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringAuthorization.html
     */
    toCreateVpcPeeringAuthorization() {
        this.to('gamelift:CreateVpcPeeringAuthorization');
        return this;
    }
    /**
     * Establishes a peering connection between your GameLift fleet VPC and a VPC on another account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateVpcPeeringConnection.html
     */
    toCreateVpcPeeringConnection() {
        this.to('gamelift:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Deletes an alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        this.to('gamelift:DeleteAlias');
        return this;
    }
    /**
     * Deletes a game build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteBuild.html
     */
    toDeleteBuild() {
        this.to('gamelift:DeleteBuild');
        return this;
    }
    /**
     * Deletes an empty fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        this.to('gamelift:DeleteFleet');
        return this;
    }
    /**
     * Permanently deletes a game server group and terminates FleetIQ activity for the corresponding Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteGameServerGroup.html
     */
    toDeleteGameServerGroup() {
        this.to('gamelift:DeleteGameServerGroup');
        return this;
    }
    /**
     * Deletes an existing game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteGameSessionQueue.html
     */
    toDeleteGameSessionQueue() {
        this.to('gamelift:DeleteGameSessionQueue');
        return this;
    }
    /**
     * Deletes an existing FlexMatch matchmaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingConfiguration.html
     */
    toDeleteMatchmakingConfiguration() {
        this.to('gamelift:DeleteMatchmakingConfiguration');
        return this;
    }
    /**
     * Deletes an existing FlexMatch matchmaking rule set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteMatchmakingRuleSet.html
     */
    toDeleteMatchmakingRuleSet() {
        this.to('gamelift:DeleteMatchmakingRuleSet');
        return this;
    }
    /**
     * Deletes a set of auto-scaling rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScalingPolicy.html
     */
    toDeleteScalingPolicy() {
        this.to('gamelift:DeleteScalingPolicy');
        return this;
    }
    /**
     * Deletes a Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteScript.html
     */
    toDeleteScript() {
        this.to('gamelift:DeleteScript');
        return this;
    }
    /**
     * Cancels a VPC peering authorization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringAuthorization.html
     */
    toDeleteVpcPeeringAuthorization() {
        this.to('gamelift:DeleteVpcPeeringAuthorization');
        return this;
    }
    /**
     * Removes a peering connection between VPCs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeleteVpcPeeringConnection.html
     */
    toDeleteVpcPeeringConnection() {
        this.to('gamelift:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Removes a game server from a game server group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DeregisterGameServer.html
     */
    toDeregisterGameServer() {
        this.to('gamelift:DeregisterGameServer');
        return this;
    }
    /**
     * Retrieves properties for an alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeAlias.html
     */
    toDescribeAlias() {
        this.to('gamelift:DescribeAlias');
        return this;
    }
    /**
     * Retrieves properties for a game build.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeBuild.html
     */
    toDescribeBuild() {
        this.to('gamelift:DescribeBuild');
        return this;
    }
    /**
     * Retrieves the maximum allowed and current usage for EC2 instance types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeEC2InstanceLimits.html
     */
    toDescribeEC2InstanceLimits() {
        this.to('gamelift:DescribeEC2InstanceLimits');
        return this;
    }
    /**
     * Retrieves general properties, including status, for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes.html
     */
    toDescribeFleetAttributes() {
        this.to('gamelift:DescribeFleetAttributes');
        return this;
    }
    /**
     * Retrieves the current capacity setting for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetCapacity.html
     */
    toDescribeFleetCapacity() {
        this.to('gamelift:DescribeFleetCapacity');
        return this;
    }
    /**
     * Retrieves entries from a fleet's event log.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetEvents.html
     */
    toDescribeFleetEvents() {
        this.to('gamelift:DescribeFleetEvents');
        return this;
    }
    /**
     * Retrieves the inbound connection permissions for a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetPortSettings.html
     */
    toDescribeFleetPortSettings() {
        this.to('gamelift:DescribeFleetPortSettings');
        return this;
    }
    /**
     * Retrieves utilization statistics for fleets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetUtilization.html
     */
    toDescribeFleetUtilization() {
        this.to('gamelift:DescribeFleetUtilization');
        return this;
    }
    /**
     * Retrieves properties for a game server.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameServer.html
     */
    toDescribeGameServer() {
        this.to('gamelift:DescribeGameServer');
        return this;
    }
    /**
     * Retrieves properties for a game server group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameServerGroup.html
     */
    toDescribeGameServerGroup() {
        this.to('gamelift:DescribeGameServerGroup');
        return this;
    }
    /**
     * Retrieves the status of EC2 instances in a game server group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameServerInstances.html
     */
    toDescribeGameServerInstances() {
        this.to('gamelift:DescribeGameServerInstances');
        return this;
    }
    /**
     * Retrieves properties for game sessions in a fleet, including the protection policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionDetails.html
     */
    toDescribeGameSessionDetails() {
        this.to('gamelift:DescribeGameSessionDetails');
        return this;
    }
    /**
     * Retrieves details of a game session placement request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionPlacement.html
     */
    toDescribeGameSessionPlacement() {
        this.to('gamelift:DescribeGameSessionPlacement');
        return this;
    }
    /**
     * Retrieves properties for game session queues.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessionQueues.html
     */
    toDescribeGameSessionQueues() {
        this.to('gamelift:DescribeGameSessionQueues');
        return this;
    }
    /**
     * Retrieves properties for game sessions in a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeGameSessions.html
     */
    toDescribeGameSessions() {
        this.to('gamelift:DescribeGameSessions');
        return this;
    }
    /**
     * Retrieves information about instances in a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeInstances.html
     */
    toDescribeInstances() {
        this.to('gamelift:DescribeInstances');
        return this;
    }
    /**
     * Retrieves details of matchmaking tickets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html
     */
    toDescribeMatchmaking() {
        this.to('gamelift:DescribeMatchmaking');
        return this;
    }
    /**
     * Retrieves properties for FlexMatch matchmakers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingConfigurations.html
     */
    toDescribeMatchmakingConfigurations() {
        this.to('gamelift:DescribeMatchmakingConfigurations');
        return this;
    }
    /**
     * Retrieves properties for FlexMatch matchmaking rule sets.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmakingRuleSets.html
     */
    toDescribeMatchmakingRuleSets() {
        this.to('gamelift:DescribeMatchmakingRuleSets');
        return this;
    }
    /**
     * Retrieves properties for player sessions in a game session.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribePlayerSessions.html
     */
    toDescribePlayerSessions() {
        this.to('gamelift:DescribePlayerSessions');
        return this;
    }
    /**
     * Retrieves the current runtime configuration for a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeRuntimeConfiguration.html
     */
    toDescribeRuntimeConfiguration() {
        this.to('gamelift:DescribeRuntimeConfiguration');
        return this;
    }
    /**
     * Retrieves all scaling policies that are applied to a fleet.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScalingPolicies.html
     */
    toDescribeScalingPolicies() {
        this.to('gamelift:DescribeScalingPolicies');
        return this;
    }
    /**
     * Retrieves properties for a Realtime Servers script.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeScript.html
     */
    toDescribeScript() {
        this.to('gamelift:DescribeScript');
        return this;
    }
    /**
     * Retrieves valid VPC peering authorizations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringAuthorizations.html
     */
    toDescribeVpcPeeringAuthorizations() {
        this.to('gamelift:DescribeVpcPeeringAuthorizations');
        return this;
    }
    /**
     * Retrieves details on active or pending VPC peering connections.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeVpcPeeringConnections.html
     */
    toDescribeVpcPeeringConnections() {
        this.to('gamelift:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Retrieves the location of stored logs for a game session.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetGameSessionLogUrl.html
     */
    toGetGameSessionLogUrl() {
        this.to('gamelift:GetGameSessionLogUrl');
        return this;
    }
    /**
     * Requests remote access to a specified fleet instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess.html
     */
    toGetInstanceAccess() {
        this.to('gamelift:GetInstanceAccess');
        return this;
    }
    /**
     * Retrieves all aliases that are defined in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListAliases.html
     */
    toListAliases() {
        this.to('gamelift:ListAliases');
        return this;
    }
    /**
     * Retrieves all game build in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListBuilds.html
     */
    toListBuilds() {
        this.to('gamelift:ListBuilds');
        return this;
    }
    /**
     * Retrieves a list of fleet IDs for all fleets in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListFleets.html
     */
    toListFleets() {
        this.to('gamelift:ListFleets');
        return this;
    }
    /**
     * Retrieves all game server groups that are defined in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListGameServerGroups.html
     */
    toListGameServerGroups() {
        this.to('gamelift:ListGameServerGroups');
        return this;
    }
    /**
     * Retrieves all game servers that are currently running in a game server group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListGameServers.html
     */
    toListGameServers() {
        this.to('gamelift:ListGameServers');
        return this;
    }
    /**
     * Retrieves properties for all Realtime Servers scripts in the current region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListScripts.html
     */
    toListScripts() {
        this.to('gamelift:ListScripts');
        return this;
    }
    /**
     * Lists tags for GameLift resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('gamelift:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a fleet auto-scaling policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_PutScalingPolicy.html
     */
    toPutScalingPolicy() {
        this.to('gamelift:PutScalingPolicy');
        return this;
    }
    /**
     * Notifies GameLift FleetIQ when a new game server is ready to host gameplay.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_RegisterGameServer.html
     */
    toRegisterGameServer() {
        this.to('gamelift:RegisterGameServer');
        return this;
    }
    /**
     * Retrieves fresh upload credentials to use when uploading a new game build.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_RequestUploadCredentials.html
     */
    toRequestUploadCredentials() {
        this.to('gamelift:RequestUploadCredentials');
        return this;
    }
    /**
     * Retrieves the fleet ID associated with an alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ResolveAlias.html
     */
    toResolveAlias() {
        this.to('gamelift:ResolveAlias');
        return this;
    }
    /**
     * Reinstates suspended FleetIQ activity for a game server group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ResumeGameServerGroup.html
     */
    toResumeGameServerGroup() {
        this.to('gamelift:ResumeGameServerGroup');
        return this;
    }
    /**
     * Retrieves game sessions that match a set of search criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_SearchGameSessions.html
     */
    toSearchGameSessions() {
        this.to('gamelift:SearchGameSessions');
        return this;
    }
    /**
     * Resumes auto-scaling activity on a fleet after it was suspended with StopFleetActions().
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartFleetActions.html
     */
    toStartFleetActions() {
        this.to('gamelift:StartFleetActions');
        return this;
    }
    /**
     * Sends a game session placement request to a game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartGameSessionPlacement.html
     */
    toStartGameSessionPlacement() {
        this.to('gamelift:StartGameSessionPlacement');
        return this;
    }
    /**
     * Requests FlexMatch matchmaking to fill available player slots in an existing game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html
     */
    toStartMatchBackfill() {
        this.to('gamelift:StartMatchBackfill');
        return this;
    }
    /**
     * Requests FlexMatch matchmaking for one or a group of players and game session placement for a resulting match.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchmaking.html
     */
    toStartMatchmaking() {
        this.to('gamelift:StartMatchmaking');
        return this;
    }
    /**
     * Suspends auto-scaling activity on a fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html
     */
    toStopFleetActions() {
        this.to('gamelift:StopFleetActions');
        return this;
    }
    /**
     * Cancels a game session placement request that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopGameSessionPlacement.html
     */
    toStopGameSessionPlacement() {
        this.to('gamelift:StopGameSessionPlacement');
        return this;
    }
    /**
     * Cancels a matchmaking or match backfill request that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopMatchmaking.html
     */
    toStopMatchmaking() {
        this.to('gamelift:StopMatchmaking');
        return this;
    }
    /**
     * Temporarily stops FleetIQ activity for a game server group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_SuspendGameServerGroup.html
     */
    toSuspendGameServerGroup() {
        this.to('gamelift:SuspendGameServerGroup');
        return this;
    }
    /**
     * Tags GameLift resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        this.to('gamelift:TagResource');
        return this;
    }
    /**
     * Untags GameLift resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('gamelift:UntagResource');
        return this;
    }
    /**
     * Updates the properties of an existing alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateAlias.html
     */
    toUpdateAlias() {
        this.to('gamelift:UpdateAlias');
        return this;
    }
    /**
     * Updates an existing build's metadata.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateBuild.html
     */
    toUpdateBuild() {
        this.to('gamelift:UpdateBuild');
        return this;
    }
    /**
     * Updates the general properties of an existing fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetAttributes.html
     */
    toUpdateFleetAttributes() {
        this.to('gamelift:UpdateFleetAttributes');
        return this;
    }
    /**
     * Adjusts a fleet's capacity settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetCapacity.html
     */
    toUpdateFleetCapacity() {
        this.to('gamelift:UpdateFleetCapacity');
        return this;
    }
    /**
     * Adjusts a fleet's port settings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateFleetPortSettings.html
     */
    toUpdateFleetPortSettings() {
        this.to('gamelift:UpdateFleetPortSettings');
        return this;
    }
    /**
     * Changes game server properties, health status, or utilization status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameServer.html
     */
    toUpdateGameServer() {
        this.to('gamelift:UpdateGameServer');
        return this;
    }
    /**
     * Updates properties for game server group, including allowed instance types.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameServerGroup.html
     */
    toUpdateGameServerGroup() {
        this.to('gamelift:UpdateGameServerGroup');
        return this;
    }
    /**
     * Updates the properties of an existing game session.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSession.html
     */
    toUpdateGameSession() {
        this.to('gamelift:UpdateGameSession');
        return this;
    }
    /**
     * Updates properties of an existing game session queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateGameSessionQueue.html
     */
    toUpdateGameSessionQueue() {
        this.to('gamelift:UpdateGameSessionQueue');
        return this;
    }
    /**
     * Updates properties of an existing FlexMatch matchmaking configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateMatchmakingConfiguration.html
     */
    toUpdateMatchmakingConfiguration() {
        this.to('gamelift:UpdateMatchmakingConfiguration');
        return this;
    }
    /**
     * Updates how server processes are configured on instances in an existing fleet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateRuntimeConfiguration.html
     */
    toUpdateRuntimeConfiguration() {
        this.to('gamelift:UpdateRuntimeConfiguration');
        return this;
    }
    /**
     * Updates the metadata and content of an existing Realtime Servers script.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateScript.html
     */
    toUpdateScript() {
        this.to('gamelift:UpdateScript');
        return this;
    }
    /**
     * Validates the syntax of a FlexMatch matchmaking rule set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gamelift/latest/apireference/API_ValidateMatchmakingRuleSet.html
     */
    toValidateMatchmakingRuleSet() {
        this.to('gamelift:ValidateMatchmakingRuleSet');
        return this;
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_Alias.html
     *
     * @param aliasId - Identifier for the aliasId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlias(aliasId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}::alias/${AliasId}';
        arn = arn.replace('${AliasId}', aliasId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_Build.html
     *
     * @param buildId - Identifier for the buildId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBuild(buildId, accountId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${AccountId}:build/${BuildId}';
        arn = arn.replace('${BuildId}', buildId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type script to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_Script.html
     *
     * @param scriptId - Identifier for the scriptId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScript(scriptId, accountId, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${AccountId}:script/${ScriptId}';
        arn = arn.replace('${ScriptId}', scriptId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_FleetAttributes.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:fleet/${FleetId}';
        arn = arn.replace('${FleetId}', fleetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gameSessionQueue to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_GameSessionQueue.html
     *
     * @param gameSessionQueueName - Identifier for the gameSessionQueueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGameSessionQueue(gameSessionQueueName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:gamesessionqueue/${GameSessionQueueName}';
        arn = arn.replace('${GameSessionQueueName}', gameSessionQueueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type matchmakingConfiguration to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_MatchmakingConfiguration.html
     *
     * @param matchmakingConfigurationName - Identifier for the matchmakingConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchmakingConfiguration(matchmakingConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:matchmakingconfiguration/${MatchmakingConfigurationName}';
        arn = arn.replace('${MatchmakingConfigurationName}', matchmakingConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type matchmakingRuleSet to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_MatchmakingRuleSet.html
     *
     * @param matchmakingRuleSetName - Identifier for the matchmakingRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchmakingRuleSet(matchmakingRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:matchmakingruleset/${MatchmakingRuleSetName}';
        arn = arn.replace('${MatchmakingRuleSetName}', matchmakingRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gameServerGroup to the statement
     *
     * https://docs.aws.amazon.com/gamelift/latest/developerguide/API_GameServerGroup.html
     *
     * @param gameServerGroupName - Identifier for the gameServerGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGameServerGroup(gameServerGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:gamelift:${Region}:${Account}:gameservergroup/${GameServerGroupName}';
        arn = arn.replace('${GameServerGroupName}', gameServerGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Gamelift = Gamelift;
//# sourceMappingURL=data:application/json;base64,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