"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesis = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesis](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesis.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesis extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesis](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesis.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesis';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToStream",
                "RemoveTagsFromStream"
            ],
            "Write": [
                "CreateStream",
                "DecreaseStreamRetentionPeriod",
                "DeleteStream",
                "DeregisterStreamConsumer",
                "DisableEnhancedMonitoring",
                "EnableEnhancedMonitoring",
                "IncreaseStreamRetentionPeriod",
                "MergeShards",
                "PutRecord",
                "PutRecords",
                "RegisterStreamConsumer",
                "SplitShard",
                "StartStreamEncryption",
                "StopStreamEncryption",
                "UpdateShardCount"
            ],
            "Read": [
                "DescribeLimits",
                "DescribeStream",
                "DescribeStreamConsumer",
                "DescribeStreamSummary",
                "GetRecords",
                "GetShardIterator",
                "ListTagsForStream",
                "SubscribeToShard"
            ],
            "List": [
                "ListShards",
                "ListStreamConsumers",
                "ListStreams"
            ]
        };
    }
    /**
     * Adds or updates tags for the specified Amazon Kinesis stream. Each stream can have up to 10 tags.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_AddTagsToStream.html
     */
    toAddTagsToStream() {
        this.to('kinesis:AddTagsToStream');
        return this;
    }
    /**
     * Creates a Amazon Kinesis stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_CreateStream.html
     */
    toCreateStream() {
        this.to('kinesis:CreateStream');
        return this;
    }
    /**
     * Decreases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DecreaseStreamRetentionPeriod.html
     */
    toDecreaseStreamRetentionPeriod() {
        this.to('kinesis:DecreaseStreamRetentionPeriod');
        return this;
    }
    /**
     * Deletes a stream and all its shards and data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteStream.html
     */
    toDeleteStream() {
        this.to('kinesis:DeleteStream');
        return this;
    }
    /**
     * Deregisters a stream consumer with a Kinesis data stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeregisterStreamConsumer.html
     */
    toDeregisterStreamConsumer() {
        this.to('kinesis:DeregisterStreamConsumer');
        return this;
    }
    /**
     * Describes the shard limits and usage for the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        this.to('kinesis:DescribeLimits');
        return this;
    }
    /**
     * Describes the specified stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStream.html
     */
    toDescribeStream() {
        this.to('kinesis:DescribeStream');
        return this;
    }
    /**
     * Gets the description of a registered stream consumer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamConsumer.html
     */
    toDescribeStreamConsumer() {
        this.to('kinesis:DescribeStreamConsumer');
        return this;
    }
    /**
     * Provides a summarized description of the specified Kinesis data stream without the shard list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamSummary.html
     */
    toDescribeStreamSummary() {
        this.to('kinesis:DescribeStreamSummary');
        return this;
    }
    /**
     * Disables enhanced monitoring.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DisableEnhancedMonitoring.html
     */
    toDisableEnhancedMonitoring() {
        this.to('kinesis:DisableEnhancedMonitoring');
        return this;
    }
    /**
     * API_EnableEnhancedMonitoring.html
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_EnableEnhancedMonitoring.html
     */
    toEnableEnhancedMonitoring() {
        this.to('kinesis:EnableEnhancedMonitoring');
        return this;
    }
    /**
     * Gets data records from a shard.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html
     */
    toGetRecords() {
        this.to('kinesis:GetRecords');
        return this;
    }
    /**
     * Gets a shard iterator. A shard iterator expires five minutes after it is returned to the requester.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html
     */
    toGetShardIterator() {
        this.to('kinesis:GetShardIterator');
        return this;
    }
    /**
     * Increases the stream's retention period, which is the length of time data records are accessible after they are added to the stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_IncreaseStreamRetentionPeriod.html
     */
    toIncreaseStreamRetentionPeriod() {
        this.to('kinesis:IncreaseStreamRetentionPeriod');
        return this;
    }
    /**
     * Lists the shards in a stream and provides information about each shard.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListShards.html
     */
    toListShards() {
        this.to('kinesis:ListShards');
        return this;
    }
    /**
     * Lists the stream consumers registered to receive data from a Kinesis stream using enhanced fan-out, and provides information about each consumer.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreamConsumers.html
     */
    toListStreamConsumers() {
        this.to('kinesis:ListStreamConsumers');
        return this;
    }
    /**
     * Lists your streams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        this.to('kinesis:ListStreams');
        return this;
    }
    /**
     * Lists the tags for the specified Amazon Kinesis stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForStream.html
     */
    toListTagsForStream() {
        this.to('kinesis:ListTagsForStream');
        return this;
    }
    /**
     * Merges two adjacent shards in a stream and combines them into a single shard to reduce the stream's capacity to ingest and transport data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_MergeShards.html
     */
    toMergeShards() {
        this.to('kinesis:MergeShards');
        return this;
    }
    /**
     * Writes a single data record from a producer into an Amazon Kinesis stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecord.html
     */
    toPutRecord() {
        this.to('kinesis:PutRecord');
        return this;
    }
    /**
     * Writes multiple data records from a producer into an Amazon Kinesis stream in a single call (also referred to as a PutRecords request).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecords.html
     */
    toPutRecords() {
        this.to('kinesis:PutRecords');
        return this;
    }
    /**
     * Registers a stream consumer with a Kinesis data stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RegisterStreamConsumer.html
     */
    toRegisterStreamConsumer() {
        this.to('kinesis:RegisterStreamConsumer');
        return this;
    }
    /**
     * Description for SplitShard
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RemoveTagsFromStream.html
     */
    toRemoveTagsFromStream() {
        this.to('kinesis:RemoveTagsFromStream');
        return this;
    }
    /**
     * Description for SplitShard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SplitShard.html
     */
    toSplitShard() {
        this.to('kinesis:SplitShard');
        return this;
    }
    /**
     * Grants permission to enable or update server-side encryption using an AWS KMS key for a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StartStreamEncryption.html
     */
    toStartStreamEncryption() {
        this.to('kinesis:StartStreamEncryption');
        return this;
    }
    /**
     * Grants permission to disable server-side encryption for a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StopStreamEncryption.html
     */
    toStopStreamEncryption() {
        this.to('kinesis:StopStreamEncryption');
        return this;
    }
    /**
     * Listening to a specific shard with enhanced fan-out.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SubscribeToShard.html
     */
    toSubscribeToShard() {
        this.to('kinesis:SubscribeToShard');
        return this;
    }
    /**
     * Updates the shard count of the specified stream to the specified number of shards.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateShardCount.html
     */
    toUpdateShardCount() {
        this.to('kinesis:UpdateShardCount');
        return this;
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-streams.html
     *
     * @param streamName - Identifier for the streamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStream(streamName, account, region, partition) {
        var arn = 'arn:${Partition}:kinesis:${Region}:${Account}:stream/${StreamName}';
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type consumer to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-consumers.html
     *
     * @param streamType - Identifier for the streamType.
     * @param streamName - Identifier for the streamName.
     * @param consumerName - Identifier for the consumerName.
     * @param consumerCreationTimpstamp - Identifier for the consumerCreationTimpstamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConsumer(streamType, streamName, consumerName, consumerCreationTimpstamp, account, region, partition) {
        var arn = 'arn:${Partition}:kinesis:${Region}:${Account}:${StreamType}/${StreamName}/consumer/${ConsumerName}:${ConsumerCreationTimpstamp}';
        arn = arn.replace('${StreamType}', streamType);
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${ConsumerName}', consumerName);
        arn = arn.replace('${ConsumerCreationTimpstamp}', consumerCreationTimpstamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type kmsKey to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#master_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onKmsKey(keyId, account, region, partition) {
        var arn = 'arn:${Partition}:kms:${Region}:${Account}:key/${KeyId}';
        arn = arn.replace('${KeyId}', keyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kinesis = Kinesis;
//# sourceMappingURL=data:application/json;base64,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