"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Macie = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [macie](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacieclassic.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Macie extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [macie](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmacieclassic.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'macie';
        this.accessLevelList = {
            "Write": [
                "AssociateMemberAccount",
                "AssociateS3Resources",
                "DisassociateMemberAccount",
                "DisassociateS3Resources",
                "UpdateS3Resources"
            ],
            "List": [
                "ListMemberAccounts",
                "ListS3Resources"
            ]
        };
    }
    /**
     * Enables the user to associate a specified AWS account with Amazon Macie as a member account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateMemberAccount.html
     */
    toAssociateMemberAccount() {
        this.to('macie:AssociateMemberAccount');
        return this;
    }
    /**
     * Enables the user to associate specified S3 resources with Amazon Macie for monitoring and data classification.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_AssociateS3Resources.html
     */
    toAssociateS3Resources() {
        this.to('macie:AssociateS3Resources');
        return this;
    }
    /**
     * Enables the user to remove the specified member account from Amazon Macie.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateMemberAccount.html
     */
    toDisassociateMemberAccount() {
        this.to('macie:DisassociateMemberAccount');
        return this;
    }
    /**
     * Enables the user to remove specified S3 resources from being monitored by Amazon Macie.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_DisassociateS3Resources.html
     */
    toDisassociateS3Resources() {
        this.to('macie:DisassociateS3Resources');
        return this;
    }
    /**
     * Enables the user to list all Amazon Macie member accounts for the current Macie master account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListMemberAccounts.html
     */
    toListMemberAccounts() {
        this.to('macie:ListMemberAccounts');
        return this;
    }
    /**
     * Enables the user to list all the S3 resources associated with Amazon Macie.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_ListS3Resources.html
     */
    toListS3Resources() {
        this.to('macie:ListS3Resources');
        return this;
    }
    /**
     * Enables the user to update the classification types for the specified S3 resources.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsSourceArn()
     *
     * https://docs.aws.amazon.com/macie/1.0/APIReference/API_UpdateS3Resources.html
     */
    toUpdateS3Resources() {
        this.to('macie:UpdateS3Resources');
        return this;
    }
}
exports.Macie = Macie;
//# sourceMappingURL=data:application/json;base64,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