"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mq = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mq](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmq.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mq extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mq](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmq.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mq';
        this.accessLevelList = {
            "Write": [
                "CreateBroker",
                "CreateConfiguration",
                "CreateTags",
                "CreateUser",
                "DeleteBroker",
                "DeleteTags",
                "DeleteUser",
                "RebootBroker",
                "UpdateBroker",
                "UpdateConfiguration",
                "UpdateUser"
            ],
            "Read": [
                "DescribeBroker",
                "DescribeBrokerEngineTypes",
                "DescribeBrokerInstanceOptions",
                "DescribeConfiguration",
                "DescribeConfigurationRevision",
                "DescribeUser"
            ],
            "List": [
                "ListBrokers",
                "ListConfigurationRevisions",
                "ListConfigurations",
                "ListTags",
                "ListUsers"
            ]
        };
    }
    /**
     * Grants permission to create a broker.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateSecurityGroup
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeInternetGateways
     * - ec2:DescribeNetworkInterfacePermissions
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyNetworkInterfaceAttribute
     * - iam:CreateServiceLinkedRole
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-post
     */
    toCreateBroker() {
        this.to('mq:CreateBroker');
        return this;
    }
    /**
     * Grants permission to create a new configuration for the specified configuration name. Amazon MQ uses the default configuration (the engine type and engine version).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-post
     */
    toCreateConfiguration() {
        this.to('mq:CreateConfiguration');
        return this;
    }
    /**
     * Grants permission to create tags.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-post
     */
    toCreateTags() {
        this.to('mq:CreateTags');
        return this;
    }
    /**
     * Grants permission to create an ActiveMQ user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-post
     */
    toCreateUser() {
        this.to('mq:CreateUser');
        return this;
    }
    /**
     * Grants permission to delete a broker.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteNetworkInterfacePermission
     * - ec2:DeleteVpcEndpoints
     * - ec2:DetachNetworkInterface
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-delete
     */
    toDeleteBroker() {
        this.to('mq:DeleteBroker');
        return this;
    }
    /**
     * Grants permission to delete tags.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    toDeleteTags() {
        this.to('mq:DeleteTags');
        return this;
    }
    /**
     * Grants permission to delete an ActiveMQ user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-delete
     */
    toDeleteUser() {
        this.to('mq:DeleteUser');
        return this;
    }
    /**
     * Grants permission to return information about the specified broker.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toDescribeBroker() {
        this.to('mq:DescribeBroker');
        return this;
    }
    /**
     * Grants permission to return information about broker engines.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-engine-types.html#broker-engine-types-http-methods
     */
    toDescribeBrokerEngineTypes() {
        this.to('mq:DescribeBrokerEngineTypes');
        return this;
    }
    /**
     * Grants permission to return information about the broker instance options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-instance-options.html#broker-engine-types-http-methods
     */
    toDescribeBrokerInstanceOptions() {
        this.to('mq:DescribeBrokerInstanceOptions');
        return this;
    }
    /**
     * Grants permission to return information about the specified configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-get
     */
    toDescribeConfiguration() {
        this.to('mq:DescribeConfiguration');
        return this;
    }
    /**
     * Grants permission to return the specified configuration revision for the specified configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration-revision.html#rest-api-configuration-revision-methods-get
     */
    toDescribeConfigurationRevision() {
        this.to('mq:DescribeConfigurationRevision');
        return this;
    }
    /**
     * Grants permission to return information about an ActiveMQ user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-get
     */
    toDescribeUser() {
        this.to('mq:DescribeUser');
        return this;
    }
    /**
     * Grants permission to return a list of all brokers.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-get
     */
    toListBrokers() {
        this.to('mq:ListBrokers');
        return this;
    }
    /**
     * Grants permission to return a list of all existing revisions for the specified configuration.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-revisions.html#rest-api-revisions-methods-get
     */
    toListConfigurationRevisions() {
        this.to('mq:ListConfigurationRevisions');
        return this;
    }
    /**
     * Grants permission to return a list of all configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-get
     */
    toListConfigurations() {
        this.to('mq:ListConfigurations');
        return this;
    }
    /**
     * Grants permission to return a list of tags.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-get
     */
    toListTags() {
        this.to('mq:ListTags');
        return this;
    }
    /**
     * Grants permission to return a list of all ActiveMQ users.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-users.html#rest-api-users-methods-get
     */
    toListUsers() {
        this.to('mq:ListUsers');
        return this;
    }
    /**
     * Grants permission to reboot a broker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-restart.html#rest-api-reboot-methods-post
     */
    toRebootBroker() {
        this.to('mq:RebootBroker');
        return this;
    }
    /**
     * Grants permission to add a pending configuration change to a broker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toUpdateBroker() {
        this.to('mq:UpdateBroker');
        return this;
    }
    /**
     * Grants permission to update the specified configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-put
     */
    toUpdateConfiguration() {
        this.to('mq:UpdateConfiguration');
        return this;
    }
    /**
     * Grants permission to update the information for an ActiveMQ user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-put
     */
    toUpdateUser() {
        this.to('mq:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type brokers to the statement
     *
     * @param brokerId - Identifier for the brokerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBrokers(brokerId, account, region, partition) {
        var arn = 'arn:${Partition}:mq:${Region}:${Account}:broker:${Broker-id}';
        arn = arn.replace('${Broker-id}', brokerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type configurations to the statement
     *
     * @param configurationId - Identifier for the configurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurations(configurationId, account, region, partition) {
        var arn = 'arn:${Partition}:mq:${Region}:${Account}:configuration:${Configuration-id}';
        arn = arn.replace('${Configuration-id}', configurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mq = Mq;
//# sourceMappingURL=data:application/json;base64,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