"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workmail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [workmail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkmail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workmail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [workmail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworkmail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'workmail';
        this.accessLevelList = {
            "Write": [
                "AddMembersToGroup",
                "AssociateDelegateToResource",
                "AssociateMemberToGroup",
                "CancelMailboxExportJob",
                "CreateAlias",
                "CreateGroup",
                "CreateInboundMailFlowRule",
                "CreateMailDomain",
                "CreateMailUser",
                "CreateOrganization",
                "CreateOutboundMailFlowRule",
                "CreateResource",
                "CreateSmtpGateway",
                "CreateUser",
                "DeleteAccessControlRule",
                "DeleteAlias",
                "DeleteGroup",
                "DeleteInboundMailFlowRule",
                "DeleteMailDomain",
                "DeleteMailboxPermissions",
                "DeleteMobileDevice",
                "DeleteOrganization",
                "DeleteOutboundMailFlowRule",
                "DeleteResource",
                "DeleteRetentionPolicy",
                "DeleteSmtpGateway",
                "DeleteUser",
                "DeregisterFromWorkMail",
                "DisableMailGroups",
                "DisableMailUsers",
                "DisassociateDelegateFromResource",
                "DisassociateMemberFromGroup",
                "EnableMailDomain",
                "EnableMailGroups",
                "EnableMailUsers",
                "PutAccessControlRule",
                "PutMailboxPermissions",
                "PutRetentionPolicy",
                "RegisterToWorkMail",
                "RemoveMembersFromGroup",
                "ResetPassword",
                "ResetUserPassword",
                "SetAdmin",
                "SetDefaultMailDomain",
                "SetJournalingRules",
                "SetMailGroupDetails",
                "SetMailUserDetails",
                "SetMobilePolicyDetails",
                "StartMailboxExportJob",
                "TestInboundMailFlowRules",
                "TestOutboundMailFlowRules",
                "UpdateInboundMailFlowRule",
                "UpdateMailboxQuota",
                "UpdateOutboundMailFlowRule",
                "UpdatePrimaryEmailAddress",
                "UpdateResource",
                "UpdateSmtpGateway",
                "WipeMobileDevice"
            ],
            "List": [
                "DescribeDirectories",
                "DescribeGroup",
                "DescribeKmsKeys",
                "DescribeMailDomains",
                "DescribeMailGroups",
                "DescribeMailUsers",
                "DescribeOrganization",
                "DescribeOrganizations",
                "DescribeResource",
                "DescribeUser",
                "ListAccessControlRules",
                "ListAliases",
                "ListGroupMembers",
                "ListGroups",
                "ListInboundMailFlowRules",
                "ListMailboxExportJobs",
                "ListMailboxPermissions",
                "ListOrganizations",
                "ListOutboundMailFlowRules",
                "ListResourceDelegates",
                "ListResources",
                "ListSmtpGateways",
                "ListTagsForResource",
                "ListUsers"
            ],
            "Read": [
                "DescribeInboundMailFlowRule",
                "DescribeMailboxExportJob",
                "DescribeOutboundMailFlowRule",
                "DescribeSmtpGateway",
                "GetAccessControlEffect",
                "GetDefaultRetentionPolicy",
                "GetJournalingRules",
                "GetMailDomainDetails",
                "GetMailGroupDetails",
                "GetMailUserDetails",
                "GetMailboxDetails",
                "GetMobileDeviceDetails",
                "GetMobileDevicesForUser",
                "GetMobilePolicyDetails",
                "ListMembersInMailGroup",
                "SearchMembers"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Adds a list of members (users or groups) to a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toAddMembersToGroup() {
        this.to('workmail:AddMembersToGroup');
        return this;
    }
    /**
     * Adds a member (user or group) to the resource's set of delegates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateDelegateToResource.html
     */
    toAssociateDelegateToResource() {
        this.to('workmail:AssociateDelegateToResource');
        return this;
    }
    /**
     * Adds a member (user or group) to the group's set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_AssociateMemberToGroup.html
     */
    toAssociateMemberToGroup() {
        this.to('workmail:AssociateMemberToGroup');
        return this;
    }
    /**
     * Cancels a currently running mailbox export job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CancelMailboxExportJob.html
     */
    toCancelMailboxExportJob() {
        this.to('workmail:CancelMailboxExportJob');
        return this;
    }
    /**
     * Adds an alias to the set of a given member (user or group) of WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        this.to('workmail:CreateAlias');
        return this;
    }
    /**
     * Creates a group that can be used in WorkMail by calling the RegisterToWorkMail operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('workmail:CreateGroup');
        return this;
    }
    /**
     * Create an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    toCreateInboundMailFlowRule() {
        this.to('workmail:CreateInboundMailFlowRule');
        return this;
    }
    /**
     * Creates a mail domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    toCreateMailDomain() {
        this.to('workmail:CreateMailDomain');
        return this;
    }
    /**
     * Creates a user in the directory and the WorkMail storage but does not enable the user for mail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html
     */
    toCreateMailUser() {
        this.to('workmail:CreateMailUser');
        return this;
    }
    /**
     * Creates an organization, either using an existing directory or creates a new directory on-the-fly. Also creates and enables the complementary mail domain. Optionally creates KMS key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    toCreateOrganization() {
        this.to('workmail:CreateOrganization');
        return this;
    }
    /**
     * Create an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/create-email-rules.html
     */
    toCreateOutboundMailFlowRule() {
        this.to('workmail:CreateOutboundMailFlowRule');
        return this;
    }
    /**
     * Creates a new WorkMail resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateResource.html
     */
    toCreateResource() {
        this.to('workmail:CreateResource');
        return this;
    }
    /**
     * Register an SMTP device against a WorkMail organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toCreateSmtpGateway() {
        this.to('workmail:CreateSmtpGateway');
        return this;
    }
    /**
     * Creates a user who can be used in WorkMail by calling the RegisterToWorkMail operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.to('workmail:CreateUser');
        return this;
    }
    /**
     * Deletes an access control rule for the specified WorkMail organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAccessControlRule.html
     */
    toDeleteAccessControlRule() {
        this.to('workmail:DeleteAccessControlRule');
        return this;
    }
    /**
     * Remove one or more specified aliases from a set of aliases for a given user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        this.to('workmail:DeleteAlias');
        return this;
    }
    /**
     * Deletes a group from WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('workmail:DeleteGroup');
        return this;
    }
    /**
     * Remove an inbound email flow rule to no longer apply to emails sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    toDeleteInboundMailFlowRule() {
        this.to('workmail:DeleteInboundMailFlowRule');
        return this;
    }
    /**
     * Removes an unused mail domain from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_domain.html
     */
    toDeleteMailDomain() {
        this.to('workmail:DeleteMailDomain');
        return this;
    }
    /**
     * Deletes permissions granted to a member (user or group).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteMailboxPermissions.html
     */
    toDeleteMailboxPermissions() {
        this.to('workmail:DeleteMailboxPermissions');
        return this;
    }
    /**
     * Removes a mobile device from a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remove_mobile_device
     */
    toDeleteMobileDevice() {
        this.to('workmail:DeleteMobileDevice');
        return this;
    }
    /**
     * Removes an organization from an account, either removing the directory from directory services or leaving it available for re-use
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_organization.html
     */
    toDeleteOrganization() {
        this.to('workmail:DeleteOrganization');
        return this;
    }
    /**
     * Remove an outbound email flow rule to no longer apply to emails sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove-email-flow-rule.html
     */
    toDeleteOutboundMailFlowRule() {
        this.to('workmail:DeleteOutboundMailFlowRule');
        return this;
    }
    /**
     * Deletes the specified resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteResource.html
     */
    toDeleteResource() {
        this.to('workmail:DeleteResource');
        return this;
    }
    /**
     * Deletes the retention policy based on the supplied organization and policy identifiers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    toDeleteRetentionPolicy() {
        this.to('workmail:DeleteRetentionPolicy');
        return this;
    }
    /**
     * Remove an SMTP device from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toDeleteSmtpGateway() {
        this.to('workmail:DeleteSmtpGateway');
        return this;
    }
    /**
     * Deletes a user from WorkMail and all subsequent systems. The action cannot be undone.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        this.to('workmail:DeleteUser');
        return this;
    }
    /**
     * Mark a user, group, or resource as no longer used in WorkMail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DeregisterFromWorkMail.html
     */
    toDeregisterFromWorkMail() {
        this.to('workmail:DeregisterFromWorkMail');
        return this;
    }
    /**
     * Shows a list of directories available for use in creating an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    toDescribeDirectories() {
        this.to('workmail:DescribeDirectories');
        return this;
    }
    /**
     * Returns the data available for the group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        this.to('workmail:DescribeGroup');
        return this;
    }
    /**
     * Returns the details of an inbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    toDescribeInboundMailFlowRule() {
        this.to('workmail:DescribeInboundMailFlowRule');
        return this;
    }
    /**
     * Shows a list of KMS Keys available for use in creating an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_organization.html
     */
    toDescribeKmsKeys() {
        this.to('workmail:DescribeKmsKeys');
        return this;
    }
    /**
     * Shows the details of all mail domains associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    toDescribeMailDomains() {
        this.to('workmail:DescribeMailDomains');
        return this;
    }
    /**
     * Shows the details of all groups associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toDescribeMailGroups() {
        this.to('workmail:DescribeMailGroups');
        return this;
    }
    /**
     * Shows the details of all users associated with the orgaization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    toDescribeMailUsers() {
        this.to('workmail:DescribeMailUsers');
        return this;
    }
    /**
     * Retrieve details of a mailbox export job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeMailboxExportJob.html
     */
    toDescribeMailboxExportJob() {
        this.to('workmail:DescribeMailboxExportJob');
        return this;
    }
    /**
     * Provides more information regarding a given organization based on its identifier.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        this.to('workmail:DescribeOrganization');
        return this;
    }
    /**
     * Shows a summary of all organizations associated with the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     */
    toDescribeOrganizations() {
        this.to('workmail:DescribeOrganizations');
        return this;
    }
    /**
     * Returns the details of an outbound mail flow rule configured for an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    toDescribeOutboundMailFlowRule() {
        this.to('workmail:DescribeOutboundMailFlowRule');
        return this;
    }
    /**
     * Returns the data available for the resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeResource.html
     */
    toDescribeResource() {
        this.to('workmail:DescribeResource');
        return this;
    }
    /**
     * Returns the details of an SMTP device registered against an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toDescribeSmtpGateway() {
        this.to('workmail:DescribeSmtpGateway');
        return this;
    }
    /**
     * Provides information regarding the user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        this.to('workmail:DescribeUser');
        return this;
    }
    /**
     * Disable a mail group when it is not being used and, to allow it to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/remove_group.html
     */
    toDisableMailGroups() {
        this.to('workmail:DisableMailGroups');
        return this;
    }
    /**
     * Disable a user mailbox when it is no longer being used, and to allow it to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-mailboxes.html#delete_user_mailbox
     */
    toDisableMailUsers() {
        this.to('workmail:DisableMailUsers');
        return this;
    }
    /**
     * Removes a member from the resource's set of delegates.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateDelegateFromResource.html
     */
    toDisassociateDelegateFromResource() {
        this.to('workmail:DisassociateDelegateFromResource');
        return this;
    }
    /**
     * Removes a member from a group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_DisassociateMemberFromGroup.html
     */
    toDisassociateMemberFromGroup() {
        this.to('workmail:DisassociateMemberFromGroup');
        return this;
    }
    /**
     * Enable a mail domain in the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_domain.html
     */
    toEnableMailDomain() {
        this.to('workmail:EnableMailDomain');
        return this;
    }
    /**
     * Enable a mail group after it has been created to allow it to receive mail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/enable_existing_group.html
     */
    toEnableMailGroups() {
        this.to('workmail:EnableMailGroups');
        return this;
    }
    /**
     * Enable a user's mailbox after it has been created to allow it to receive mail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#enable_existing_user
     */
    toEnableMailUsers() {
        this.to('workmail:EnableMailUsers');
        return this;
    }
    /**
     * Gets the effects of an organization's access control rules as they apply to a specified IPv4 address, access protocol action, or user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetAccessControlEffect.html
     */
    toGetAccessControlEffect() {
        this.to('workmail:GetAccessControlEffect');
        return this;
    }
    /**
     * Retrieves the retention policy associated at an organizational level.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetDefaultRetentionPolicy.html
     */
    toGetDefaultRetentionPolicy() {
        this.to('workmail:GetDefaultRetentionPolicy');
        return this;
    }
    /**
     * Returns journaling and fallback email addresses configured for email journaling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    toGetJournalingRules() {
        this.to('workmail:GetJournalingRules');
        return this;
    }
    /**
     * Get the details of the mail domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/domains_overview.html
     */
    toGetMailDomainDetails() {
        this.to('workmail:GetMailDomainDetails');
        return this;
    }
    /**
     * Get the details of the mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toGetMailGroupDetails() {
        this.to('workmail:GetMailGroupDetails');
        return this;
    }
    /**
     * Get the details of the user's mailbox and account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    toGetMailUserDetails() {
        this.to('workmail:GetMailUserDetails');
        return this;
    }
    /**
     * Returns the details of the user's mailbox.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_GetMailboxDetails.html
     */
    toGetMailboxDetails() {
        this.to('workmail:GetMailboxDetails');
        return this;
    }
    /**
     * Get the details of the mobile device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    toGetMobileDeviceDetails() {
        this.to('workmail:GetMobileDeviceDetails');
        return this;
    }
    /**
     * Get a list of the mobile devices associated with the user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html
     */
    toGetMobileDevicesForUser() {
        this.to('workmail:GetMobileDevicesForUser');
        return this;
    }
    /**
     * Get the details of the mobile device policy associated with the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    toGetMobilePolicyDetails() {
        this.to('workmail:GetMobilePolicyDetails');
        return this;
    }
    /**
     * Lists the access control rules for the specified organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAccessControlRules.html
     */
    toListAccessControlRules() {
        this.to('workmail:ListAccessControlRules');
        return this;
    }
    /**
     * Creates a paginated call to list the aliases associated with a given entity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        this.to('workmail:ListAliases');
        return this;
    }
    /**
     * Returns an overview of the members of a group. Users and groups can be members of a group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroupMembers.html
     */
    toListGroupMembers() {
        this.to('workmail:ListGroupMembers');
        return this;
    }
    /**
     * Returns summaries of the organization's groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        this.to('workmail:ListGroups');
        return this;
    }
    /**
     * Returns a list of inbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-actions
     */
    toListInboundMailFlowRules() {
        this.to('workmail:ListInboundMailFlowRules');
        return this;
    }
    /**
     * List mailbox export jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxExportJobs.html
     */
    toListMailboxExportJobs() {
        this.to('workmail:ListMailboxExportJobs');
        return this;
    }
    /**
     * Lists the mailbox permissions associated with a user, group, or resource mailbox.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListMailboxPermissions.html
     */
    toListMailboxPermissions() {
        this.to('workmail:ListMailboxPermissions');
        return this;
    }
    /**
     * Get a list of all the members in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toListMembersInMailGroup() {
        this.to('workmail:ListMembersInMailGroup');
        return this;
    }
    /**
     * Returns summaries of the customer's non-deleted organizations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListOrganizations.html
     */
    toListOrganizations() {
        this.to('workmail:ListOrganizations');
        return this;
    }
    /**
     * Returns a list of outbound mail flow rules configured for an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/email-flows.html#email-flows-rule-outbound
     */
    toListOutboundMailFlowRules() {
        this.to('workmail:ListOutboundMailFlowRules');
        return this;
    }
    /**
     * Lists the delegates associated with a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResourceDelegates.html
     */
    toListResourceDelegates() {
        this.to('workmail:ListResourceDelegates');
        return this;
    }
    /**
     * Returns summaries of the organization's resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListResources.html
     */
    toListResources() {
        this.to('workmail:ListResources');
        return this;
    }
    /**
     * Returns a list of SMTP devices registered against the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toListSmtpGateways() {
        this.to('workmail:ListSmtpGateways');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon WorkMail organization resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('workmail:ListTagsForResource');
        return this;
    }
    /**
     * Returns summaries of the organization's users.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.to('workmail:ListUsers');
        return this;
    }
    /**
     * Adds a new access control rule for the specified organization. The rule allows or denies access to the organization for the specified IPv4 addresses, access protocol actions, and user IDs. Adding a new rule with the same name as an existing rule replaces the older rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutAccessControlRule.html
     */
    toPutAccessControlRule() {
        this.to('workmail:PutAccessControlRule');
        return this;
    }
    /**
     * Sets permissions for a user, group, or resource. This replaces any pre-existing permissions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutMailboxPermissions.html
     */
    toPutMailboxPermissions() {
        this.to('workmail:PutMailboxPermissions');
        return this;
    }
    /**
     * Adds or updates the retention policy for the specified organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_PutRetentionPolicy.html
     */
    toPutRetentionPolicy() {
        this.to('workmail:PutRetentionPolicy');
        return this;
    }
    /**
     * Registers an existing and disabled user, group, or resource for use by associating a mailbox and calendaring capabilities.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_RegisterToWorkMail.html
     */
    toRegisterToWorkMail() {
        this.to('workmail:RegisterToWorkMail');
        return this;
    }
    /**
     * Remove members from a mail group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toRemoveMembersFromGroup() {
        this.to('workmail:RemoveMembersFromGroup');
        return this;
    }
    /**
     * Allows the administrator to reset the password for a user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_ResetPassword.html
     */
    toResetPassword() {
        this.to('workmail:ResetPassword');
        return this;
    }
    /**
     * Reset the password for a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html#reset_user_password
     */
    toResetUserPassword() {
        this.to('workmail:ResetUserPassword');
        return this;
    }
    /**
     * Prefix search to find a specific user in a mail group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/groups_overview.html
     */
    toSearchMembers() {
        this.to('workmail:SearchMembers');
        return this;
    }
    /**
     * Mark a user as being an administrator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/users_overview.html
     */
    toSetAdmin() {
        this.to('workmail:SetAdmin');
        return this;
    }
    /**
     * Set the default mail domain for the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/default_domain.html
     */
    toSetDefaultMailDomain() {
        this.to('workmail:SetDefaultMailDomain');
        return this;
    }
    /**
     * Set journaling and fallback email addresses for email journaling
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/journaling_overview.html
     */
    toSetJournalingRules() {
        this.to('workmail:SetJournalingRules');
        return this;
    }
    /**
     * Set the details of the mail group which has just been created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/add_new_group.html
     */
    toSetMailGroupDetails() {
        this.to('workmail:SetMailGroupDetails');
        return this;
    }
    /**
     * Set the details for the user account which has just been created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-users.html
     */
    toSetMailUserDetails() {
        this.to('workmail:SetMailUserDetails');
        return this;
    }
    /**
     * Set the details of a mobile policy associated with the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/edit_organization_mobile_policy.html
     */
    toSetMobilePolicyDetails() {
        this.to('workmail:SetMobilePolicyDetails');
        return this;
    }
    /**
     * Start a new mailbox export job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_StartMailboxExportJob.html
     */
    toStartMailboxExportJob() {
        this.to('workmail:StartMailboxExportJob');
        return this;
    }
    /**
     * Grants permission to tag the specified Amazon WorkMail organization resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('workmail:TagResource');
        return this;
    }
    /**
     * Test what inbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    toTestInboundMailFlowRules() {
        this.to('workmail:TestInboundMailFlowRules');
        return this;
    }
    /**
     * Test what outbound rules will apply to an email with a given sender and recipient
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/test-email-flow-rule.html
     */
    toTestOutboundMailFlowRules() {
        this.to('workmail:TestOutboundMailFlowRules');
        return this;
    }
    /**
     * Grants permission to untag the specified Amazon WorkMail organization resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('workmail:UntagResource');
        return this;
    }
    /**
     * Update the details of an inbound email flow rule which will apply to all email sent to an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    toUpdateInboundMailFlowRule() {
        this.to('workmail:UpdateInboundMailFlowRule');
        return this;
    }
    /**
     * Updates the maximum size (in MB) of the user's mailbox.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateMailboxQuota.html
     */
    toUpdateMailboxQuota() {
        this.to('workmail:UpdateMailboxQuota');
        return this;
    }
    /**
     * Update the details of an outbound email flow rule which will apply to all email sent from an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/modify-email-flow-rule.html
     */
    toUpdateOutboundMailFlowRule() {
        this.to('workmail:UpdateOutboundMailFlowRule');
        return this;
    }
    /**
     * Updates the primary email for a user, group, or resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdatePrimaryEmailAddress.html
     */
    toUpdatePrimaryEmailAddress() {
        this.to('workmail:UpdatePrimaryEmailAddress');
        return this;
    }
    /**
     * Updates data for the resource. To retrieve the latest information, it must be preceded by a DescribeResource call.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/APIReference/API_UpdateResource.html
     */
    toUpdateResource() {
        this.to('workmail:UpdateResource');
        return this;
    }
    /**
     * Update the details of an existing SMTP device registered against an organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/smtp-gateway.html
     */
    toUpdateSmtpGateway() {
        this.to('workmail:UpdateSmtpGateway');
        return this;
    }
    /**
     * Remotely wipe the mobile device associated with a user's account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/manage-devices.html#remote_wipe_device
     */
    toWipeMobileDevice() {
        this.to('workmail:WipeMobileDevice');
        return this;
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/workmail/latest/adminguide/organizations_overview.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganization(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:workmail:${Region}:${Account}:organization/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Workmail = Workmail;
//# sourceMappingURL=data:application/json;base64,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