"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appconfig = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appconfig](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappconfig.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appconfig extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appconfig](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappconfig.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appconfig';
        this.accessLevelList = {
            "Write": [
                "CreateApplication",
                "CreateConfigurationProfile",
                "CreateDeploymentStrategy",
                "CreateEnvironment",
                "CreateHostedConfigurationVersion",
                "DeleteApplication",
                "DeleteConfigurationProfile",
                "DeleteDeploymentStrategy",
                "DeleteEnvironment",
                "DeleteHostedConfigurationVersion",
                "StartDeployment",
                "StopDeployment",
                "UpdateApplication",
                "UpdateConfigurationProfile",
                "UpdateDeploymentStrategy",
                "UpdateEnvironment",
                "ValidateConfiguration"
            ],
            "Read": [
                "GetApplication",
                "GetConfiguration",
                "GetConfigurationProfile",
                "GetDeployment",
                "GetDeploymentStrategy",
                "GetEnvironment",
                "GetHostedConfigurationVersion",
                "ListTagsForResource"
            ],
            "List": [
                "ListApplications",
                "ListConfigurationProfiles",
                "ListDeploymentStrategies",
                "ListDeployments",
                "ListEnvironments",
                "ListHostedConfigurationVersions"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        this.to('appconfig:CreateApplication');
        return this;
    }
    /**
     * Grants permission to create a configuration profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_CreateConfigurationProfile.html
     */
    toCreateConfigurationProfile() {
        this.to('appconfig:CreateConfigurationProfile');
        return this;
    }
    /**
     * Grants permission to create a deployment strategy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_CreateDeploymentStrategy.html
     */
    toCreateDeploymentStrategy() {
        this.to('appconfig:CreateDeploymentStrategy');
        return this;
    }
    /**
     * Grants permission to create an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        this.to('appconfig:CreateEnvironment');
        return this;
    }
    /**
     * Grants permission to create a hosted configuration version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_CreateHostedConfigurationVersion.html
     */
    toCreateHostedConfigurationVersion() {
        this.to('appconfig:CreateHostedConfigurationVersion');
        return this;
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        this.to('appconfig:DeleteApplication');
        return this;
    }
    /**
     * Grants permission to delete a configuration profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_DeleteConfigurationProfile.html
     */
    toDeleteConfigurationProfile() {
        this.to('appconfig:DeleteConfigurationProfile');
        return this;
    }
    /**
     * Grants permission to delete a deployment strategy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_DeleteDeploymentStrategy.html
     */
    toDeleteDeploymentStrategy() {
        this.to('appconfig:DeleteDeploymentStrategy');
        return this;
    }
    /**
     * Grants permission to delete an environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        this.to('appconfig:DeleteEnvironment');
        return this;
    }
    /**
     * Grants permission to delete a hosted configuration version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_DeleteHostedConfigurationVersion.html
     */
    toDeleteHostedConfigurationVersion() {
        this.to('appconfig:DeleteHostedConfigurationVersion');
        return this;
    }
    /**
     * Grants permission to view details about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        this.to('appconfig:GetApplication');
        return this;
    }
    /**
     * Grants permission to view details about a configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetConfiguration.html
     */
    toGetConfiguration() {
        this.to('appconfig:GetConfiguration');
        return this;
    }
    /**
     * Grants permission to view details about a configuration profile
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetConfigurationProfile.html
     */
    toGetConfigurationProfile() {
        this.to('appconfig:GetConfigurationProfile');
        return this;
    }
    /**
     * Grants permission to view details about a deployment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetDeployment.html
     */
    toGetDeployment() {
        this.to('appconfig:GetDeployment');
        return this;
    }
    /**
     * Grants permission to view details about a deployment strategy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetDeploymentStrategy.html
     */
    toGetDeploymentStrategy() {
        this.to('appconfig:GetDeploymentStrategy');
        return this;
    }
    /**
     * Grants permission to view details about an environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetEnvironment.html
     */
    toGetEnvironment() {
        this.to('appconfig:GetEnvironment');
        return this;
    }
    /**
     * Grants permission to view details about a hosted configuration version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_GetHostedConfigurationVersion.html
     */
    toGetHostedConfigurationVersion() {
        this.to('appconfig:GetHostedConfigurationVersion');
        return this;
    }
    /**
     * Grants permission to list the applications in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListApplications.html
     */
    toListApplications() {
        this.to('appconfig:ListApplications');
        return this;
    }
    /**
     * Grants permission to list the configuration profiles for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListConfigurationProfiles.html
     */
    toListConfigurationProfiles() {
        this.to('appconfig:ListConfigurationProfiles');
        return this;
    }
    /**
     * Grants permission to list the deployment strategies for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListDeploymentStrategies.html
     */
    toListDeploymentStrategies() {
        this.to('appconfig:ListDeploymentStrategies');
        return this;
    }
    /**
     * Grants permission to list the deployments for an environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListDeployments.html
     */
    toListDeployments() {
        this.to('appconfig:ListDeployments');
        return this;
    }
    /**
     * Grants permission to list the environments for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListEnvironments.html
     */
    toListEnvironments() {
        this.to('appconfig:ListEnvironments');
        return this;
    }
    /**
     * Grants permission to list the hosted configuration versions for a configuration profile
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListHostedConfigurationVersions.html
     */
    toListHostedConfigurationVersions() {
        this.to('appconfig:ListHostedConfigurationVersions');
        return this;
    }
    /**
     * Grants permission to view a list of resource tags for a specified resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('appconfig:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to initiate a deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_StartDeployment.html
     */
    toStartDeployment() {
        this.to('appconfig:StartDeployment');
        return this;
    }
    /**
     * Grants permission to stop a deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_StopDeployment.html
     */
    toStopDeployment() {
        this.to('appconfig:StopDeployment');
        return this;
    }
    /**
     * Grants permission to tag an appconfig resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('appconfig:TagResource');
        return this;
    }
    /**
     * Grants permission to untag an appconfig resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('appconfig:UntagResource');
        return this;
    }
    /**
     * Grants permission to modify an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.to('appconfig:UpdateApplication');
        return this;
    }
    /**
     * Grants permission to modify a configuration profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_UpdateConfigurationProfile.html
     */
    toUpdateConfigurationProfile() {
        this.to('appconfig:UpdateConfigurationProfile');
        return this;
    }
    /**
     * Grants permission to modify a deployment strategy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_UpdateDeploymentStrategy.html
     */
    toUpdateDeploymentStrategy() {
        this.to('appconfig:UpdateDeploymentStrategy');
        return this;
    }
    /**
     * Grants permission to modify an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_UpdateEnvironment.html
     */
    toUpdateEnvironment() {
        this.to('appconfig:UpdateEnvironment');
        return this;
    }
    /**
     * Grants permission to validate a configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_ValidateConfiguration.html
     */
    toValidateConfiguration() {
        this.to('appconfig:ValidateConfiguration');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-application.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationId, account, region, partition) {
        var arn = 'arn:${Partition}:appconfig:${Region}:${Account}:application/${ApplicationId}';
        arn = arn.replace('${ApplicationId}', applicationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-environment.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(applicationId, environmentId, account, region, partition) {
        var arn = 'arn:${Partition}:appconfig:${Region}:${Account}:application/${ApplicationId}/environment/${EnvironmentId}';
        arn = arn.replace('${ApplicationId}', applicationId);
        arn = arn.replace('${EnvironmentId}', environmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type configurationprofile to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-configuration-and-profile.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param configurationProfileId - Identifier for the configurationProfileId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationprofile(applicationId, configurationProfileId, account, region, partition) {
        var arn = 'arn:${Partition}:appconfig:${Region}:${Account}:application/${ApplicationId}/configurationprofile/${ConfigurationProfileId}';
        arn = arn.replace('${ApplicationId}', applicationId);
        arn = arn.replace('${ConfigurationProfileId}', configurationProfileId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentstrategy to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-deployment-strategy.html
     *
     * @param deploymentStrategyId - Identifier for the deploymentStrategyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentstrategy(deploymentStrategyId, account, region, partition) {
        var arn = 'arn:${Partition}:appconfig:${Region}:${Account}:deploymentstrategy/${DeploymentStrategyId}';
        arn = arn.replace('${DeploymentStrategyId}', deploymentStrategyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deployment to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-deploying.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param environmentId - Identifier for the environmentId.
     * @param deploymentNumber - Identifier for the deploymentNumber.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(applicationId, environmentId, deploymentNumber, account, region, partition) {
        var arn = 'arn:${Partition}:appconfig:${Region}:${Account}:application/${ApplicationId}/environment/${EnvironmentId}/deployment/${DeploymentNumber}';
        arn = arn.replace('${ApplicationId}', applicationId);
        arn = arn.replace('${EnvironmentId}', environmentId);
        arn = arn.replace('${DeploymentNumber}', deploymentNumber);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hostedconfigurationversion to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-configuration-and-profile.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param configurationProfileId - Identifier for the configurationProfileId.
     * @param versionNumber - Identifier for the versionNumber.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHostedconfigurationversion(applicationId, configurationProfileId, versionNumber, account, region, partition) {
        var arn = 'arn:${Partition}:appconfig:${Region}:${Account}:application/${ApplicationId}/configurationprofile/${ConfigurationProfileId}/hostedconfigurationversion/${VersionNumber}';
        arn = arn.replace('${ApplicationId}', applicationId);
        arn = arn.replace('${ConfigurationProfileId}', configurationProfileId);
        arn = arn.replace('${VersionNumber}', versionNumber);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appconfig = Appconfig;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzYXBwY29uZmlnLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzYXBwY29uZmlnLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHNDQUE0QztBQUU1Qzs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsd0JBQWU7SUFHNUM7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFSTixrQkFBYSxHQUFHLFdBQVcsQ0FBQztRQWdkekIsb0JBQWUsR0FBb0I7WUFDM0MsT0FBTyxFQUFFO2dCQUNQLG1CQUFtQjtnQkFDbkIsNEJBQTRCO2dCQUM1QiwwQkFBMEI7Z0JBQzFCLG1CQUFtQjtnQkFDbkIsa0NBQWtDO2dCQUNsQyxtQkFBbUI7Z0JBQ25CLDRCQUE0QjtnQkFDNUIsMEJBQTBCO2dCQUMxQixtQkFBbUI7Z0JBQ25CLGtDQUFrQztnQkFDbEMsaUJBQWlCO2dCQUNqQixnQkFBZ0I7Z0JBQ2hCLG1CQUFtQjtnQkFDbkIsNEJBQTRCO2dCQUM1QiwwQkFBMEI7Z0JBQzFCLG1CQUFtQjtnQkFDbkIsdUJBQXVCO2FBQ3hCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLGdCQUFnQjtnQkFDaEIsa0JBQWtCO2dCQUNsQix5QkFBeUI7Z0JBQ3pCLGVBQWU7Z0JBQ2YsdUJBQXVCO2dCQUN2QixnQkFBZ0I7Z0JBQ2hCLCtCQUErQjtnQkFDL0IscUJBQXFCO2FBQ3RCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLGtCQUFrQjtnQkFDbEIsMkJBQTJCO2dCQUMzQiwwQkFBMEI7Z0JBQzFCLGlCQUFpQjtnQkFDakIsa0JBQWtCO2dCQUNsQixpQ0FBaUM7YUFDbEM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQWpmRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLElBQUksQ0FBQyxFQUFFLENBQUMsNENBQTRDLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxJQUFJLENBQUMsRUFBRSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxJQUFJLENBQUMsRUFBRSxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDckQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDRCQUE0QjtRQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBOENEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxhQUFxQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9GLElBQUksR0FBRyxHQUFHLDhFQUE4RSxDQUFDO1FBQ3pGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGtCQUFrQixFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3JELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGFBQWEsQ0FBQyxhQUFxQixFQUFFLGFBQXFCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEgsSUFBSSxHQUFHLEdBQUcsMkdBQTJHLENBQUM7UUFDdEgsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDckQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDckQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksc0JBQXNCLENBQUMsYUFBcUIsRUFBRSxzQkFBOEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN4SSxJQUFJLEdBQUcsR0FBRyw2SEFBNkgsQ0FBQztRQUN4SSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNyRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQywyQkFBMkIsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1FBQ3ZFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CLENBQUMsb0JBQTRCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0csSUFBSSxHQUFHLEdBQUcsNEZBQTRGLENBQUM7UUFDdkcsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMseUJBQXlCLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztRQUNuRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksWUFBWSxDQUFDLGFBQXFCLEVBQUUsYUFBcUIsRUFBRSxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMvSSxJQUFJLEdBQUcsR0FBRywwSUFBMEksQ0FBQztRQUNySixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNyRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNyRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSw0QkFBNEIsQ0FBQyxhQUFxQixFQUFFLHNCQUE4QixFQUFFLGFBQXFCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckssSUFBSSxHQUFHLEdBQUcseUtBQXlLLENBQUM7UUFDcEwsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDckQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsMkJBQTJCLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztRQUN2RSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNyRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBem9CRCw4QkF5b0JDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2FwcGNvbmZpZ10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzYXBwY29uZmlnLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQXBwY29uZmlnIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnYXBwY29uZmlnJztcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthcHBjb25maWddKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2FwcGNvbmZpZy5odG1sKS5cbiAgICpcbiAgICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvciAoc2lkPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2lkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBcHBsaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBcHBsaWNhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6Q3JlYXRlQXBwbGljYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBjb25maWd1cmF0aW9uIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVDb25maWd1cmF0aW9uUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDb25maWd1cmF0aW9uUHJvZmlsZSgpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6Q3JlYXRlQ29uZmlndXJhdGlvblByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBkZXBsb3ltZW50IHN0cmF0ZWd5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRGVwbG95bWVudFN0cmF0ZWd5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURlcGxveW1lbnRTdHJhdGVneSgpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6Q3JlYXRlRGVwbG95bWVudFN0cmF0ZWd5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIGVudmlyb25tZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRW52aXJvbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRW52aXJvbm1lbnQoKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkNyZWF0ZUVudmlyb25tZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgaG9zdGVkIGNvbmZpZ3VyYXRpb24gdmVyc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9uKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpDcmVhdGVIb3N0ZWRDb25maWd1cmF0aW9uVmVyc2lvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFwcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUFwcGxpY2F0aW9uKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpEZWxldGVBcHBsaWNhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNvbmZpZ3VyYXRpb24gcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUNvbmZpZ3VyYXRpb25Qcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNvbmZpZ3VyYXRpb25Qcm9maWxlKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpEZWxldGVDb25maWd1cmF0aW9uUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGRlcGxveW1lbnQgc3RyYXRlZ3lcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVEZXBsb3ltZW50U3RyYXRlZ3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRGVwbG95bWVudFN0cmF0ZWd5KCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpEZWxldGVEZXBsb3ltZW50U3RyYXRlZ3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gZW52aXJvbm1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVFbnZpcm9ubWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVFbnZpcm9ubWVudCgpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6RGVsZXRlRW52aXJvbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBob3N0ZWQgY29uZmlndXJhdGlvbiB2ZXJzaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlSG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb24oKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkRlbGV0ZUhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBkZXRhaWxzIGFib3V0IGFuIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfR2V0QXBwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QXBwbGljYXRpb24oKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkdldEFwcGxpY2F0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBkZXRhaWxzIGFib3V0IGEgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0dldENvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6R2V0Q29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgZGV0YWlscyBhYm91dCBhIGNvbmZpZ3VyYXRpb24gcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0dldENvbmZpZ3VyYXRpb25Qcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldENvbmZpZ3VyYXRpb25Qcm9maWxlKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpHZXRDb25maWd1cmF0aW9uUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgZGV0YWlscyBhYm91dCBhIGRlcGxveW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9HZXREZXBsb3ltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERlcGxveW1lbnQoKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkdldERlcGxveW1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2aWV3IGRldGFpbHMgYWJvdXQgYSBkZXBsb3ltZW50IHN0cmF0ZWd5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfR2V0RGVwbG95bWVudFN0cmF0ZWd5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERlcGxveW1lbnRTdHJhdGVneSgpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6R2V0RGVwbG95bWVudFN0cmF0ZWd5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBkZXRhaWxzIGFib3V0IGFuIGVudmlyb25tZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfR2V0RW52aXJvbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RW52aXJvbm1lbnQoKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkdldEVudmlyb25tZW50Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBkZXRhaWxzIGFib3V0IGEgaG9zdGVkIGNvbmZpZ3VyYXRpb24gdmVyc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfR2V0SG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb24oKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkdldEhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgYXBwbGljYXRpb25zIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfTGlzdEFwcGxpY2F0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QXBwbGljYXRpb25zKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpMaXN0QXBwbGljYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgY29uZmlndXJhdGlvbiBwcm9maWxlcyBmb3IgYW4gYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0xpc3RDb25maWd1cmF0aW9uUHJvZmlsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENvbmZpZ3VyYXRpb25Qcm9maWxlcygpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6TGlzdENvbmZpZ3VyYXRpb25Qcm9maWxlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGRlcGxveW1lbnQgc3RyYXRlZ2llcyBmb3IgeW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RGVwbG95bWVudFN0cmF0ZWdpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERlcGxveW1lbnRTdHJhdGVnaWVzKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpMaXN0RGVwbG95bWVudFN0cmF0ZWdpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBkZXBsb3ltZW50cyBmb3IgYW4gZW52aXJvbm1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0xpc3REZXBsb3ltZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGVwbG95bWVudHMoKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkxpc3REZXBsb3ltZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGVudmlyb25tZW50cyBmb3IgYW4gYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX0xpc3RFbnZpcm9ubWVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEVudmlyb25tZW50cygpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6TGlzdEVudmlyb25tZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGhvc3RlZCBjb25maWd1cmF0aW9uIHZlcnNpb25zIGZvciBhIGNvbmZpZ3VyYXRpb24gcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfTGlzdEhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb25zKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpMaXN0SG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBhIGxpc3Qgb2YgcmVzb3VyY2UgdGFncyBmb3IgYSBzcGVjaWZpZWQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOkxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBpbml0aWF0ZSBhIGRlcGxveW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9TdGFydERlcGxveW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnREZXBsb3ltZW50KCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpTdGFydERlcGxveW1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdG9wIGEgZGVwbG95bWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX1N0b3BEZXBsb3ltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BEZXBsb3ltZW50KCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpTdG9wRGVwbG95bWVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHRhZyBhbiBhcHBjb25maWcgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOlRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdW50YWcgYW4gYXBwY29uZmlnIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcGNvbmZpZy8yMDE5LTEwLTA5L0FQSVJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbW9kaWZ5IGFuIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUFwcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUFwcGxpY2F0aW9uKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpVcGRhdGVBcHBsaWNhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSBhIGNvbmZpZ3VyYXRpb24gcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDb25maWd1cmF0aW9uUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDb25maWd1cmF0aW9uUHJvZmlsZSgpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6VXBkYXRlQ29uZmlndXJhdGlvblByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgYSBkZXBsb3ltZW50IHN0cmF0ZWd5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBjb25maWcvMjAxOS0xMC0wOS9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURlcGxveW1lbnRTdHJhdGVneS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEZXBsb3ltZW50U3RyYXRlZ3koKSB7XG4gICAgdGhpcy50bygnYXBwY29uZmlnOlVwZGF0ZURlcGxveW1lbnRTdHJhdGVneScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSBhbiBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVFbnZpcm9ubWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVFbnZpcm9ubWVudCgpIHtcbiAgICB0aGlzLnRvKCdhcHBjb25maWc6VXBkYXRlRW52aXJvbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2YWxpZGF0ZSBhIGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwY29uZmlnLzIwMTktMTAtMDkvQVBJUmVmZXJlbmNlL0FQSV9WYWxpZGF0ZUNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVmFsaWRhdGVDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ2FwcGNvbmZpZzpWYWxpZGF0ZUNvbmZpZ3VyYXRpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQ3JlYXRlQXBwbGljYXRpb25cIixcbiAgICAgIFwiQ3JlYXRlQ29uZmlndXJhdGlvblByb2ZpbGVcIixcbiAgICAgIFwiQ3JlYXRlRGVwbG95bWVudFN0cmF0ZWd5XCIsXG4gICAgICBcIkNyZWF0ZUVudmlyb25tZW50XCIsXG4gICAgICBcIkNyZWF0ZUhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9uXCIsXG4gICAgICBcIkRlbGV0ZUFwcGxpY2F0aW9uXCIsXG4gICAgICBcIkRlbGV0ZUNvbmZpZ3VyYXRpb25Qcm9maWxlXCIsXG4gICAgICBcIkRlbGV0ZURlcGxveW1lbnRTdHJhdGVneVwiLFxuICAgICAgXCJEZWxldGVFbnZpcm9ubWVudFwiLFxuICAgICAgXCJEZWxldGVIb3N0ZWRDb25maWd1cmF0aW9uVmVyc2lvblwiLFxuICAgICAgXCJTdGFydERlcGxveW1lbnRcIixcbiAgICAgIFwiU3RvcERlcGxveW1lbnRcIixcbiAgICAgIFwiVXBkYXRlQXBwbGljYXRpb25cIixcbiAgICAgIFwiVXBkYXRlQ29uZmlndXJhdGlvblByb2ZpbGVcIixcbiAgICAgIFwiVXBkYXRlRGVwbG95bWVudFN0cmF0ZWd5XCIsXG4gICAgICBcIlVwZGF0ZUVudmlyb25tZW50XCIsXG4gICAgICBcIlZhbGlkYXRlQ29uZmlndXJhdGlvblwiXG4gICAgXSxcbiAgICBcIlJlYWRcIjogW1xuICAgICAgXCJHZXRBcHBsaWNhdGlvblwiLFxuICAgICAgXCJHZXRDb25maWd1cmF0aW9uXCIsXG4gICAgICBcIkdldENvbmZpZ3VyYXRpb25Qcm9maWxlXCIsXG4gICAgICBcIkdldERlcGxveW1lbnRcIixcbiAgICAgIFwiR2V0RGVwbG95bWVudFN0cmF0ZWd5XCIsXG4gICAgICBcIkdldEVudmlyb25tZW50XCIsXG4gICAgICBcIkdldEhvc3RlZENvbmZpZ3VyYXRpb25WZXJzaW9uXCIsXG4gICAgICBcIkxpc3RUYWdzRm9yUmVzb3VyY2VcIlxuICAgIF0sXG4gICAgXCJMaXN0XCI6IFtcbiAgICAgIFwiTGlzdEFwcGxpY2F0aW9uc1wiLFxuICAgICAgXCJMaXN0Q29uZmlndXJhdGlvblByb2ZpbGVzXCIsXG4gICAgICBcIkxpc3REZXBsb3ltZW50U3RyYXRlZ2llc1wiLFxuICAgICAgXCJMaXN0RGVwbG95bWVudHNcIixcbiAgICAgIFwiTGlzdEVudmlyb25tZW50c1wiLFxuICAgICAgXCJMaXN0SG9zdGVkQ29uZmlndXJhdGlvblZlcnNpb25zXCJcbiAgICBdLFxuICAgIFwiVGFnZ2luZ1wiOiBbXG4gICAgICBcIlRhZ1Jlc291cmNlXCIsXG4gICAgICBcIlVudGFnUmVzb3VyY2VcIlxuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBwbGljYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3lzdGVtcy1tYW5hZ2VyL2xhdGVzdC91c2VyZ3VpZGUvYXBwY29uZmlnLWNyZWF0aW5nLWFwcGxpY2F0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcGxpY2F0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwbGljYXRpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BcHBsaWNhdGlvbihhcHBsaWNhdGlvbklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmFwcGNvbmZpZzoke1JlZ2lvbn06JHtBY2NvdW50fTphcHBsaWNhdGlvbi8ke0FwcGxpY2F0aW9uSWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBcHBsaWNhdGlvbklkfScsIGFwcGxpY2F0aW9uSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGVudmlyb25tZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL2FwcGNvbmZpZy1jcmVhdGluZy1lbnZpcm9ubWVudC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBhcHBsaWNhdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcGxpY2F0aW9uSWQuXG4gICAqIEBwYXJhbSBlbnZpcm9ubWVudElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGVudmlyb25tZW50SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRW52aXJvbm1lbnQoYXBwbGljYXRpb25JZDogc3RyaW5nLCBlbnZpcm9ubWVudElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmFwcGNvbmZpZzoke1JlZ2lvbn06JHtBY2NvdW50fTphcHBsaWNhdGlvbi8ke0FwcGxpY2F0aW9uSWR9L2Vudmlyb25tZW50LyR7RW52aXJvbm1lbnRJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FwcGxpY2F0aW9uSWR9JywgYXBwbGljYXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7RW52aXJvbm1lbnRJZH0nLCBlbnZpcm9ubWVudElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjb25maWd1cmF0aW9ucHJvZmlsZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zeXN0ZW1zLW1hbmFnZXIvbGF0ZXN0L3VzZXJndWlkZS9hcHBjb25maWctY3JlYXRpbmctY29uZmlndXJhdGlvbi1hbmQtcHJvZmlsZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBhcHBsaWNhdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcGxpY2F0aW9uSWQuXG4gICAqIEBwYXJhbSBjb25maWd1cmF0aW9uUHJvZmlsZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNvbmZpZ3VyYXRpb25Qcm9maWxlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ29uZmlndXJhdGlvbnByb2ZpbGUoYXBwbGljYXRpb25JZDogc3RyaW5nLCBjb25maWd1cmF0aW9uUHJvZmlsZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmFwcGNvbmZpZzoke1JlZ2lvbn06JHtBY2NvdW50fTphcHBsaWNhdGlvbi8ke0FwcGxpY2F0aW9uSWR9L2NvbmZpZ3VyYXRpb25wcm9maWxlLyR7Q29uZmlndXJhdGlvblByb2ZpbGVJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FwcGxpY2F0aW9uSWR9JywgYXBwbGljYXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Q29uZmlndXJhdGlvblByb2ZpbGVJZH0nLCBjb25maWd1cmF0aW9uUHJvZmlsZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkZXBsb3ltZW50c3RyYXRlZ3kgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3lzdGVtcy1tYW5hZ2VyL2xhdGVzdC91c2VyZ3VpZGUvYXBwY29uZmlnLWNyZWF0aW5nLWRlcGxveW1lbnQtc3RyYXRlZ3kuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGVwbG95bWVudFN0cmF0ZWd5SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZGVwbG95bWVudFN0cmF0ZWd5SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGVwbG95bWVudHN0cmF0ZWd5KGRlcGxveW1lbnRTdHJhdGVneUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmFwcGNvbmZpZzoke1JlZ2lvbn06JHtBY2NvdW50fTpkZXBsb3ltZW50c3RyYXRlZ3kvJHtEZXBsb3ltZW50U3RyYXRlZ3lJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0RlcGxveW1lbnRTdHJhdGVneUlkfScsIGRlcGxveW1lbnRTdHJhdGVneUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkZXBsb3ltZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL2FwcGNvbmZpZy1kZXBsb3lpbmcuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBwbGljYXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBhcHBsaWNhdGlvbklkLlxuICAgKiBAcGFyYW0gZW52aXJvbm1lbnRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBlbnZpcm9ubWVudElkLlxuICAgKiBAcGFyYW0gZGVwbG95bWVudE51bWJlciAtIElkZW50aWZpZXIgZm9yIHRoZSBkZXBsb3ltZW50TnVtYmVyLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRlcGxveW1lbnQoYXBwbGljYXRpb25JZDogc3RyaW5nLCBlbnZpcm9ubWVudElkOiBzdHJpbmcsIGRlcGxveW1lbnROdW1iZXI6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06YXBwY29uZmlnOiR7UmVnaW9ufToke0FjY291bnR9OmFwcGxpY2F0aW9uLyR7QXBwbGljYXRpb25JZH0vZW52aXJvbm1lbnQvJHtFbnZpcm9ubWVudElkfS9kZXBsb3ltZW50LyR7RGVwbG95bWVudE51bWJlcn0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FwcGxpY2F0aW9uSWR9JywgYXBwbGljYXRpb25JZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7RW52aXJvbm1lbnRJZH0nLCBlbnZpcm9ubWVudElkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtEZXBsb3ltZW50TnVtYmVyfScsIGRlcGxveW1lbnROdW1iZXIpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGhvc3RlZGNvbmZpZ3VyYXRpb252ZXJzaW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL2FwcGNvbmZpZy1jcmVhdGluZy1jb25maWd1cmF0aW9uLWFuZC1wcm9maWxlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcGxpY2F0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwbGljYXRpb25JZC5cbiAgICogQHBhcmFtIGNvbmZpZ3VyYXRpb25Qcm9maWxlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY29uZmlndXJhdGlvblByb2ZpbGVJZC5cbiAgICogQHBhcmFtIHZlcnNpb25OdW1iZXIgLSBJZGVudGlmaWVyIGZvciB0aGUgdmVyc2lvbk51bWJlci5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkhvc3RlZGNvbmZpZ3VyYXRpb252ZXJzaW9uKGFwcGxpY2F0aW9uSWQ6IHN0cmluZywgY29uZmlndXJhdGlvblByb2ZpbGVJZDogc3RyaW5nLCB2ZXJzaW9uTnVtYmVyOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmFwcGNvbmZpZzoke1JlZ2lvbn06JHtBY2NvdW50fTphcHBsaWNhdGlvbi8ke0FwcGxpY2F0aW9uSWR9L2NvbmZpZ3VyYXRpb25wcm9maWxlLyR7Q29uZmlndXJhdGlvblByb2ZpbGVJZH0vaG9zdGVkY29uZmlndXJhdGlvbnZlcnNpb24vJHtWZXJzaW9uTnVtYmVyfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QXBwbGljYXRpb25JZH0nLCBhcHBsaWNhdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtDb25maWd1cmF0aW9uUHJvZmlsZUlkfScsIGNvbmZpZ3VyYXRpb25Qcm9maWxlSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1ZlcnNpb25OdW1iZXJ9JywgdmVyc2lvbk51bWJlcik7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==