"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Budgets = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [budgets](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbudgetservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Budgets extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [budgets](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbudgetservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'budgets';
        this.accessLevelList = {
            "Write": [
                "CreateBudgetAction",
                "DeleteBudgetAction",
                "ExecuteBudgetAction",
                "ModifyBudget",
                "UpdateBudgetAction"
            ],
            "Read": [
                "DescribeBudgetAction",
                "DescribeBudgetActionHistories",
                "DescribeBudgetActionsForAccount",
                "DescribeBudgetActionsForBudget",
                "ViewBudget"
            ]
        };
    }
    /**
     * Grants permissions to create and define a response that you can configure to execute once your budget has exceeded a specific budget threshold.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toCreateBudgetAction() {
        this.to('budgets:CreateBudgetAction');
        return this;
    }
    /**
     * Grants permissions to delete an action that is associated with a specific budget.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toDeleteBudgetAction() {
        this.to('budgets:DeleteBudgetAction');
        return this;
    }
    /**
     * Grants permissions to retrieve the details of specific budget action associated with a budget.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toDescribeBudgetAction() {
        this.to('budgets:DescribeBudgetAction');
        return this;
    }
    /**
     * Grants permissions to retrieve a historical view of the budget actions statuses associated with a particular budget action. These status include statues such as 'Standby', 'Pending' and 'Executed'.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toDescribeBudgetActionHistories() {
        this.to('budgets:DescribeBudgetActionHistories');
        return this;
    }
    /**
     * Grants permissions to retrieve the details of all of the budget actions associated with your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toDescribeBudgetActionsForAccount() {
        this.to('budgets:DescribeBudgetActionsForAccount');
        return this;
    }
    /**
     * Grants permissions to retrieve the details of all of the budget actions associated with a budget.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toDescribeBudgetActionsForBudget() {
        this.to('budgets:DescribeBudgetActionsForBudget');
        return this;
    }
    /**
     * Grants permissions to initiate a pending budget action as well as reverse a previously executed budget action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toExecuteBudgetAction() {
        this.to('budgets:ExecuteBudgetAction');
        return this;
    }
    /**
     * Grants permissions to modify budgets and budget details
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toModifyBudget() {
        this.to('budgets:ModifyBudget');
        return this;
    }
    /**
     * Grants permissions to update the details of a specific budget action associated with a budget.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toUpdateBudgetAction() {
        this.to('budgets:UpdateBudgetAction');
        return this;
    }
    /**
     * Grants permissions to view budgets and budget details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toViewBudget() {
        this.to('budgets:ViewBudget');
        return this;
    }
    /**
     * Adds a resource of type budget to the statement
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-managing-costs.html
     *
     * @param budgetName - Identifier for the budgetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBudget(budgetName, account, partition) {
        var arn = 'arn:${Partition}:budgets::${Account}:budget/${BudgetName}';
        arn = arn.replace('${BudgetName}', budgetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type budgetAction to the statement
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-managing-costs.html
     *
     * @param budgetName - Identifier for the budgetName.
     * @param actionId - Identifier for the actionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBudgetAction(budgetName, actionId, account, partition) {
        var arn = 'arn:${Partition}:budgets::${Account}:budget/${BudgetName}/action/${ActionId}';
        arn = arn.replace('${BudgetName}', budgetName);
        arn = arn.replace('${ActionId}', actionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Budgets = Budgets;
//# sourceMappingURL=data:application/json;base64,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