"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Acm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [acm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Acm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [acm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'acm';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToCertificate",
                "RemoveTagsFromCertificate"
            ],
            "Write": [
                "DeleteCertificate",
                "ImportCertificate",
                "RenewCertificate",
                "RequestCertificate",
                "ResendValidationEmail",
                "UpdateCertificateOptions"
            ],
            "Read": [
                "DescribeCertificate",
                "ExportCertificate",
                "GetCertificate",
                "ListTagsForCertificate"
            ],
            "List": [
                "ListCertificates"
            ]
        };
    }
    /**
     * Adds one or more tags to a certificate.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_AddTagsToCertificate.html
     */
    toAddTagsToCertificate() {
        this.to('acm:AddTagsToCertificate');
        return this;
    }
    /**
     * Deletes a certificate and its associated private key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        this.to('acm:DeleteCertificate');
        return this;
    }
    /**
     * Returns a list of the fields contained in the specified certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        this.to('acm:DescribeCertificate');
        return this;
    }
    /**
     * Exports a private certificate issued by a private certificate authority (CA) for use anywhere.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ExportCertificate.html
     */
    toExportCertificate() {
        this.to('acm:ExportCertificate');
        return this;
    }
    /**
     * Retrieves a certificate and certificate chain for the certificate specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_GetCertificate.html
     */
    toGetCertificate() {
        this.to('acm:GetCertificate');
        return this;
    }
    /**
     * Imports a 3rd party SSL/TLS certificate into AWS Certificate Manager (ACM).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ImportCertificate.html
     */
    toImportCertificate() {
        this.to('acm:ImportCertificate');
        return this;
    }
    /**
     * Retrieves a list of the certificate ARNs and the domain name for each ARN.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ListCertificates.html
     */
    toListCertificates() {
        this.to('acm:ListCertificates');
        return this;
    }
    /**
     * Lists the tags that have been applied to the certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ListTagsForCertificate.html
     */
    toListTagsForCertificate() {
        this.to('acm:ListTagsForCertificate');
        return this;
    }
    /**
     * Remove one or more tags from a certificate. A tag consists of a key-value pair
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RemoveTagsFromCertificate.html
     */
    toRemoveTagsFromCertificate() {
        this.to('acm:RemoveTagsFromCertificate');
        return this;
    }
    /**
     * Renews an eligable private certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RenewCertificate.html
     */
    toRenewCertificate() {
        this.to('acm:RenewCertificate');
        return this;
    }
    /**
     * Requests a public or private certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RequestCertificate.html
     */
    toRequestCertificate() {
        this.to('acm:RequestCertificate');
        return this;
    }
    /**
     * Resends an email to request domain ownership validation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ResendValidationEmail.html
     */
    toResendValidationEmail() {
        this.to('acm:ResendValidationEmail');
        return this;
    }
    /**
     * Updates a certificate. Use to specify whether to opt in to or out of certificate transparency logging.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_UpdateCertificateOptions.html
     */
    toUpdateCertificateOptions() {
        this.to('acm:UpdateCertificateOptions');
        return this;
    }
    /**
     * Adds a resource of type certificate to the statement
     *
     * https://docs.aws.amazon.com/acm/latest/userguide/authen-overview.html#acm-resources-operations
     *
     * @param certificateId - Identifier for the certificateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificate(certificateId, account, region, partition) {
        var arn = 'arn:${Partition}:acm:${Region}:${Account}:certificate/${CertificateId}';
        arn = arn.replace('${CertificateId}', certificateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Acm = Acm;
//# sourceMappingURL=data:application/json;base64,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