"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcmPca = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AcmPca extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'acm-pca';
        this.accessLevelList = {
            "Tagging": [
                "CreateCertificateAuthority",
                "TagCertificateAuthority",
                "UntagCertificateAuthority"
            ],
            "Write": [
                "CreateCertificateAuthorityAuditReport",
                "DeleteCertificateAuthority",
                "ImportCertificateAuthorityCertificate",
                "IssueCertificate",
                "RestoreCertificateAuthority",
                "RevokeCertificate",
                "UpdateCertificateAuthority"
            ],
            "Permissions management": [
                "CreatePermission",
                "DeletePermission",
                "DeletePolicy",
                "PutPolicy"
            ],
            "Read": [
                "DescribeCertificateAuthority",
                "DescribeCertificateAuthorityAuditReport",
                "GetCertificate",
                "GetCertificateAuthorityCertificate",
                "GetCertificateAuthorityCsr",
                "GetPolicy",
                "ListPermissions",
                "ListTags"
            ],
            "List": [
                "ListCertificateAuthorities"
            ]
        };
    }
    /**
     * Creates an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html
     */
    toCreateCertificateAuthority() {
        this.to('acm-pca:CreateCertificateAuthority');
        return this;
    }
    /**
     * Creates an audit report for an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html
     */
    toCreateCertificateAuthorityAuditReport() {
        this.to('acm-pca:CreateCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Creates a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html
     */
    toCreatePermission() {
        this.to('acm-pca:CreatePermission');
        return this;
    }
    /**
     * Deletes an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html
     */
    toDeleteCertificateAuthority() {
        this.to('acm-pca:DeleteCertificateAuthority');
        return this;
    }
    /**
     * Deletes a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html
     */
    toDeletePermission() {
        this.to('acm-pca:DeletePermission');
        return this;
    }
    /**
     * Deletes the policy for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.to('acm-pca:DeletePolicy');
        return this;
    }
    /**
     * Returns a list of the configuration and status fields contained in the specified ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html
     */
    toDescribeCertificateAuthority() {
        this.to('acm-pca:DescribeCertificateAuthority');
        return this;
    }
    /**
     * Returns the status and information about an ACM Private CA audit report.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html
     */
    toDescribeCertificateAuthorityAuditReport() {
        this.to('acm-pca:DescribeCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html
     */
    toGetCertificate() {
        this.to('acm-pca:GetCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html
     */
    toGetCertificateAuthorityCertificate() {
        this.to('acm-pca:GetCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html
     */
    toGetCertificateAuthorityCsr() {
        this.to('acm-pca:GetCertificateAuthorityCsr');
        return this;
    }
    /**
     * Retrieves the policy on an ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        this.to('acm-pca:GetPolicy');
        return this;
    }
    /**
     * Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html
     */
    toImportCertificateAuthorityCertificate() {
        this.to('acm-pca:ImportCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Issues an ACM Private CA certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTemplateArn()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html
     */
    toIssueCertificate() {
        this.to('acm-pca:IssueCertificate');
        return this;
    }
    /**
     * Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html
     */
    toListCertificateAuthorities() {
        this.to('acm-pca:ListCertificateAuthorities');
        return this;
    }
    /**
     * Lists the permissions that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html
     */
    toListPermissions() {
        this.to('acm-pca:ListPermissions');
        return this;
    }
    /**
     * Lists the tags that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.to('acm-pca:ListTags');
        return this;
    }
    /**
     * Puts a policy on an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html
     */
    toPutPolicy() {
        this.to('acm-pca:PutPolicy');
        return this;
    }
    /**
     * Restores an ACM Private CA from the deleted state to the state it was in when deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html
     */
    toRestoreCertificateAuthority() {
        this.to('acm-pca:RestoreCertificateAuthority');
        return this;
    }
    /**
     * Revokes a certificate issued by an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html
     */
    toRevokeCertificate() {
        this.to('acm-pca:RevokeCertificate');
        return this;
    }
    /**
     * Adds one or more tags to an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html
     */
    toTagCertificateAuthority() {
        this.to('acm-pca:TagCertificateAuthority');
        return this;
    }
    /**
     * Remove one or more tags from an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html
     */
    toUntagCertificateAuthority() {
        this.to('acm-pca:UntagCertificateAuthority');
        return this;
    }
    /**
     * Updates the configuration of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html
     */
    toUpdateCertificateAuthority() {
        this.to('acm-pca:UpdateCertificateAuthority');
        return this;
    }
    /**
     * Adds a resource of type certificate-authority to the statement
     *
     * https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations
     *
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificateAuthority(certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}';
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters issue certificate requests based on the presence of TemplateArn in the request.
     *
     * Applies to actions:
     * - .toIssueCertificate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateArn(value, operator) {
        return this.if(`acm-pca:TemplateArn`, value, operator || 'StringLike');
    }
}
exports.AcmPca = AcmPca;
//# sourceMappingURL=data:application/json;base64,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