"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudtrail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudtrail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudtrail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudtrail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudtrail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudtrail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudtrail';
        this.accessLevelList = {
            "Tagging": [
                "AddTags",
                "RemoveTags"
            ],
            "Write": [
                "CreateTrail",
                "DeleteTrail",
                "PutEventSelectors",
                "PutInsightSelectors",
                "StartLogging",
                "StopLogging",
                "UpdateTrail"
            ],
            "Read": [
                "DescribeTrails",
                "GetEventSelectors",
                "GetInsightSelectors",
                "GetTrail",
                "GetTrailStatus",
                "ListPublicKeys",
                "ListTags",
                "LookupEvents"
            ],
            "List": [
                "ListTrails"
            ]
        };
    }
    /**
     * Grants permission to add one or more tags to a trail, up to a limit of 10
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        this.to('cloudtrail:AddTags');
        return this;
    }
    /**
     * Grants permission to create a trail that specifies the settings for delivery of log data to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_CreateTrail.html
     */
    toCreateTrail() {
        this.to('cloudtrail:CreateTrail');
        return this;
    }
    /**
     * Grants permission to delete a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DeleteTrail.html
     */
    toDeleteTrail() {
        this.to('cloudtrail:DeleteTrail');
        return this;
    }
    /**
     * Grants permission to list settings for the trails associated with the current region for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_DescribeTrails.html
     */
    toDescribeTrails() {
        this.to('cloudtrail:DescribeTrails');
        return this;
    }
    /**
     * Grants permission to list settings for event selectors configured for a trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetEventSelectors.html
     */
    toGetEventSelectors() {
        this.to('cloudtrail:GetEventSelectors');
        return this;
    }
    /**
     * Grants permission to list CloudTrail Insights selectors that are configured for a trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetInsightSelectors.html
     */
    toGetInsightSelectors() {
        this.to('cloudtrail:GetInsightSelectors');
        return this;
    }
    /**
     * Grants permission to list settings for the trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrail.html
     */
    toGetTrail() {
        this.to('cloudtrail:GetTrail');
        return this;
    }
    /**
     * Grants permission to retrieve a JSON-formatted list of information about the specified trail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_GetTrailStatus.html
     */
    toGetTrailStatus() {
        this.to('cloudtrail:GetTrailStatus');
        return this;
    }
    /**
     * Grants permission to list the public keys whose private keys were used to sign trail digest files within a specified time range
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys() {
        this.to('cloudtrail:ListPublicKeys');
        return this;
    }
    /**
     * Grants permission to list the tags for trails in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.to('cloudtrail:ListTags');
        return this;
    }
    /**
     * Grants permission to list trails associated with the current region for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_ListTrails.html
     */
    toListTrails() {
        this.to('cloudtrail:ListTrails');
        return this;
    }
    /**
     * Grants permission to look up API activity events captured by CloudTrail that create, update, or delete resources in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html
     */
    toLookupEvents() {
        this.to('cloudtrail:LookupEvents');
        return this;
    }
    /**
     * Grants permission to create and update event selectors for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutEventSelectors.html
     */
    toPutEventSelectors() {
        this.to('cloudtrail:PutEventSelectors');
        return this;
    }
    /**
     * Grants permission to create and update CloudTrail Insights selectors for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_PutInsightSelectors.html
     */
    toPutInsightSelectors() {
        this.to('cloudtrail:PutInsightSelectors');
        return this;
    }
    /**
     * Grants permission to remove tags from a trail
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        this.to('cloudtrail:RemoveTags');
        return this;
    }
    /**
     * Grants permission to start the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StartLogging.html
     */
    toStartLogging() {
        this.to('cloudtrail:StartLogging');
        return this;
    }
    /**
     * Grants permission to stop the recording of AWS API calls and log file delivery for a trail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_StopLogging.html
     */
    toStopLogging() {
        this.to('cloudtrail:StopLogging');
        return this;
    }
    /**
     * Grants permission to update the settings that specify delivery of log files
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_UpdateTrail.html
     */
    toUpdateTrail() {
        this.to('cloudtrail:UpdateTrail');
        return this;
    }
    /**
     * Adds a resource of type trail to the statement
     *
     * https://docs.aws.amazon.com/awscloudtrail/latest/userguide/how-cloudtrail-works.html
     *
     * @param trailName - Identifier for the trailName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrail(trailName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudtrail:${Region}:${Account}:trail/${TrailName}';
        arn = arn.replace('${TrailName}', trailName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudtrail = Cloudtrail;
//# sourceMappingURL=data:application/json;base64,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