"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ds = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ds extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ds';
        this.accessLevelList = {
            "Write": [
                "AcceptSharedDirectory",
                "AddIpRoutes",
                "AuthorizeApplication",
                "CancelSchemaExtension",
                "CreateAlias",
                "CreateComputer",
                "CreateConditionalForwarder",
                "CreateLogSubscription",
                "CreateSnapshot",
                "CreateTrust",
                "DeleteConditionalForwarder",
                "DeleteDirectory",
                "DeleteLogSubscription",
                "DeleteSnapshot",
                "DeleteTrust",
                "DeregisterCertificate",
                "DeregisterEventTopic",
                "DisableLDAPS",
                "DisableRadius",
                "DisableSso",
                "EnableLDAPS",
                "EnableRadius",
                "EnableSso",
                "RegisterCertificate",
                "RegisterEventTopic",
                "RejectSharedDirectory",
                "RemoveIpRoutes",
                "ResetUserPassword",
                "RestoreFromSnapshot",
                "ShareDirectory",
                "StartSchemaExtension",
                "UnauthorizeApplication",
                "UnshareDirectory",
                "UpdateConditionalForwarder",
                "UpdateNumberOfDomainControllers",
                "UpdateRadius",
                "UpdateTrust"
            ],
            "Tagging": [
                "AddTagsToResource",
                "ConnectDirectory",
                "CreateDirectory",
                "CreateIdentityPoolDirectory",
                "CreateMicrosoftAD",
                "RemoveTagsFromResource"
            ],
            "Read": [
                "CheckAlias",
                "DescribeCertificate",
                "DescribeConditionalForwarders",
                "DescribeDomainControllers",
                "DescribeEventTopics",
                "DescribeLDAPSSettings",
                "DescribeSharedDirectories",
                "DescribeSnapshots",
                "DescribeTrusts",
                "GetAuthorizedApplicationDetails",
                "GetDirectoryLimits",
                "GetSnapshotLimits",
                "ListAuthorizedApplications",
                "ListIpRoutes",
                "ListLogSubscriptions",
                "ListTagsForResource",
                "VerifyTrust"
            ],
            "List": [
                "DescribeDirectories",
                "ListCertificates",
                "ListSchemaExtensions"
            ]
        };
    }
    /**
     * Accepts a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html
     */
    toAcceptSharedDirectory() {
        this.to('ds:AcceptSharedDirectory');
        return this;
    }
    /**
     * Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:DescribeSecurityGroups
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html
     */
    toAddIpRoutes() {
        this.to('ds:AddIpRoutes');
        return this;
    }
    /**
     * Adds or overwrites one or more tags for the specified Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('ds:AddTagsToResource');
        return this;
    }
    /**
     * Authorizes an application for your AWS Directory.
     *
     * Access Level: Write
     */
    toAuthorizeApplication() {
        this.to('ds:AuthorizeApplication');
        return this;
    }
    /**
     * Cancels an in-progress schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html
     */
    toCancelSchemaExtension() {
        this.to('ds:CancelSchemaExtension');
        return this;
    }
    /**
     * Verifies that the alias is available for use.
     *
     * Access Level: Read
     */
    toCheckAlias() {
        this.to('ds:CheckAlias');
        return this;
    }
    /**
     * Creates an AD Connector to connect to an on-premises directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html
     */
    toConnectDirectory() {
        this.to('ds:ConnectDirectory');
        return this;
    }
    /**
     * Creates an alias for a directory and assigns the alias to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html
     */
    toCreateAlias() {
        this.to('ds:CreateAlias');
        return this;
    }
    /**
     * Creates a computer account in the specified directory, and joins the computer to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html
     */
    toCreateComputer() {
        this.to('ds:CreateComputer');
        return this;
    }
    /**
     * Creates a conditional forwarder associated with your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html
     */
    toCreateConditionalForwarder() {
        this.to('ds:CreateConditionalForwarder');
        return this;
    }
    /**
     * Creates a Simple AD directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html
     */
    toCreateDirectory() {
        this.to('ds:CreateDirectory');
        return this;
    }
    /**
     * Creates a IdentityPool Directory in the AWS cloud.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     */
    toCreateIdentityPoolDirectory() {
        this.to('ds:CreateIdentityPoolDirectory');
        return this;
    }
    /**
     * Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html
     */
    toCreateLogSubscription() {
        this.to('ds:CreateLogSubscription');
        return this;
    }
    /**
     * Creates a Microsoft AD in the AWS cloud.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html
     */
    toCreateMicrosoftAD() {
        this.to('ds:CreateMicrosoftAD');
        return this;
    }
    /**
     * Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.to('ds:CreateSnapshot');
        return this;
    }
    /**
     * Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html
     */
    toCreateTrust() {
        this.to('ds:CreateTrust');
        return this;
    }
    /**
     * Deletes a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html
     */
    toDeleteConditionalForwarder() {
        this.to('ds:DeleteConditionalForwarder');
        return this;
    }
    /**
     * Deletes an AWS Directory Service directory.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteSecurityGroup
     * - ec2:DescribeNetworkInterfaces
     * - ec2:RevokeSecurityGroupEgress
     * - ec2:RevokeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html
     */
    toDeleteDirectory() {
        this.to('ds:DeleteDirectory');
        return this;
    }
    /**
     * Deletes the specified log subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html
     */
    toDeleteLogSubscription() {
        this.to('ds:DeleteLogSubscription');
        return this;
    }
    /**
     * Deletes a directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.to('ds:DeleteSnapshot');
        return this;
    }
    /**
     * Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html
     */
    toDeleteTrust() {
        this.to('ds:DeleteTrust');
        return this;
    }
    /**
     * Deletes from the system the certificate that was registered for a secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html
     */
    toDeregisterCertificate() {
        this.to('ds:DeregisterCertificate');
        return this;
    }
    /**
     * Removes the specified directory as a publisher to the specified SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html
     */
    toDeregisterEventTopic() {
        this.to('ds:DeregisterEventTopic');
        return this;
    }
    /**
     * Displays information about the certificate registered for a secured LDAP connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        this.to('ds:DescribeCertificate');
        return this;
    }
    /**
     * Obtains information about the conditional forwarders for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html
     */
    toDescribeConditionalForwarders() {
        this.to('ds:DescribeConditionalForwarders');
        return this;
    }
    /**
     * Obtains information about the directories that belong to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html
     */
    toDescribeDirectories() {
        this.to('ds:DescribeDirectories');
        return this;
    }
    /**
     * Provides information about any domain controllers in your directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html
     */
    toDescribeDomainControllers() {
        this.to('ds:DescribeDomainControllers');
        return this;
    }
    /**
     * Obtains information about which SNS topics receive status messages from the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html
     */
    toDescribeEventTopics() {
        this.to('ds:DescribeEventTopics');
        return this;
    }
    /**
     * Describes the status of LDAP security for the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html
     */
    toDescribeLDAPSSettings() {
        this.to('ds:DescribeLDAPSSettings');
        return this;
    }
    /**
     * Returns the shared directories in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html
     */
    toDescribeSharedDirectories() {
        this.to('ds:DescribeSharedDirectories');
        return this;
    }
    /**
     * Obtains information about the directory snapshots that belong to this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.to('ds:DescribeSnapshots');
        return this;
    }
    /**
     * Obtains information about the trust relationships for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html
     */
    toDescribeTrusts() {
        this.to('ds:DescribeTrusts');
        return this;
    }
    /**
     * Deactivates LDAP secure calls for the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html
     */
    toDisableLDAPS() {
        this.to('ds:DisableLDAPS');
        return this;
    }
    /**
     * Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html
     */
    toDisableRadius() {
        this.to('ds:DisableRadius');
        return this;
    }
    /**
     * Disables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html
     */
    toDisableSso() {
        this.to('ds:DisableSso');
        return this;
    }
    /**
     * Activates the switch for the specific directory to always use LDAP secure calls.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html
     */
    toEnableLDAPS() {
        this.to('ds:EnableLDAPS');
        return this;
    }
    /**
     * Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html
     */
    toEnableRadius() {
        this.to('ds:EnableRadius');
        return this;
    }
    /**
     * Enables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html
     */
    toEnableSso() {
        this.to('ds:EnableSso');
        return this;
    }
    /**
     *
     *
     * Access Level: Read
     */
    toGetAuthorizedApplicationDetails() {
        this.to('ds:GetAuthorizedApplicationDetails');
        return this;
    }
    /**
     * Obtains directory limit information for the current region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html
     */
    toGetDirectoryLimits() {
        this.to('ds:GetDirectoryLimits');
        return this;
    }
    /**
     * Obtains the manual snapshot limits for a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html
     */
    toGetSnapshotLimits() {
        this.to('ds:GetSnapshotLimits');
        return this;
    }
    /**
     * Obtains the aws applications authorized for a directory.
     *
     * Access Level: Read
     */
    toListAuthorizedApplications() {
        this.to('ds:ListAuthorizedApplications');
        return this;
    }
    /**
     * For the specified directory, lists all the certificates registered for a secured LDAP connection.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html
     */
    toListCertificates() {
        this.to('ds:ListCertificates');
        return this;
    }
    /**
     * Lists the address blocks that you have added to a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html
     */
    toListIpRoutes() {
        this.to('ds:ListIpRoutes');
        return this;
    }
    /**
     * Lists the active log subscriptions for the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html
     */
    toListLogSubscriptions() {
        this.to('ds:ListLogSubscriptions');
        return this;
    }
    /**
     * Lists all schema extensions applied to a Microsoft AD Directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html
     */
    toListSchemaExtensions() {
        this.to('ds:ListSchemaExtensions');
        return this;
    }
    /**
     * Lists all tags on an Amazon Directory Services directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('ds:ListTagsForResource');
        return this;
    }
    /**
     * Registers a certificate for secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html
     */
    toRegisterCertificate() {
        this.to('ds:RegisterCertificate');
        return this;
    }
    /**
     * Associates a directory with an SNS topic.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sns:GetTopicAttributes
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html
     */
    toRegisterEventTopic() {
        this.to('ds:RegisterEventTopic');
        return this;
    }
    /**
     * Rejects a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html
     */
    toRejectSharedDirectory() {
        this.to('ds:RejectSharedDirectory');
        return this;
    }
    /**
     * Removes IP address blocks from a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html
     */
    toRemoveIpRoutes() {
        this.to('ds:RemoveIpRoutes');
        return this;
    }
    /**
     * Removes tags from an Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DeleteTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('ds:RemoveTagsFromResource');
        return this;
    }
    /**
     * Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html
     */
    toResetUserPassword() {
        this.to('ds:ResetUserPassword');
        return this;
    }
    /**
     * Restores a directory using an existing directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html
     */
    toRestoreFromSnapshot() {
        this.to('ds:RestoreFromSnapshot');
        return this;
    }
    /**
     * Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html
     */
    toShareDirectory() {
        this.to('ds:ShareDirectory');
        return this;
    }
    /**
     * Applies a schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html
     */
    toStartSchemaExtension() {
        this.to('ds:StartSchemaExtension');
        return this;
    }
    /**
     * Unauthorizes an application from your AWS Directory.
     *
     * Access Level: Write
     */
    toUnauthorizeApplication() {
        this.to('ds:UnauthorizeApplication');
        return this;
    }
    /**
     * Stops the directory sharing between the directory owner and consumer accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html
     */
    toUnshareDirectory() {
        this.to('ds:UnshareDirectory');
        return this;
    }
    /**
     * Updates a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html
     */
    toUpdateConditionalForwarder() {
        this.to('ds:UpdateConditionalForwarder');
        return this;
    }
    /**
     * Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html
     */
    toUpdateNumberOfDomainControllers() {
        this.to('ds:UpdateNumberOfDomainControllers');
        return this;
    }
    /**
     * Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html
     */
    toUpdateRadius() {
        this.to('ds:UpdateRadius');
        return this;
    }
    /**
     * Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html
     */
    toUpdateTrust() {
        this.to('ds:UpdateTrust');
        return this;
    }
    /**
     * Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html
     */
    toVerifyTrust() {
        this.to('ds:VerifyTrust');
        return this;
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Ds = Ds;
//# sourceMappingURL=data:application/json;base64,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