"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediaconvert = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediaconvert extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediaconvert.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediaconvert';
        this.accessLevelList = {
            "Write": [
                "AssociateCertificate",
                "CancelJob",
                "CreateJob",
                "CreateJobTemplate",
                "CreatePreset",
                "CreateQueue",
                "DeleteJobTemplate",
                "DeletePreset",
                "DeleteQueue",
                "DisassociateCertificate",
                "UpdateJobTemplate",
                "UpdatePreset",
                "UpdateQueue"
            ],
            "List": [
                "DescribeEndpoints",
                "ListJobTemplates",
                "ListJobs",
                "ListPresets",
                "ListQueues"
            ],
            "Read": [
                "GetJob",
                "GetJobTemplate",
                "GetPreset",
                "GetQueue",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     */
    toAssociateCertificate() {
        this.to('mediaconvert:AssociateCertificate');
        return this;
    }
    /**
     * Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCancelJob() {
        this.to('mediaconvert:CancelJob');
        return this;
    }
    /**
     * Grants permission to create and submit an AWS Elemental MediaConvert job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toCreateJob() {
        this.to('mediaconvert:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCreateJobTemplate() {
        this.to('mediaconvert:CreateJobTemplate');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toCreatePreset() {
        this.to('mediaconvert:CreatePreset');
        return this;
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toCreateQueue() {
        this.to('mediaconvert:CreateQueue');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toDeleteJobTemplate() {
        this.to('mediaconvert:DeleteJobTemplate');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toDeletePreset() {
        this.to('mediaconvert:DeletePreset');
        return this;
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toDeleteQueue() {
        this.to('mediaconvert:DeleteQueue');
        return this;
    }
    /**
     * Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html
     */
    toDescribeEndpoints() {
        this.to('mediaconvert:DescribeEndpoints');
        return this;
    }
    /**
     * Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html
     */
    toDisassociateCertificate() {
        this.to('mediaconvert:DisassociateCertificate');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toGetJob() {
        this.to('mediaconvert:GetJob');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toGetJobTemplate() {
        this.to('mediaconvert:GetJobTemplate');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert output preset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toGetPreset() {
        this.to('mediaconvert:GetPreset');
        return this;
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job queue
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toGetQueue() {
        this.to('mediaconvert:GetQueue');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     */
    toListJobTemplates() {
        this.to('mediaconvert:ListJobTemplates');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toListJobs() {
        this.to('mediaconvert:ListJobs');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert output presets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toListPresets() {
        this.to('mediaconvert:ListPresets');
        return this;
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job queues
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toListQueues() {
        this.to('mediaconvert:ListQueues');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toListTagsForResource() {
        this.to('mediaconvert:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to add tags to a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     */
    toTagResource() {
        this.to('mediaconvert:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toUntagResource() {
        this.to('mediaconvert:UntagResource');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toUpdateJobTemplate() {
        this.to('mediaconvert:UpdateJobTemplate');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toUpdatePreset() {
        this.to('mediaconvert:UpdatePreset');
        return this;
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toUpdateQueue() {
        this.to('mediaconvert:UpdateQueue');
        return this;
    }
    /**
     * Adds a resource of type Job to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:jobs/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Queue to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(queueName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:queues/${QueueName}';
        arn = arn.replace('${QueueName}', queueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Preset to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     *
     * @param presetName - Identifier for the presetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPreset(presetName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:presets/${PresetName}';
        arn = arn.replace('${PresetName}', presetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type JobTemplate to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     *
     * @param jobTemplateName - Identifier for the jobTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobTemplate(jobTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:jobTemplates/${JobTemplateName}';
        arn = arn.replace('${JobTemplateName}', jobTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CertificateAssociation to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     *
     * @param certificateArn - Identifier for the certificateArn.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCertificateAssociation(certificateArn, account, region, partition) {
        var arn = 'arn:${Partition}:mediaconvert:${Region}:${Account}:certificates/${CertificateArn}';
        arn = arn.replace('${CertificateArn}', certificateArn);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediaconvert = Mediaconvert;
//# sourceMappingURL=data:application/json;base64,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