"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glue = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glue extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glue';
        this.accessLevelList = {
            "Write": [
                "BatchCreatePartition",
                "BatchDeleteConnection",
                "BatchDeletePartition",
                "BatchDeleteTable",
                "BatchDeleteTableVersion",
                "BatchStopJobRun",
                "CancelMLTaskRun",
                "CreateClassifier",
                "CreateConnection",
                "CreateCrawler",
                "CreateDatabase",
                "CreateDevEndpoint",
                "CreateJob",
                "CreateMLTransform",
                "CreatePartition",
                "CreateScript",
                "CreateSecurityConfiguration",
                "CreateTable",
                "CreateTrigger",
                "CreateUserDefinedFunction",
                "CreateWorkflow",
                "DeleteClassifier",
                "DeleteConnection",
                "DeleteCrawler",
                "DeleteDatabase",
                "DeleteDevEndpoint",
                "DeleteJob",
                "DeleteMLTransform",
                "DeletePartition",
                "DeleteResourcePolicy",
                "DeleteSecurityConfiguration",
                "DeleteTable",
                "DeleteTableVersion",
                "DeleteTrigger",
                "DeleteUserDefinedFunction",
                "DeleteWorkflow",
                "GetMapping",
                "ImportCatalogToGlue",
                "PutDataCatalogEncryptionSettings",
                "PutResourcePolicy",
                "PutWorkflowRunProperties",
                "ResetJobBookmark",
                "StartCrawler",
                "StartCrawlerSchedule",
                "StartExportLabelsTaskRun",
                "StartImportLabelsTaskRun",
                "StartJobRun",
                "StartMLEvaluationTaskRun",
                "StartMLLabelingSetGenerationTaskRun",
                "StartTrigger",
                "StartWorkflowRun",
                "StopCrawler",
                "StopCrawlerSchedule",
                "StopTrigger",
                "UpdateClassifier",
                "UpdateConnection",
                "UpdateCrawler",
                "UpdateCrawlerSchedule",
                "UpdateDatabase",
                "UpdateDevEndpoint",
                "UpdateJob",
                "UpdateMLTransform",
                "UpdatePartition",
                "UpdateTable",
                "UpdateTrigger",
                "UpdateUserDefinedFunction",
                "UpdateWorkflow",
                "UseMLTransforms"
            ],
            "Read": [
                "BatchGetCrawlers",
                "BatchGetDevEndpoints",
                "BatchGetJobs",
                "BatchGetPartition",
                "BatchGetTriggers",
                "BatchGetWorkflows",
                "GetCatalogImportStatus",
                "GetClassifier",
                "GetClassifiers",
                "GetConnection",
                "GetConnections",
                "GetCrawler",
                "GetCrawlerMetrics",
                "GetCrawlers",
                "GetDataCatalogEncryptionSettings",
                "GetDatabase",
                "GetDatabases",
                "GetDataflowGraph",
                "GetDevEndpoint",
                "GetDevEndpoints",
                "GetJob",
                "GetJobBookmark",
                "GetJobRun",
                "GetJobRuns",
                "GetJobs",
                "GetMLTaskRun",
                "GetMLTransform",
                "GetPartition",
                "GetPartitions",
                "GetPlan",
                "GetResourcePolicy",
                "GetSecurityConfiguration",
                "GetSecurityConfigurations",
                "GetTable",
                "GetTableVersion",
                "GetTableVersions",
                "GetTables",
                "GetTags",
                "GetTrigger",
                "GetTriggers",
                "GetUserDefinedFunction",
                "GetUserDefinedFunctions",
                "GetWorkflow",
                "GetWorkflowRun",
                "GetWorkflowRunProperties",
                "GetWorkflowRuns",
                "SearchTables"
            ],
            "List": [
                "GetMLTaskRuns",
                "GetMLTransforms",
                "ListCrawlers",
                "ListDevEndpoints",
                "ListJobs",
                "ListMLTransforms",
                "ListTriggers",
                "ListWorkflows"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchCreatePartition
     */
    toBatchCreatePartition() {
        this.to('glue:BatchCreatePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-BatchDeleteConnection
     */
    toBatchDeleteConnection() {
        this.to('glue:BatchDeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchDeletePartition
     */
    toBatchDeletePartition() {
        this.to('glue:BatchDeletePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more tables
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-BatchDeleteTable
     */
    toBatchDeleteTable() {
        this.to('glue:BatchDeleteTable');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    toBatchDeleteTableVersion() {
        this.to('glue:BatchDeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve one or more crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-BatchGetCrawlers
     */
    toBatchGetCrawlers() {
        this.to('glue:BatchGetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-BatchGetDevEndpoints
     */
    toBatchGetDevEndpoints() {
        this.to('glue:BatchGetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve one or more jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-BatchGetJobs
     */
    toBatchGetJobs() {
        this.to('glue:BatchGetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve one or more partitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchGetPartition
     */
    toBatchGetPartition() {
        this.to('glue:BatchGetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve one or more triggers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-BatchGetTriggers
     */
    toBatchGetTriggers() {
        this.to('glue:BatchGetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more workflows
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-BatchGetWorkflows
     */
    toBatchGetWorkflows() {
        this.to('glue:BatchGetWorkflows');
        return this;
    }
    /**
     * Grants permission to stop one or more job runs for a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-BatchStopStartJobRun
     */
    toBatchStopJobRun() {
        this.to('glue:BatchStopJobRun');
        return this;
    }
    /**
     * Grants permission to stop a running ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CancelMLTaskRun
     */
    toCancelMLTaskRun() {
        this.to('glue:CancelMLTaskRun');
        return this;
    }
    /**
     * Grants permission to create a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-CreateClassifier
     */
    toCreateClassifier() {
        this.to('glue:CreateClassifier');
        return this;
    }
    /**
     * Grants permission to create a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-CreateConnection
     */
    toCreateConnection() {
        this.to('glue:CreateConnection');
        return this;
    }
    /**
     * Grants permission to create a crawler
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-CreateCrawler
     */
    toCreateCrawler() {
        this.to('glue:CreateCrawler');
        return this;
    }
    /**
     * Grants permission to create a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-CreateDatabase
     */
    toCreateDatabase() {
        this.to('glue:CreateDatabase');
        return this;
    }
    /**
     * Grants permission to create a development endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-CreateDevEndpoint
     */
    toCreateDevEndpoint() {
        this.to('glue:CreateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to create a job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-CreateJob
     */
    toCreateJob() {
        this.to('glue:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CreateMLTransform
     */
    toCreateMLTransform() {
        this.to('glue:CreateMLTransform');
        return this;
    }
    /**
     * Grants permission to create a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-CreatePartition
     */
    toCreatePartition() {
        this.to('glue:CreatePartition');
        return this;
    }
    /**
     * Grants permission to create a script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-CreateScript
     */
    toCreateScript() {
        this.to('glue:CreateScript');
        return this;
    }
    /**
     * Grants permission to create a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-CreateSecurityConfiguration
     */
    toCreateSecurityConfiguration() {
        this.to('glue:CreateSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to create a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-CreateTable
     */
    toCreateTable() {
        this.to('glue:CreateTable');
        return this;
    }
    /**
     * Grants permission to create a trigger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-CreateTrigger
     */
    toCreateTrigger() {
        this.to('glue:CreateTrigger');
        return this;
    }
    /**
     * Grants permission to create a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-CreateUserDefinedFunction
     */
    toCreateUserDefinedFunction() {
        this.to('glue:CreateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to create a workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-CreateWorkflow
     */
    toCreateWorkflow() {
        this.to('glue:CreateWorkflow');
        return this;
    }
    /**
     * Grants permission to delete a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-DeleteClassifier
     */
    toDeleteClassifier() {
        this.to('glue:DeleteClassifier');
        return this;
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-DeleteConnection
     */
    toDeleteConnection() {
        this.to('glue:DeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-DeleteCrawler
     */
    toDeleteCrawler() {
        this.to('glue:DeleteCrawler');
        return this;
    }
    /**
     * Grants permission to delete a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-DeleteDatabase
     */
    toDeleteDatabase() {
        this.to('glue:DeleteDatabase');
        return this;
    }
    /**
     * Grants permission to delete a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-DeleteDevEndpoint
     */
    toDeleteDevEndpoint() {
        this.to('glue:DeleteDevEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-DeleteJob
     */
    toDeleteJob() {
        this.to('glue:DeleteJob');
        return this;
    }
    /**
     * Grants permission to delete an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-DeleteMLTransform
     */
    toDeleteMLTransform() {
        this.to('glue:DeleteMLTransform');
        return this;
    }
    /**
     * Grants permission to delete a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-DeletePartition
     */
    toDeletePartition() {
        this.to('glue:DeletePartition');
        return this;
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteResourcePolicy
     */
    toDeleteResourcePolicy() {
        this.to('glue:DeleteResourcePolicy');
        return this;
    }
    /**
     * Grants permission to delete a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteSecurityConfiguration
     */
    toDeleteSecurityConfiguration() {
        this.to('glue:DeleteSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTable
     */
    toDeleteTable() {
        this.to('glue:DeleteTable');
        return this;
    }
    /**
     * Grants permission to delete a version of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    toDeleteTableVersion() {
        this.to('glue:DeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to delete a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-DeleteTrigger
     */
    toDeleteTrigger() {
        this.to('glue:DeleteTrigger');
        return this;
    }
    /**
     * Grants permission to delete a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-DeleteUserDefinedFunction
     */
    toDeleteUserDefinedFunction() {
        this.to('glue:DeleteUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-DeleteWorkflow
     */
    toDeleteWorkflow() {
        this.to('glue:DeleteWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve the catalog import status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-GetCatalogImportStatus
     */
    toGetCatalogImportStatus() {
        this.to('glue:GetCatalogImportStatus');
        return this;
    }
    /**
     * Grants permission to retrieve a classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifier
     */
    toGetClassifier() {
        this.to('glue:GetClassifier');
        return this;
    }
    /**
     * Grants permission to list all classifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifiers
     */
    toGetClassifiers() {
        this.to('glue:GetClassifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnection
     */
    toGetConnection() {
        this.to('glue:GetConnection');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnections
     */
    toGetConnections() {
        this.to('glue:GetConnections');
        return this;
    }
    /**
     * Grants permission to retrieve a crawler
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawler
     */
    toGetCrawler() {
        this.to('glue:GetCrawler');
        return this;
    }
    /**
     * Grants permission to retrieve metrics about crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlerMetrics
     */
    toGetCrawlerMetrics() {
        this.to('glue:GetCrawlerMetrics');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlers
     */
    toGetCrawlers() {
        this.to('glue:GetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve catalog encryption settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetDataCatalogEncryptionSettings
     */
    toGetDataCatalogEncryptionSettings() {
        this.to('glue:GetDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabase
     */
    toGetDatabase() {
        this.to('glue:GetDatabase');
        return this;
    }
    /**
     * Grants permission to retrieve all databases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabases
     */
    toGetDatabases() {
        this.to('glue:GetDatabases');
        return this;
    }
    /**
     * Grants permission to transform a script into a directed acyclic graph (DAG)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetDataflowGraph
     */
    toGetDataflowGraph() {
        this.to('glue:GetDataflowGraph');
        return this;
    }
    /**
     * Grants permission to retrieve a development endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoint
     */
    toGetDevEndpoint() {
        this.to('glue:GetDevEndpoint');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoints
     */
    toGetDevEndpoints() {
        this.to('glue:GetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJob
     */
    toGetJob() {
        this.to('glue:GetJob');
        return this;
    }
    /**
     * Grants permission to retrieve a job bookmark
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-job-GetJobBookmark
     */
    toGetJobBookmark() {
        this.to('glue:GetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve a job run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRun
     */
    toGetJobRun() {
        this.to('glue:GetJobRun');
        return this;
    }
    /**
     * Grants permission to retrieve all job runs of a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRuns
     */
    toGetJobRuns() {
        this.to('glue:GetJobRuns');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJobs
     */
    toGetJobs() {
        this.to('glue:GetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Task Run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRun
     */
    toGetMLTaskRun() {
        this.to('glue:GetMLTaskRun');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Task Runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRuns
     */
    toGetMLTaskRuns() {
        this.to('glue:GetMLTaskRuns');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Transform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransform
     */
    toGetMLTransform() {
        this.to('glue:GetMLTransform');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransforms
     */
    toGetMLTransforms() {
        this.to('glue:GetMLTransforms');
        return this;
    }
    /**
     * Grants permission to create a mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetMapping
     */
    toGetMapping() {
        this.to('glue:GetMapping');
        return this;
    }
    /**
     * Grants permission to retrieve a partition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartition
     */
    toGetPartition() {
        this.to('glue:GetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve the partitions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartitions
     */
    toGetPartitions() {
        this.to('glue:GetPartitions');
        return this;
    }
    /**
     * Grants permission to retrieve a mapping for a script
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetPlan
     */
    toGetPlan() {
        this.to('glue:GetPlan');
        return this;
    }
    /**
     * Grants permission to retrieve a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetResourcePolicy
     */
    toGetResourcePolicy() {
        this.to('glue:GetResourcePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a security configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfiguration
     */
    toGetSecurityConfiguration() {
        this.to('glue:GetSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve one or more security configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfigurations
     */
    toGetSecurityConfigurations() {
        this.to('glue:GetSecurityConfigurations');
        return this;
    }
    /**
     * Grants permission to retrieve a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTable
     */
    toGetTable() {
        this.to('glue:GetTable');
        return this;
    }
    /**
     * Grants permission to retrieve a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersion
     */
    toGetTableVersion() {
        this.to('glue:GetTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersions
     */
    toGetTableVersions() {
        this.to('glue:GetTableVersions');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTables
     */
    toGetTables() {
        this.to('glue:GetTables');
        return this;
    }
    /**
     * Grants permission to retrieve all tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    toGetTags() {
        this.to('glue:GetTags');
        return this;
    }
    /**
     * Grants permission to retrieve a trigger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTrigger
     */
    toGetTrigger() {
        this.to('glue:GetTrigger');
        return this;
    }
    /**
     * Grants permission to retrieve the triggers associated with a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTriggers
     */
    toGetTriggers() {
        this.to('glue:GetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve a function definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunction
     */
    toGetUserDefinedFunction() {
        this.to('glue:GetUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to retrieve multiple function definitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunctions
     */
    toGetUserDefinedFunctions() {
        this.to('glue:GetUserDefinedFunctions');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflow
     */
    toGetWorkflow() {
        this.to('glue:GetWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRun
     */
    toGetWorkflowRun() {
        this.to('glue:GetWorkflowRun');
        return this;
    }
    /**
     * Grants permission to retrieve workflow run properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRunProperties
     */
    toGetWorkflowRunProperties() {
        this.to('glue:GetWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to retrieve all runs of a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRuns
     */
    toGetWorkflowRuns() {
        this.to('glue:GetWorkflowRuns');
        return this;
    }
    /**
     * Grants permission to import an Athena data catalog into AWS Glue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-ImportCatalogToGlue
     */
    toImportCatalogToGlue() {
        this.to('glue:ImportCatalogToGlue');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-ListCrawlers
     */
    toListCrawlers() {
        this.to('glue:ListCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-ListDevEndpoints
     */
    toListDevEndpoints() {
        this.to('glue:ListDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-ListJobs
     */
    toListJobs() {
        this.to('glue:ListJobs');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-ListMLTransforms
     */
    toListMLTransforms() {
        this.to('glue:ListMLTransforms');
        return this;
    }
    /**
     * Grants permission to retrieve all triggers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-ListTriggers
     */
    toListTriggers() {
        this.to('glue:ListTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve all workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-ListWorkflows
     */
    toListWorkflows() {
        this.to('glue:ListWorkflows');
        return this;
    }
    /**
     * Grants permission to update catalog encryption settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutDataCatalogEncryptionSettings
     */
    toPutDataCatalogEncryptionSettings() {
        this.to('glue:PutDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to update a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutResourcePolicy
     */
    toPutResourcePolicy() {
        this.to('glue:PutResourcePolicy');
        return this;
    }
    /**
     * Grants permission to update workflow run properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-PutWorkflowRunProperties
     */
    toPutWorkflowRunProperties() {
        this.to('glue:PutWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to reset a job bookmark
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-ResetJobBookmark
     */
    toResetJobBookmark() {
        this.to('glue:ResetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in the catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-SearchTables
     */
    toSearchTables() {
        this.to('glue:SearchTables');
        return this;
    }
    /**
     * Grants permission to start a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler
     */
    toStartCrawler() {
        this.to('glue:StartCrawler');
        return this;
    }
    /**
     * Grants permission to change the schedule state of a crawler to SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StartCrawlerSchedule
     */
    toStartCrawlerSchedule() {
        this.to('glue:StartCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to start an Export Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartExportLabelsTaskRun
     */
    toStartExportLabelsTaskRun() {
        this.to('glue:StartExportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start an Import Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartImportLabelsTaskRun
     */
    toStartImportLabelsTaskRun() {
        this.to('glue:StartImportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start running a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun
     */
    toStartJobRun() {
        this.to('glue:StartJobRun');
        return this;
    }
    /**
     * Grants permission to start an Evaluation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLEvaluationTaskRun
     */
    toStartMLEvaluationTaskRun() {
        this.to('glue:StartMLEvaluationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a Labeling Set Generation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLLabelingSetGenerationTaskRun
     */
    toStartMLLabelingSetGenerationTaskRun() {
        this.to('glue:StartMLLabelingSetGenerationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StartTrigger
     */
    toStartTrigger() {
        this.to('glue:StartTrigger');
        return this;
    }
    /**
     * Grants permission to start running a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-StartWorkflowRun
     */
    toStartWorkflowRun() {
        this.to('glue:StartWorkflowRun');
        return this;
    }
    /**
     * Grants permission to stop a running crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StopCrawler
     */
    toStopCrawler() {
        this.to('glue:StopCrawler');
        return this;
    }
    /**
     * Grants permission to set the schedule state of a crawler to NOT_SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StopCrawlerSchedule
     */
    toStopCrawlerSchedule() {
        this.to('glue:StopCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to stop a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StopTrigger
     */
    toStopTrigger() {
        this.to('glue:StopTrigger');
        return this;
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-TagResource
     */
    toTagResource() {
        this.to('glue:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags associated with a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    toUntagResource() {
        this.to('glue:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-UpdateClassifier
     */
    toUpdateClassifier() {
        this.to('glue:UpdateClassifier');
        return this;
    }
    /**
     * Grants permission to update a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-UpdateConnection
     */
    toUpdateConnection() {
        this.to('glue:UpdateConnection');
        return this;
    }
    /**
     * Grants permission to update a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-UpdateCrawler
     */
    toUpdateCrawler() {
        this.to('glue:UpdateCrawler');
        return this;
    }
    /**
     * Grants permission to update the schedule of a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-UpdateCrawlerSchedule
     */
    toUpdateCrawlerSchedule() {
        this.to('glue:UpdateCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to update a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-UpdateDatabase
     */
    toUpdateDatabase() {
        this.to('glue:UpdateDatabase');
        return this;
    }
    /**
     * Grants permission to update a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-UpdateDevEndpoint
     */
    toUpdateDevEndpoint() {
        this.to('glue:UpdateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to update a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-UpdateJob
     */
    toUpdateJob() {
        this.to('glue:UpdateJob');
        return this;
    }
    /**
     * Grants permission to update an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-UpdateMLTransform
     */
    toUpdateMLTransform() {
        this.to('glue:UpdateMLTransform');
        return this;
    }
    /**
     * Grants permission to update a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-UpdatePartition
     */
    toUpdatePartition() {
        this.to('glue:UpdatePartition');
        return this;
    }
    /**
     * Grants permission to update a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-UpdateTable
     */
    toUpdateTable() {
        this.to('glue:UpdateTable');
        return this;
    }
    /**
     * Grants permission to update a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-UpdateTrigger
     */
    toUpdateTrigger() {
        this.to('glue:UpdateTrigger');
        return this;
    }
    /**
     * Grants permission to update a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-UpdateUserDefinedFunction
     */
    toUpdateUserDefinedFunction() {
        this.to('glue:UpdateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to update a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-UpdateWorkflow
     */
    toUpdateWorkflow() {
        this.to('glue:UpdateWorkflow');
        return this;
    }
    /**
     * Grants permission to use an ML Transform from within a Glue ETL Script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html
     */
    toUseMLTransforms() {
        this.to('glue:UseMLTransforms');
        return this;
    }
    /**
     * Adds a resource of type catalog to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCatalog(account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:catalog';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatabase(databaseName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:database/${DatabaseName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(databaseName, tableName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:table/${DatabaseName}/${TableName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tableversion to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param tableVersionName - Identifier for the tableVersionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTableversion(databaseName, tableName, tableVersionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:tableVersion/${DatabaseName}/${TableName}/${TableVersionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${TableVersionName}', tableVersionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param connectionName - Identifier for the connectionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnection(connectionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:connection/${ConnectionName}';
        arn = arn.replace('${ConnectionName}', connectionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type userdefinedfunction to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param userDefinedFunctionName - Identifier for the userDefinedFunctionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUserdefinedfunction(databaseName, userDefinedFunctionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:userDefinedFunction/${DatabaseName}/${UserDefinedFunctionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${UserDefinedFunctionName}', userDefinedFunctionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type devendpoint to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param devEndpointName - Identifier for the devEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevendpoint(devEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:devendpoint/${DevEndpointName}';
        arn = arn.replace('${DevEndpointName}', devEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:job/${JobName}';
        arn = arn.replace('${JobName}', jobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trigger to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param triggerName - Identifier for the triggerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrigger(triggerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:trigger/${TriggerName}';
        arn = arn.replace('${TriggerName}', triggerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type crawler to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param crawlerName - Identifier for the crawlerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCrawler(crawlerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:crawler/${CrawlerName}';
        arn = arn.replace('${CrawlerName}', crawlerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflow(workflowName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:workflow/${WorkflowName}';
        arn = arn.replace('${WorkflowName}', workflowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mlTransform to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param transformId - Identifier for the transformId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMlTransform(transformId, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:mlTransform/${TransformId}';
        arn = arn.replace('${TransformId}', transformId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Glue = Glue;
//# sourceMappingURL=data:application/json;base64,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