"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iot1click = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iot1click](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot1-click.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iot1click extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iot1click](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot1-click.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iot1click';
        this.accessLevelList = {
            "Write": [
                "AssociateDeviceWithPlacement",
                "CreatePlacement",
                "CreateProject",
                "DeletePlacement",
                "DeleteProject",
                "DisassociateDeviceFromPlacement",
                "InvokeDeviceMethod",
                "TagResource",
                "UntagResource",
                "UpdateDeviceState",
                "UpdatePlacement",
                "UpdateProject"
            ],
            "Read": [
                "ClaimDevicesByClaimCode",
                "DescribeDevice",
                "DescribePlacement",
                "DescribeProject",
                "FinalizeDeviceClaim",
                "GetDeviceMethods",
                "GetDevicesInPlacement",
                "InitiateDeviceClaim",
                "ListDeviceEvents",
                "ListPlacements",
                "UnclaimDevice"
            ],
            "List": [
                "ListDevices",
                "ListProjects",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Associate a device to a placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_AssociateDeviceWithPlacement.html
     */
    toAssociateDeviceWithPlacement() {
        this.to('iot1click:AssociateDeviceWithPlacement');
        return this;
    }
    /**
     * Claim a batch of devices with a claim code.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/claims-claimcode.html
     */
    toClaimDevicesByClaimCode() {
        this.to('iot1click:ClaimDevicesByClaimCode');
        return this;
    }
    /**
     * Create a new placement in a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_CreatePlacement.html
     */
    toCreatePlacement() {
        this.to('iot1click:CreatePlacement');
        return this;
    }
    /**
     * Create a new project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_CreateProject.html
     */
    toCreateProject() {
        this.to('iot1click:CreateProject');
        return this;
    }
    /**
     * Delete a placement from a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DeletePlacement.html
     */
    toDeletePlacement() {
        this.to('iot1click:DeletePlacement');
        return this;
    }
    /**
     * Delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DeleteProject.html
     */
    toDeleteProject() {
        this.to('iot1click:DeleteProject');
        return this;
    }
    /**
     * Describe a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid.html
     */
    toDescribeDevice() {
        this.to('iot1click:DescribeDevice');
        return this;
    }
    /**
     * Describe a placement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DescribePlacement.html
     */
    toDescribePlacement() {
        this.to('iot1click:DescribePlacement');
        return this;
    }
    /**
     * Describe a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DescribeProject.html
     */
    toDescribeProject() {
        this.to('iot1click:DescribeProject');
        return this;
    }
    /**
     * Disassociate a device from a placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_DissacociateDeviceFromPlacement.html
     */
    toDisassociateDeviceFromPlacement() {
        this.to('iot1click:DisassociateDeviceFromPlacement');
        return this;
    }
    /**
     * Finalize a device claim
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-finalize-claim.html
     */
    toFinalizeDeviceClaim() {
        this.to('iot1click:FinalizeDeviceClaim');
        return this;
    }
    /**
     * Get available methods of a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-methods.html
     */
    toGetDeviceMethods() {
        this.to('iot1click:GetDeviceMethods');
        return this;
    }
    /**
     * Get devices associated to a placement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_GetDevicesInPlacement.html
     */
    toGetDevicesInPlacement() {
        this.to('iot1click:GetDevicesInPlacement');
        return this;
    }
    /**
     * Initialize a device claim
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-initiate-claim.html
     */
    toInitiateDeviceClaim() {
        this.to('iot1click:InitiateDeviceClaim');
        return this;
    }
    /**
     * Invoke a device method
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-methods.html
     */
    toInvokeDeviceMethod() {
        this.to('iot1click:InvokeDeviceMethod');
        return this;
    }
    /**
     * List past events published by a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-events.html
     */
    toListDeviceEvents() {
        this.to('iot1click:ListDeviceEvents');
        return this;
    }
    /**
     * List all devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices.html
     */
    toListDevices() {
        this.to('iot1click:ListDevices');
        return this;
    }
    /**
     * List placements in a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListPlacements.html
     */
    toListPlacements() {
        this.to('iot1click:ListPlacements');
        return this;
    }
    /**
     * List all projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListProjects.html
     */
    toListProjects() {
        this.to('iot1click:ListProjects');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('iot1click:ListTagsForResource');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_TagResource.html
     */
    toTagResource() {
        this.to('iot1click:TagResource');
        return this;
    }
    /**
     * Unclaim a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-unclaim.html
     */
    toUnclaimDevice() {
        this.to('iot1click:UnclaimDevice');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('iot1click:UntagResource');
        return this;
    }
    /**
     * Update device state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/devices-deviceid-state.html
     */
    toUpdateDeviceState() {
        this.to('iot1click:UpdateDeviceState');
        return this;
    }
    /**
     * Update a placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UpdatePlacement.html
     */
    toUpdatePlacement() {
        this.to('iot1click:UpdatePlacement');
        return this;
    }
    /**
     * Update a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_UpdateProject.html
     */
    toUpdateProject() {
        this.to('iot1click:UpdateProject');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/iot-1-click/1.0/devices-apireference/resources.html
     *
     * @param deviceId - Identifier for the deviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevice(deviceId, account, region, partition) {
        var arn = 'arn:${Partition}:iot1click:${Region}:${Account}:devices/${DeviceId}';
        arn = arn.replace('${DeviceId}', deviceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/iot-1-click/latest/projects-apireference/API_Operations.html
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:iot1click:${Region}:${Account}:projects/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Iot1click = Iot1click;
//# sourceMappingURL=data:application/json;base64,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