"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LicenseManager = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [license-manager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awslicensemanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class LicenseManager extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [license-manager](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awslicensemanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'license-manager';
        this.accessLevelList = {
            "Tagging": [
                "CreateLicenseConfiguration",
                "TagResource",
                "UntagResource"
            ],
            "Write": [
                "DeleteLicenseConfiguration",
                "UpdateLicenseConfiguration",
                "UpdateLicenseSpecificationsForResource"
            ],
            "List": [
                "GetLicenseConfiguration",
                "GetServiceSettings",
                "ListAssociationsForLicenseConfiguration",
                "ListFailuresForLicenseConfigurationOperations",
                "ListLicenseConfigurations",
                "ListLicenseSpecificationsForResource",
                "ListResourceInventory",
                "ListTagsForResource",
                "ListUsageForLicenseConfiguration"
            ],
            "Permissions management": [
                "UpdateServiceSettings"
            ]
        };
    }
    /**
     * Creates a new license configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_CreateLicenseConfiguration.html
     */
    toCreateLicenseConfiguration() {
        this.to('license-manager:CreateLicenseConfiguration');
        return this;
    }
    /**
     * Permanently deletes a license configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_DeleteLicenseConfiguration.html
     */
    toDeleteLicenseConfiguration() {
        this.to('license-manager:DeleteLicenseConfiguration');
        return this;
    }
    /**
     * Gets a license configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_GetLicenseConfiguration.html
     */
    toGetLicenseConfiguration() {
        this.to('license-manager:GetLicenseConfiguration');
        return this;
    }
    /**
     * Gets service settings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_GetServiceSettings.html
     */
    toGetServiceSettings() {
        this.to('license-manager:GetServiceSettings');
        return this;
    }
    /**
     * Lists associations for a selected license configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListAssociationsForLicenseConfiguration.html
     */
    toListAssociationsForLicenseConfiguration() {
        this.to('license-manager:ListAssociationsForLicenseConfiguration');
        return this;
    }
    /**
     * Lists the license configuration operations that failed
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListFailuresForLicenseConfigurationOperations.html
     */
    toListFailuresForLicenseConfigurationOperations() {
        this.to('license-manager:ListFailuresForLicenseConfigurationOperations');
        return this;
    }
    /**
     * Lists license configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListLicenseConfigurations.html
     */
    toListLicenseConfigurations() {
        this.to('license-manager:ListLicenseConfigurations');
        return this;
    }
    /**
     * Lists license specifications associated with a selected resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListLicenseSpecificationsForResource.html
     */
    toListLicenseSpecificationsForResource() {
        this.to('license-manager:ListLicenseSpecificationsForResource');
        return this;
    }
    /**
     * Lists resource inventory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListResourceInventory.html
     */
    toListResourceInventory() {
        this.to('license-manager:ListResourceInventory');
        return this;
    }
    /**
     * Lists tags for a selected resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('license-manager:ListTagsForResource');
        return this;
    }
    /**
     * Lists usage records for selected license configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_ListUsageForLicenseConfiguration.html
     */
    toListUsageForLicenseConfiguration() {
        this.to('license-manager:ListUsageForLicenseConfiguration');
        return this;
    }
    /**
     * Tags a selected resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('license-manager:TagResource');
        return this;
    }
    /**
     * Untags a selected resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('license-manager:UntagResource');
        return this;
    }
    /**
     * Updates an existing license configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UpdateLicenseConfiguration.html
     */
    toUpdateLicenseConfiguration() {
        this.to('license-manager:UpdateLicenseConfiguration');
        return this;
    }
    /**
     * Updates license specifications for a selected resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UpdateLicenseSpecificationsForResource.html
     */
    toUpdateLicenseSpecificationsForResource() {
        this.to('license-manager:UpdateLicenseSpecificationsForResource');
        return this;
    }
    /**
     * Updates service settings
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/license-manager/latest/APIReference/API_UpdateServiceSettings.html
     */
    toUpdateServiceSettings() {
        this.to('license-manager:UpdateServiceSettings');
        return this;
    }
    /**
     * Adds a resource of type license-configuration to the statement
     *
     * @param licenseConfigurationId - Identifier for the licenseConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     */
    onLicenseConfiguration(licenseConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:license-manager:${Region}:${Account}:license-configuration/${LicenseConfigurationId}';
        arn = arn.replace('${LicenseConfigurationId}', licenseConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on tag-value associated with the resource.
     *
     * Applies to resource types:
     * - license-configuration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`license-manager:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.LicenseManager = LicenseManager;
//# sourceMappingURL=data:application/json;base64,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