"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Organizations = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [organizations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsorganizations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Organizations extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [organizations](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsorganizations.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'organizations';
        this.accessLevelList = {
            "Write": [
                "AcceptHandshake",
                "AttachPolicy",
                "CancelHandshake",
                "CreateAccount",
                "CreateGovCloudAccount",
                "CreateOrganization",
                "CreateOrganizationalUnit",
                "CreatePolicy",
                "DeclineHandshake",
                "DeleteOrganization",
                "DeleteOrganizationalUnit",
                "DeletePolicy",
                "DeregisterDelegatedAdministrator",
                "DetachPolicy",
                "DisableAWSServiceAccess",
                "DisablePolicyType",
                "EnableAWSServiceAccess",
                "EnableAllFeatures",
                "EnablePolicyType",
                "InviteAccountToOrganization",
                "LeaveOrganization",
                "MoveAccount",
                "RegisterDelegatedAdministrator",
                "RemoveAccountFromOrganization",
                "UpdateOrganizationalUnit",
                "UpdatePolicy"
            ],
            "Read": [
                "DescribeAccount",
                "DescribeCreateAccountStatus",
                "DescribeEffectivePolicy",
                "DescribeHandshake",
                "DescribeOrganization",
                "DescribeOrganizationalUnit",
                "DescribePolicy"
            ],
            "List": [
                "ListAWSServiceAccessForOrganization",
                "ListAccounts",
                "ListAccountsForParent",
                "ListChildren",
                "ListCreateAccountStatus",
                "ListDelegatedAdministrators",
                "ListDelegatedServicesForAccount",
                "ListHandshakesForAccount",
                "ListHandshakesForOrganization",
                "ListOrganizationalUnitsForParent",
                "ListParents",
                "ListPolicies",
                "ListPoliciesForTarget",
                "ListRoots",
                "ListTagsForResource",
                "ListTargetsForPolicy"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html
     */
    toAcceptHandshake() {
        this.to('organizations:AcceptHandshake');
        return this;
    }
    /**
     * Grants permission to attach a policy to a root, an organizational unit, or an individual account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        this.to('organizations:AttachPolicy');
        return this;
    }
    /**
     * Grants permission to cancel a handshake.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html
     */
    toCancelHandshake() {
        this.to('organizations:CancelHandshake');
        return this;
    }
    /**
     * Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        this.to('organizations:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create an AWS GovCloud (US) account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html
     */
    toCreateGovCloudAccount() {
        this.to('organizations:CreateGovCloudAccount');
        return this;
    }
    /**
     * Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the master account of the new organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html
     */
    toCreateOrganization() {
        this.to('organizations:CreateOrganization');
        return this;
    }
    /**
     * Grants permission to create an organizational unit (OU) within a root or parent OU.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html
     */
    toCreateOrganizationalUnit() {
        this.to('organizations:CreateOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        this.to('organizations:CreatePolicy');
        return this;
    }
    /**
     * Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html
     */
    toDeclineHandshake() {
        this.to('organizations:DeclineHandshake');
        return this;
    }
    /**
     * Grants permission to delete the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html
     */
    toDeleteOrganization() {
        this.to('organizations:DeleteOrganization');
        return this;
    }
    /**
     * Grants permission to delete an organizational unit from a root or another OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html
     */
    toDeleteOrganizationalUnit() {
        this.to('organizations:DeleteOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to delete a policy from your organization.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.to('organizations:DeletePolicy');
        return this;
    }
    /**
     * Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html
     */
    toDeregisterDelegatedAdministrator() {
        this.to('organizations:DeregisterDelegatedAdministrator');
        return this;
    }
    /**
     * Grants permission to retrieve Organizations-related details about the specified account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html
     */
    toDescribeAccount() {
        this.to('organizations:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the current status of an asynchronous request to create an account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html
     */
    toDescribeCreateAccountStatus() {
        this.to('organizations:DescribeCreateAccountStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the effective policy for an account.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html
     */
    toDescribeEffectivePolicy() {
        this.to('organizations:DescribeEffectivePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve details about a previously requested handshake.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html
     */
    toDescribeHandshake() {
        this.to('organizations:DescribeHandshake');
        return this;
    }
    /**
     * Grants permission to retrieves details about the organization that the calling credentials belong to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        this.to('organizations:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to retrieve details about an organizational unit (OU).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html
     */
    toDescribeOrganizationalUnit() {
        this.to('organizations:DescribeOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to retrieves details about a policy.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html
     */
    toDescribePolicy() {
        this.to('organizations:DescribePolicy');
        return this;
    }
    /**
     * Grants permission to detach a policy from a target root, organizational unit, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        this.to('organizations:DetachPolicy');
        return this;
    }
    /**
     * Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html
     */
    toDisableAWSServiceAccess() {
        this.to('organizations:DisableAWSServiceAccess');
        return this;
    }
    /**
     * Grants permission to disable an organization policy type in a root.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html
     */
    toDisablePolicyType() {
        this.to('organizations:DisablePolicyType');
        return this;
    }
    /**
     * Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html
     */
    toEnableAWSServiceAccess() {
        this.to('organizations:EnableAWSServiceAccess');
        return this;
    }
    /**
     * Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html
     */
    toEnableAllFeatures() {
        this.to('organizations:EnableAllFeatures');
        return this;
    }
    /**
     * Grants permission to enable a policy type in a root.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html
     */
    toEnablePolicyType() {
        this.to('organizations:EnablePolicyType');
        return this;
    }
    /**
     * Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html
     */
    toInviteAccountToOrganization() {
        this.to('organizations:InviteAccountToOrganization');
        return this;
    }
    /**
     * Grants permission to remove a member account from its parent organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html
     */
    toLeaveOrganization() {
        this.to('organizations:LeaveOrganization');
        return this;
    }
    /**
     * Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html
     */
    toListAWSServiceAccessForOrganization() {
        this.to('organizations:ListAWSServiceAccessForOrganization');
        return this;
    }
    /**
     * Grants permission to list all of the the accounts in the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        this.to('organizations:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html
     */
    toListAccountsForParent() {
        this.to('organizations:ListAccountsForParent');
        return this;
    }
    /**
     * Grants permission to list all of the OUs or accounts that are contained in a parent OU or root.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html
     */
    toListChildren() {
        this.to('organizations:ListChildren');
        return this;
    }
    /**
     * Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html
     */
    toListCreateAccountStatus() {
        this.to('organizations:ListCreateAccountStatus');
        return this;
    }
    /**
     * Grants permission to list the AWS accounts that are designated as delegated administrators in this organization.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html
     */
    toListDelegatedAdministrators() {
        this.to('organizations:ListDelegatedAdministrators');
        return this;
    }
    /**
     * Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html
     */
    toListDelegatedServicesForAccount() {
        this.to('organizations:ListDelegatedServicesForAccount');
        return this;
    }
    /**
     * Grants permission to list all of the handshakes that are associated with an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html
     */
    toListHandshakesForAccount() {
        this.to('organizations:ListHandshakesForAccount');
        return this;
    }
    /**
     * Grants permission to list the handshakes that are associated with the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html
     */
    toListHandshakesForOrganization() {
        this.to('organizations:ListHandshakesForOrganization');
        return this;
    }
    /**
     * Grants permission to lists all of the organizational units (OUs) in a parent organizational unit or root.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html
     */
    toListOrganizationalUnitsForParent() {
        this.to('organizations:ListOrganizationalUnitsForParent');
        return this;
    }
    /**
     * Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html
     */
    toListParents() {
        this.to('organizations:ListParents');
        return this;
    }
    /**
     * Grants permission to list all of the policies in an organization.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        this.to('organizations:ListPolicies');
        return this;
    }
    /**
     * Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html
     */
    toListPoliciesForTarget() {
        this.to('organizations:ListPoliciesForTarget');
        return this;
    }
    /**
     * Grants permission to list all of the roots that are defined in the organization.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html
     */
    toListRoots() {
        this.to('organizations:ListRoots');
        return this;
    }
    /**
     * Grants permission to list all tags for the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('organizations:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list all the roots, OUs, and accounts to which a policy is attached.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html
     */
    toListTargetsForPolicy() {
        this.to('organizations:ListTargetsForPolicy');
        return this;
    }
    /**
     * Grants permission to move an account from its current root or OU to another parent root or OU.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html
     */
    toMoveAccount() {
        this.to('organizations:MoveAccount');
        return this;
    }
    /**
     * Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html
     */
    toRegisterDelegatedAdministrator() {
        this.to('organizations:RegisterDelegatedAdministrator');
        return this;
    }
    /**
     * Grants permission to removes the specified account from the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html
     */
    toRemoveAccountFromOrganization() {
        this.to('organizations:RemoveAccountFromOrganization');
        return this;
    }
    /**
     * Grants permission to add one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('organizations:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('organizations:UntagResource');
        return this;
    }
    /**
     * Grants permission to rename an organizational unit (OU).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html
     */
    toUpdateOrganizationalUnit() {
        this.to('organizations:UpdateOrganizationalUnit');
        return this;
    }
    /**
     * Grants permission to update an existing policy with a new name, description, or content.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html
     */
    toUpdatePolicy() {
        this.to('organizations:UpdatePolicy');
        return this;
    }
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccount(masterAccountId, organizationId, accountId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:account/o-${OrganizationId}/${AccountId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type handshake to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param handshakeType - Identifier for the handshakeType.
     * @param handshakeId - Identifier for the handshakeId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHandshake(masterAccountId, organizationId, handshakeType, handshakeId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:handshake/o-${OrganizationId}/${HandshakeType}/h-${HandshakeId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${HandshakeType}', handshakeType);
        arn = arn.replace('${HandshakeId}', handshakeId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganization(masterAccountId, organizationId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:organization/o-${OrganizationId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type organizationalunit to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param organizationalUnitId - Identifier for the organizationalUnitId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganizationalunit(masterAccountId, organizationId, organizationalUnitId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:ou/o-${OrganizationId}/ou-${OrganizationalUnitId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${OrganizationalUnitId}', organizationalUnitId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(masterAccountId, organizationId, policyType, policyId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:policy/o-${OrganizationId}/${PolicyType}/p-${PolicyId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${PolicyType}', policyType);
        arn = arn.replace('${PolicyId}', policyId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type awspolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAwspolicy(policyType, policyId, partition) {
        var arn = 'arn:${Partition}:organizations::aws:policy/${PolicyType}/p-${PolicyId}';
        arn = arn.replace('${PolicyType}', policyType);
        arn = arn.replace('${PolicyId}', policyId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type root to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_reference_arn-formats.html
     *
     * @param masterAccountId - Identifier for the masterAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param rootId - Identifier for the rootId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoot(masterAccountId, organizationId, rootId, partition) {
        var arn = 'arn:${Partition}:organizations::${MasterAccountId}:root/o-${OrganizationId}/r-${RootId}';
        arn = arn.replace('${MasterAccountId}', masterAccountId);
        arn = arn.replace('${OrganizationId}', organizationId);
        arn = arn.replace('${RootId}', rootId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to filter the request to only the specified policy type names.
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toAttachPolicy()
     * - .toCreatePolicy()
     * - .toDeletePolicy()
     * - .toDescribeEffectivePolicy()
     * - .toDescribePolicy()
     * - .toDetachPolicy()
     * - .toDisablePolicyType()
     * - .toEnablePolicyType()
     * - .toListPolicies()
     * - .toListPoliciesForTarget()
     * - .toListTargetsForPolicy()
     * - .toUpdatePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPolicyType(value, operator) {
        return this.if(`organizations:PolicyType`, value, operator || 'StringLike');
    }
    /**
     * Enables you to filter the request to only the specified service principal names.
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toDeregisterDelegatedAdministrator()
     * - .toDisableAWSServiceAccess()
     * - .toEnableAWSServiceAccess()
     * - .toListDelegatedAdministrators()
     * - .toRegisterDelegatedAdministrator()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServicePrincipal(value, operator) {
        return this.if(`organizations:ServicePrincipal`, value, operator || 'StringLike');
    }
}
exports.Organizations = Organizations;
//# sourceMappingURL=data:application/json;base64,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