"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securityhub = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [securityhub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecurityhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securityhub extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [securityhub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecurityhub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'securityhub';
        this.accessLevelList = {
            "Write": [
                "AcceptInvitation",
                "BatchDisableStandards",
                "BatchEnableStandards",
                "BatchImportFindings",
                "BatchUpdateFindings",
                "CreateActionTarget",
                "CreateInsight",
                "CreateMembers",
                "DeclineInvitations",
                "DeleteActionTarget",
                "DeleteInsight",
                "DeleteInvitations",
                "DeleteMembers",
                "DisableImportFindingsForProduct",
                "DisableSecurityHub",
                "DisassociateFromMasterAccount",
                "DisassociateMembers",
                "EnableImportFindingsForProduct",
                "EnableSecurityHub",
                "InviteMembers",
                "TagResource",
                "UntagResource",
                "UpdateActionTarget",
                "UpdateFindings",
                "UpdateInsight",
                "UpdateSecurityHubConfiguration",
                "UpdateStandardsControl"
            ],
            "Read": [
                "DescribeActionTargets",
                "DescribeHub",
                "DescribeProducts",
                "DescribeStandards",
                "DescribeStandardsControls",
                "GetFindings",
                "GetInsightResults",
                "GetInvitationsCount",
                "GetMasterAccount",
                "GetMembers"
            ],
            "List": [
                "GetEnabledStandards",
                "GetInsights",
                "ListEnabledProductsForImport",
                "ListInvitations",
                "ListMembers",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        this.to('securityhub:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to disable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html
     */
    toBatchDisableStandards() {
        this.to('securityhub:BatchDisableStandards');
        return this;
    }
    /**
     * Grants permission to enable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html
     */
    toBatchEnableStandards() {
        this.to('securityhub:BatchEnableStandards');
        return this;
    }
    /**
     * Grants permission to import findings into Security Hub from an integrated product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetAccount()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html
     */
    toBatchImportFindings() {
        this.to('securityhub:BatchImportFindings');
        return this;
    }
    /**
     * Grants permission to update customer-controlled fields for a selected set of Security Hub findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifASFFSyntaxPath()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html
     */
    toBatchUpdateFindings() {
        this.to('securityhub:BatchUpdateFindings');
        return this;
    }
    /**
     * Grants permission to create custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html
     */
    toCreateActionTarget() {
        this.to('securityhub:CreateActionTarget');
        return this;
    }
    /**
     * Grants permission to create insights in Security Hub. Insights are collections of related findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html
     */
    toCreateInsight() {
        this.to('securityhub:CreateInsight');
        return this;
    }
    /**
     * Grants permission to create member accounts in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        this.to('securityhub:CreateMembers');
        return this;
    }
    /**
     * Grants permission to decline Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        this.to('securityhub:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html
     */
    toDeleteActionTarget() {
        this.to('securityhub:DeleteActionTarget');
        return this;
    }
    /**
     * Grants permission to delete insights from Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html
     */
    toDeleteInsight() {
        this.to('securityhub:DeleteInsight');
        return this;
    }
    /**
     * Grants permission to delete Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        this.to('securityhub:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        this.to('securityhub:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of custom actions using the API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html
     */
    toDescribeActionTargets() {
        this.to('securityhub:DescribeActionTargets');
        return this;
    }
    /**
     * Grants permission to retrieve information about the hub resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html
     */
    toDescribeHub() {
        this.to('securityhub:DescribeHub');
        return this;
    }
    /**
     * Grants permission to retrieve information about the available Security Hub product integrations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html
     */
    toDescribeProducts() {
        this.to('securityhub:DescribeProducts');
        return this;
    }
    /**
     * Grants permission to retrieve information about Security Hub standards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html
     */
    toDescribeStandards() {
        this.to('securityhub:DescribeStandards');
        return this;
    }
    /**
     * Grants permission to retrieve information about Security Hub standards controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html
     */
    toDescribeStandardsControls() {
        this.to('securityhub:DescribeStandardsControls');
        return this;
    }
    /**
     * Grants permission to disable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html
     */
    toDisableImportFindingsForProduct() {
        this.to('securityhub:DisableImportFindingsForProduct');
        return this;
    }
    /**
     * Grants permission to disable Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html
     */
    toDisableSecurityHub() {
        this.to('securityhub:DisableSecurityHub');
        return this;
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        this.to('securityhub:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants permission to disassociate Security Hub member accounts from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        this.to('securityhub:DisassociateMembers');
        return this;
    }
    /**
     * Grants permission to enable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html
     */
    toEnableImportFindingsForProduct() {
        this.to('securityhub:EnableImportFindingsForProduct');
        return this;
    }
    /**
     * Grants permission to enable Security Hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html
     */
    toEnableSecurityHub() {
        this.to('securityhub:EnableSecurityHub');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the standards that are enabled in Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html
     */
    toGetEnabledStandards() {
        this.to('securityhub:GetEnabledStandards');
        return this;
    }
    /**
     * Grants permission to retrieve a list of findings from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        this.to('securityhub:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve insight results from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html
     */
    toGetInsightResults() {
        this.to('securityhub:GetInsightResults');
        return this;
    }
    /**
     * Grants permission to retrieve Security Hub insights
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html
     */
    toGetInsights() {
        this.to('securityhub:GetInsights');
        return this;
    }
    /**
     * Grants permission to retrieve the count of Security Hub membership invitations sent to the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        this.to('securityhub:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve details about the Security Hub master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        this.to('securityhub:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the details of Security Hub member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        this.to('securityhub:GetMembers');
        return this;
    }
    /**
     * Grants permission to invite other AWS accounts to become Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        this.to('securityhub:InviteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve the Security Hub integrated products that are currently enabled
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html
     */
    toListEnabledProductsForImport() {
        this.to('securityhub:ListEnabledProductsForImport');
        return this;
    }
    /**
     * Grants permission to retrieve the Security Hub invitations sent to the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        this.to('securityhub:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve details about Security Hub member accounts associated with the master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html
     */
    toListMembers() {
        this.to('securityhub:ListMembers');
        return this;
    }
    /**
     * Grants permission to list of tags associated with a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('securityhub:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to add tags to a Security Hub resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('securityhub:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a Security Hub resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('securityhub:UntagResource');
        return this;
    }
    /**
     * Grants permission to update custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html
     */
    toUpdateActionTarget() {
        this.to('securityhub:UpdateActionTarget');
        return this;
    }
    /**
     * Grants permission to update Security Hub findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        this.to('securityhub:UpdateFindings');
        return this;
    }
    /**
     * Grants permission to update insights in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html
     */
    toUpdateInsight() {
        this.to('securityhub:UpdateInsight');
        return this;
    }
    /**
     * Grants permission to update Security Hub configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityHubConfiguration.html
     */
    toUpdateSecurityHubConfiguration() {
        this.to('securityhub:UpdateSecurityHubConfiguration');
        return this;
    }
    /**
     * Grants permission to update Security Hub standards controls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html
     */
    toUpdateStandardsControl() {
        this.to('securityhub:UpdateStandardsControl');
        return this;
    }
    /**
     * Adds a resource of type hub to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHub(account, region, partition) {
        var arn = 'arn:${Partition}:securityhub:${Region}:${Account}:hub/default';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type product to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param company - Identifier for the company.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProduct(company, productId, account, region, partition) {
        var arn = 'arn:${Partition}:securityhub:${Region}:${Account}:product/${Company}/${ProductId}';
        arn = arn.replace('${Company}', company);
        arn = arn.replace('${ProductId}', productId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access based on the presence of specific fields and values in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-asffsyntaxpath
     *
     * Applies to actions:
     * - .toBatchUpdateFindings()
     *
     * @param aSFFSyntaxPath The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifASFFSyntaxPath(aSFFSyntaxPath, value, operator) {
        return this.if(`securityhub:ASFFSyntaxPath/${aSFFSyntaxPath}`, value, operator || 'StringLike');
    }
    /**
     * Filters access based on the presence of AwsAccountId field in the requests
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#conditions
     *
     * Applies to actions:
     * - .toBatchImportFindings()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetAccount(value, operator) {
        return this.if(`securityhub:TargetAccount`, value, operator || 'StringLike');
    }
}
exports.Securityhub = Securityhub;
//# sourceMappingURL=data:application/json;base64,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