"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicecatalog = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsservicecatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicecatalog extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsservicecatalog.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicecatalog';
        this.accessLevelList = {
            "Write": [
                "AcceptPortfolioShare",
                "AssociateBudgetWithResource",
                "AssociatePrincipalWithPortfolio",
                "AssociateProductWithPortfolio",
                "AssociateServiceActionWithProvisioningArtifact",
                "AssociateTagOptionWithResource",
                "BatchAssociateServiceActionWithProvisioningArtifact",
                "BatchDisassociateServiceActionFromProvisioningArtifact",
                "CopyProduct",
                "CreateConstraint",
                "CreatePortfolio",
                "CreateProduct",
                "CreateProvisionedProductPlan",
                "CreateProvisioningArtifact",
                "CreateServiceAction",
                "CreateTagOption",
                "DeleteConstraint",
                "DeletePortfolio",
                "DeleteProduct",
                "DeleteProvisionedProductPlan",
                "DeleteProvisioningArtifact",
                "DeleteServiceAction",
                "DeleteTagOption",
                "DisableAWSOrganizationsAccess",
                "DisassociateBudgetFromResource",
                "DisassociatePrincipalFromPortfolio",
                "DisassociateProductFromPortfolio",
                "DisassociateServiceActionFromProvisioningArtifact",
                "DisassociateTagOptionFromResource",
                "EnableAWSOrganizationsAccess",
                "ExecuteProvisionedProductPlan",
                "ExecuteProvisionedProductServiceAction",
                "ProvisionProduct",
                "RejectPortfolioShare",
                "TerminateProvisionedProduct",
                "UpdateConstraint",
                "UpdatePortfolio",
                "UpdateProduct",
                "UpdateProvisionedProduct",
                "UpdateProvisionedProductProperties",
                "UpdateProvisioningArtifact",
                "UpdateServiceAction",
                "UpdateTagOption"
            ],
            "Permissions management": [
                "CreatePortfolioShare",
                "DeletePortfolioShare"
            ],
            "Read": [
                "DescribeConstraint",
                "DescribeCopyProductStatus",
                "DescribePortfolio",
                "DescribePortfolioShareStatus",
                "DescribeProduct",
                "DescribeProductAsAdmin",
                "DescribeProductView",
                "DescribeProvisionedProduct",
                "DescribeProvisionedProductPlan",
                "DescribeProvisioningArtifact",
                "DescribeProvisioningParameters",
                "DescribeRecord",
                "DescribeServiceAction",
                "DescribeServiceActionExecutionParameters",
                "DescribeTagOption",
                "GetAWSOrganizationsAccessStatus"
            ],
            "List": [
                "ListAcceptedPortfolioShares",
                "ListBudgetsForResource",
                "ListConstraintsForPortfolio",
                "ListLaunchPaths",
                "ListOrganizationPortfolioAccess",
                "ListPortfolioAccess",
                "ListPortfolios",
                "ListPortfoliosForProduct",
                "ListPrincipalsForPortfolio",
                "ListProvisionedProductPlans",
                "ListProvisioningArtifacts",
                "ListProvisioningArtifactsForServiceAction",
                "ListRecordHistory",
                "ListResourcesForTagOption",
                "ListServiceActions",
                "ListServiceActionsForProvisioningArtifact",
                "ListStackInstancesForProvisionedProduct",
                "ListTagOptions",
                "ScanProvisionedProducts",
                "SearchProducts",
                "SearchProductsAsAdmin",
                "SearchProvisionedProducts"
            ]
        };
    }
    /**
     * Grants permission to accept a portfolio that has been shared with you
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AcceptPortfolioShare.html
     */
    toAcceptPortfolioShare() {
        this.to('servicecatalog:AcceptPortfolioShare');
        return this;
    }
    /**
     * Grants permission to associate a budget with a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateBudgetWithResource.html
     */
    toAssociateBudgetWithResource() {
        this.to('servicecatalog:AssociateBudgetWithResource');
        return this;
    }
    /**
     * Grants permission to associate an IAM principal with a portfolio, giving the specified principal access to any products associated with the specified portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociatePrincipalWithPortfolio.html
     */
    toAssociatePrincipalWithPortfolio() {
        this.to('servicecatalog:AssociatePrincipalWithPortfolio');
        return this;
    }
    /**
     * Grants permission to associate a product with a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateProductWithPortfolio.html
     */
    toAssociateProductWithPortfolio() {
        this.to('servicecatalog:AssociateProductWithPortfolio');
        return this;
    }
    /**
     * Grants permission to associate an action with a provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateServiceActionWithProvisioningArtifact.html
     */
    toAssociateServiceActionWithProvisioningArtifact() {
        this.to('servicecatalog:AssociateServiceActionWithProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to associate the specified TagOption with the specified portfolio or product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateTagOptionWithResource.html
     */
    toAssociateTagOptionWithResource() {
        this.to('servicecatalog:AssociateTagOptionWithResource');
        return this;
    }
    /**
     * Grants permission to associate multiple self-service actions with provisioning artifacts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchAssociateServiceActionWithProvisioningArtifact.html
     */
    toBatchAssociateServiceActionWithProvisioningArtifact() {
        this.to('servicecatalog:BatchAssociateServiceActionWithProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to disassociate a batch of self-service actions from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchDisassociateServiceActionFromProvisioningArtifact.html
     */
    toBatchDisassociateServiceActionFromProvisioningArtifact() {
        this.to('servicecatalog:BatchDisassociateServiceActionFromProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to copy the specified source product to the specified target product or a new product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CopyProduct.html
     */
    toCopyProduct() {
        this.to('servicecatalog:CopyProduct');
        return this;
    }
    /**
     * Grants permission to create a constraint on an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateConstraint.html
     */
    toCreateConstraint() {
        this.to('servicecatalog:CreateConstraint');
        return this;
    }
    /**
     * Grants permission to create a portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolio.html
     */
    toCreatePortfolio() {
        this.to('servicecatalog:CreatePortfolio');
        return this;
    }
    /**
     * Grants permission to share a portfolio you own with another AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolioShare.html
     */
    toCreatePortfolioShare() {
        this.to('servicecatalog:CreatePortfolioShare');
        return this;
    }
    /**
     * Grants permission to create a product and that product's first provisioning artifact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProduct.html
     */
    toCreateProduct() {
        this.to('servicecatalog:CreateProduct');
        return this;
    }
    /**
     * Grants permission to add a new provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisionedProductPlan.html
     */
    toCreateProvisionedProductPlan() {
        this.to('servicecatalog:CreateProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to add a new provisioning artifact to an existing product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisioningArtifact.html
     */
    toCreateProvisioningArtifact() {
        this.to('servicecatalog:CreateProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to create a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateServiceAction.html
     */
    toCreateServiceAction() {
        this.to('servicecatalog:CreateServiceAction');
        return this;
    }
    /**
     * Grants permission to create a TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateTagOption.html
     */
    toCreateTagOption() {
        this.to('servicecatalog:CreateTagOption');
        return this;
    }
    /**
     * Grants permission to remove and delete an existing constraint from an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteConstraint.html
     */
    toDeleteConstraint() {
        this.to('servicecatalog:DeleteConstraint');
        return this;
    }
    /**
     * Grants permission to delete a portfolio if all associations and shares have been removed from the portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolio.html
     */
    toDeletePortfolio() {
        this.to('servicecatalog:DeletePortfolio');
        return this;
    }
    /**
     * Grants permission to unshare a portfolio you own from an AWS account you previously shared the portfolio with
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolioShare.html
     */
    toDeletePortfolioShare() {
        this.to('servicecatalog:DeletePortfolioShare');
        return this;
    }
    /**
     * Grants permission to delete a product if all associations have been removed from the product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProduct.html
     */
    toDeleteProduct() {
        this.to('servicecatalog:DeleteProduct');
        return this;
    }
    /**
     * Grants permission to delete a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisionedProductPlan.html
     */
    toDeleteProvisionedProductPlan() {
        this.to('servicecatalog:DeleteProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to delete a provisioning artifact from a product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisioningArtifact.html
     */
    toDeleteProvisioningArtifact() {
        this.to('servicecatalog:DeleteProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to delete a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteServiceAction.html
     */
    toDeleteServiceAction() {
        this.to('servicecatalog:DeleteServiceAction');
        return this;
    }
    /**
     * Grants permission to delete the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteTagOption.html
     */
    toDeleteTagOption() {
        this.to('servicecatalog:DeleteTagOption');
        return this;
    }
    /**
     * Grants permission to describe a constraint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeConstraint.html
     */
    toDescribeConstraint() {
        this.to('servicecatalog:DescribeConstraint');
        return this;
    }
    /**
     * Grants permission to get the status of the specified copy product operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeCopyProductStatus.html
     */
    toDescribeCopyProductStatus() {
        this.to('servicecatalog:DescribeCopyProductStatus');
        return this;
    }
    /**
     * Grants permission to describe a portfolio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolio.html
     */
    toDescribePortfolio() {
        this.to('servicecatalog:DescribePortfolio');
        return this;
    }
    /**
     * Grants permission to get the status of the specified portfolio share operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolioShareStatus.html
     */
    toDescribePortfolioShareStatus() {
        this.to('servicecatalog:DescribePortfolioShareStatus');
        return this;
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProduct.html
     */
    toDescribeProduct() {
        this.to('servicecatalog:DescribeProduct');
        return this;
    }
    /**
     * Grants permission to describe a product as an admin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductAsAdmin.html
     */
    toDescribeProductAsAdmin() {
        this.to('servicecatalog:DescribeProductAsAdmin');
        return this;
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductView.html
     */
    toDescribeProductView() {
        this.to('servicecatalog:DescribeProductView');
        return this;
    }
    /**
     * Grants permission to describe a provisioned product
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProduct.html
     */
    toDescribeProvisionedProduct() {
        this.to('servicecatalog:DescribeProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to describe a provisioned product plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProductPlan.html
     */
    toDescribeProvisionedProductPlan() {
        this.to('servicecatalog:DescribeProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to describe a provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningArtifact.html
     */
    toDescribeProvisioningArtifact() {
        this.to('servicecatalog:DescribeProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to describe the parameters that you need to specify to successfully provision a specified provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningParameters.html
     */
    toDescribeProvisioningParameters() {
        this.to('servicecatalog:DescribeProvisioningParameters');
        return this;
    }
    /**
     * Grants permission to describe a record and lists any outputs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeRecord.html
     */
    toDescribeRecord() {
        this.to('servicecatalog:DescribeRecord');
        return this;
    }
    /**
     * Grants permission to describe a self-service action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceAction.html
     */
    toDescribeServiceAction() {
        this.to('servicecatalog:DescribeServiceAction');
        return this;
    }
    /**
     * Grants permission to get the default parameters if you executed the specified Service Action on the specified Provisioned Product
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceActionExecutionParameters.html
     */
    toDescribeServiceActionExecutionParameters() {
        this.to('servicecatalog:DescribeServiceActionExecutionParameters');
        return this;
    }
    /**
     * Grants permission to get information about the specified TagOption
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeTagOption.html
     */
    toDescribeTagOption() {
        this.to('servicecatalog:DescribeTagOption');
        return this;
    }
    /**
     * Grants permission to disable portfolio sharing through AWS Organizations feature
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisableAWSOrganizationsAccess.html
     */
    toDisableAWSOrganizationsAccess() {
        this.to('servicecatalog:DisableAWSOrganizationsAccess');
        return this;
    }
    /**
     * Grants permission to disassociate a budget from a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateBudgetFromResource.html
     */
    toDisassociateBudgetFromResource() {
        this.to('servicecatalog:DisassociateBudgetFromResource');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM principal from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociatePrincipalFromPortfolio.html
     */
    toDisassociatePrincipalFromPortfolio() {
        this.to('servicecatalog:DisassociatePrincipalFromPortfolio');
        return this;
    }
    /**
     * Grants permission to disassociate a product from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateProductFromPortfolio.html
     */
    toDisassociateProductFromPortfolio() {
        this.to('servicecatalog:DisassociateProductFromPortfolio');
        return this;
    }
    /**
     * Grants permission to disassociate the specified self-service action association from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateServiceActionFromProvisioningArtifact.html
     */
    toDisassociateServiceActionFromProvisioningArtifact() {
        this.to('servicecatalog:DisassociateServiceActionFromProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to disassociate the specified TagOption from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateTagOptionFromResource.html
     */
    toDisassociateTagOptionFromResource() {
        this.to('servicecatalog:DisassociateTagOptionFromResource');
        return this;
    }
    /**
     * Grants permission to enable portfolio sharing feature through AWS Organizations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_EnableAWSOrganizationsAccess.html
     */
    toEnableAWSOrganizationsAccess() {
        this.to('servicecatalog:EnableAWSOrganizationsAccess');
        return this;
    }
    /**
     * Grants permission to execute a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductPlan.html
     */
    toExecuteProvisionedProductPlan() {
        this.to('servicecatalog:ExecuteProvisionedProductPlan');
        return this;
    }
    /**
     * Grants permission to executes a provisioned product plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductServiceAction.html
     */
    toExecuteProvisionedProductServiceAction() {
        this.to('servicecatalog:ExecuteProvisionedProductServiceAction');
        return this;
    }
    /**
     * Grants permission to get the access status of AWS Organization portfolio share feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetAWSOrganizationsAccessStatus.html
     */
    toGetAWSOrganizationsAccessStatus() {
        this.to('servicecatalog:GetAWSOrganizationsAccessStatus');
        return this;
    }
    /**
     * Grants permission to list the portfolios that have been shared with you and you have accepted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAcceptedPortfolioShares.html
     */
    toListAcceptedPortfolioShares() {
        this.to('servicecatalog:ListAcceptedPortfolioShares');
        return this;
    }
    /**
     * Grants permission to list all the budgets associated to a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListBudgetsForResource.html
     */
    toListBudgetsForResource() {
        this.to('servicecatalog:ListBudgetsForResource');
        return this;
    }
    /**
     * Grants permission to list constraints associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListConstraintsForPortfolio.html
     */
    toListConstraintsForPortfolio() {
        this.to('servicecatalog:ListConstraintsForPortfolio');
        return this;
    }
    /**
     * Grants permission to list the different ways to launch a given product as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListLaunchPaths.html
     */
    toListLaunchPaths() {
        this.to('servicecatalog:ListLaunchPaths');
        return this;
    }
    /**
     * Grants permission to list the organization nodes that have access to the specified portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListOrganizationPortfolioAccess.html
     */
    toListOrganizationPortfolioAccess() {
        this.to('servicecatalog:ListOrganizationPortfolioAccess');
        return this;
    }
    /**
     * Grants permission to list the AWS accounts you have shared a given portfolio with
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolioAccess.html
     */
    toListPortfolioAccess() {
        this.to('servicecatalog:ListPortfolioAccess');
        return this;
    }
    /**
     * Grants permission to list the portfolios in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolios.html
     */
    toListPortfolios() {
        this.to('servicecatalog:ListPortfolios');
        return this;
    }
    /**
     * Grants permission to list the portfolios associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfoliosForProduct.html
     */
    toListPortfoliosForProduct() {
        this.to('servicecatalog:ListPortfoliosForProduct');
        return this;
    }
    /**
     * Grants permission to list the IAM principals associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPrincipalsForPortfolio.html
     */
    toListPrincipalsForPortfolio() {
        this.to('servicecatalog:ListPrincipalsForPortfolio');
        return this;
    }
    /**
     * Grants permission to list the provisioned product plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisionedProductPlans.html
     */
    toListProvisionedProductPlans() {
        this.to('servicecatalog:ListProvisionedProductPlans');
        return this;
    }
    /**
     * Grants permission to list the provisioning artifacts associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifacts.html
     */
    toListProvisioningArtifacts() {
        this.to('servicecatalog:ListProvisioningArtifacts');
        return this;
    }
    /**
     * Grants permission to list all provisioning artifacts for the specified self-service action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifactsForServiceAction.html
     */
    toListProvisioningArtifactsForServiceAction() {
        this.to('servicecatalog:ListProvisioningArtifactsForServiceAction');
        return this;
    }
    /**
     * Grants permission to list all the records in your account or all the records related to a given provisioned product
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListRecordHistory.html
     */
    toListRecordHistory() {
        this.to('servicecatalog:ListRecordHistory');
        return this;
    }
    /**
     * Grants permission to list the resources associated with the specified TagOption
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListResourcesForTagOption.html
     */
    toListResourcesForTagOption() {
        this.to('servicecatalog:ListResourcesForTagOption');
        return this;
    }
    /**
     * Grants permission to list all self-service actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActions.html
     */
    toListServiceActions() {
        this.to('servicecatalog:ListServiceActions');
        return this;
    }
    /**
     * Grants permission to list all the service actions associated with the specified provisioning artifact in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActionsForProvisioningArtifact.html
     */
    toListServiceActionsForProvisioningArtifact() {
        this.to('servicecatalog:ListServiceActionsForProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to list account, region and status of each stack instances that are associated with a CFN_STACKSET type provisioned product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListStackInstancesForProvisionedProduct.html
     */
    toListStackInstancesForProvisionedProduct() {
        this.to('servicecatalog:ListStackInstancesForProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to list the specified TagOptions or all TagOptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListTagOptions.html
     */
    toListTagOptions() {
        this.to('servicecatalog:ListTagOptions');
        return this;
    }
    /**
     * Grants permission to provision a product with a specified provisioning artifact and launch parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProvisionProduct.html
     */
    toProvisionProduct() {
        this.to('servicecatalog:ProvisionProduct');
        return this;
    }
    /**
     * Grants permission to reject a portfolio that has been shared with you that you previously accepted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_RejectPortfolioShare.html
     */
    toRejectPortfolioShare() {
        this.to('servicecatalog:RejectPortfolioShare');
        return this;
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ScanProvisionedProducts.html
     */
    toScanProvisionedProducts() {
        this.to('servicecatalog:ScanProvisionedProducts');
        return this;
    }
    /**
     * Grants permission to list the products available to you as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProducts.html
     */
    toSearchProducts() {
        this.to('servicecatalog:SearchProducts');
        return this;
    }
    /**
     * Grants permission to list all the products in your account or all the products associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProductsAsAdmin.html
     */
    toSearchProductsAsAdmin() {
        this.to('servicecatalog:SearchProductsAsAdmin');
        return this;
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProvisionedProducts.html
     */
    toSearchProvisionedProducts() {
        this.to('servicecatalog:SearchProvisionedProducts');
        return this;
    }
    /**
     * Grants permission to terminate an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_TerminateProvisionedProduct.html
     */
    toTerminateProvisionedProduct() {
        this.to('servicecatalog:TerminateProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to update the metadata fields of an existing constraint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateConstraint.html
     */
    toUpdateConstraint() {
        this.to('servicecatalog:UpdateConstraint');
        return this;
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdatePortfolio.html
     */
    toUpdatePortfolio() {
        this.to('servicecatalog:UpdatePortfolio');
        return this;
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProduct.html
     */
    toUpdateProduct() {
        this.to('servicecatalog:UpdateProduct');
        return this;
    }
    /**
     * Grants permission to update an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProduct.html
     */
    toUpdateProvisionedProduct() {
        this.to('servicecatalog:UpdateProvisionedProduct');
        return this;
    }
    /**
     * Grants permission to update the properties of an existing provisioned product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProductProperties.html
     */
    toUpdateProvisionedProductProperties() {
        this.to('servicecatalog:UpdateProvisionedProductProperties');
        return this;
    }
    /**
     * Grants permission to update the metadata fields of an existing provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisioningArtifact.html
     */
    toUpdateProvisioningArtifact() {
        this.to('servicecatalog:UpdateProvisioningArtifact');
        return this;
    }
    /**
     * Grants permission to update a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateServiceAction.html
     */
    toUpdateServiceAction() {
        this.to('servicecatalog:UpdateServiceAction');
        return this;
    }
    /**
     * Grants permission to update the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateTagOption.html
     */
    toUpdateTagOption() {
        this.to('servicecatalog:UpdateTagOption');
        return this;
    }
    /**
     * Adds a resource of type Portfolio to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_PortfolioDetail.html
     *
     * @param portfolioId - Identifier for the portfolioId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortfolio(portfolioId, account, region, partition) {
        var arn = 'arn:${Partition}:catalog:${Region}:${Account}:portfolio/${PortfolioId}';
        arn = arn.replace('${PortfolioId}', portfolioId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Product to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProductViewDetail.html
     *
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProduct(productId, account, region, partition) {
        var arn = 'arn:${Partition}:catalog:${Region}:${Account}:product/${ProductId}';
        arn = arn.replace('${ProductId}', productId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters users to see and perform actions on resources created by anyone in the account
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountLevel(value, operator) {
        return this.if(`servicecatalog:accountLevel`, value, operator || 'StringLike');
    }
    /**
     * Filters users to see and perform actions on resources created either by them or by anyone federating into the same role as them
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoleLevel(value, operator) {
        return this.if(`servicecatalog:roleLevel`, value, operator || 'StringLike');
    }
    /**
     * Filters users to see and perform actions on only resources that they created
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toDescribeRecord()
     * - .toListRecordHistory()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserLevel(value, operator) {
        return this.if(`servicecatalog:userLevel`, value, operator || 'StringLike');
    }
}
exports.Servicecatalog = Servicecatalog;
//# sourceMappingURL=data:application/json;base64,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