"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Timestream = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [timestream](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstimestream.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Timestream extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [timestream](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstimestream.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'timestream';
        this.accessLevelList = {
            "Write": [
                "CancelQuery",
                "CreateDatabase",
                "CreateTable",
                "DeleteDatabase",
                "DeleteTable",
                "UpdateDatabase",
                "UpdateTable",
                "WriteRecords"
            ],
            "Read": [
                "DescribeDatabase",
                "DescribeTable",
                "Select",
                "SelectValues"
            ],
            "List": [
                "DescribeEndpoints",
                "ListDatabases",
                "ListMeasures",
                "ListTables",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants Permission to cancel queries in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_CancelQuery.html
     */
    toCancelQuery() {
        this.to('timestream:CancelQuery');
        return this;
    }
    /**
     * Grants permissions to create a database in your account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_CreateDatabase.html
     */
    toCreateDatabase() {
        this.to('timestream:CreateDatabase');
        return this;
    }
    /**
     * Grants permissions to create a table in your account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_CreateTable.html
     */
    toCreateTable() {
        this.to('timestream:CreateTable');
        return this;
    }
    /**
     * Grants permissions to delete a database in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_DeleteDatabase.html
     */
    toDeleteDatabase() {
        this.to('timestream:DeleteDatabase');
        return this;
    }
    /**
     * Grants permissions to delete a table in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_DeleteTable.html
     */
    toDeleteTable() {
        this.to('timestream:DeleteTable');
        return this;
    }
    /**
     * Grants permissions to describe a database in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_DescribeDatabase.html
     */
    toDescribeDatabase() {
        this.to('timestream:DescribeDatabase');
        return this;
    }
    /**
     * Grants permissions to describe timestream endpoints.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints() {
        this.to('timestream:DescribeEndpoints');
        return this;
    }
    /**
     * Grants Permissions to describe a table in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_DescribeTable.html
     */
    toDescribeTable() {
        this.to('timestream:DescribeTable');
        return this;
    }
    /**
     * Grants Permission to list databases in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_ListDatabases.html
     */
    toListDatabases() {
        this.to('timestream:ListDatabases');
        return this;
    }
    /**
     * Grants Permissions to list measures of a table in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_ListMeasures.html
     */
    toListMeasures() {
        this.to('timestream:ListMeasures');
        return this;
    }
    /**
     * Grants Permission to list tables in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_ListTables.html
     */
    toListTables() {
        this.to('timestream:ListTables');
        return this;
    }
    /**
     * Grants permissions to list tags of a resource in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('timestream:ListTagsForResource');
        return this;
    }
    /**
     * Grants Permission to issue 'select from table' queries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_Query.html
     */
    toSelect() {
        this.to('timestream:Select');
        return this;
    }
    /**
     * Grants Permissions to issue 'select 1' queries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_Query.html
     */
    toSelectValues() {
        this.to('timestream:SelectValues');
        return this;
    }
    /**
     * Grants permissions to add tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('timestream:TagResource');
        return this;
    }
    /**
     * Grants permissions to remove a tag from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('timestream:UntagResource');
        return this;
    }
    /**
     * Grants permissions to update a database in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_UpdateDatabase.html
     */
    toUpdateDatabase() {
        this.to('timestream:UpdateDatabase');
        return this;
    }
    /**
     * Grants permissions to update a table in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_UpdateTable.html
     */
    toUpdateTable() {
        this.to('timestream:UpdateTable');
        return this;
    }
    /**
     * Grants permissions to ingest data to a table in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Timestream/latest/APIReference/API_WriteRecords.html
     */
    toWriteRecords() {
        this.to('timestream:WriteRecords');
        return this;
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/Timestream/latest/UserGuide/database-example-policies.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatabase(databaseName, account, region, partition) {
        var arn = 'arn:${Partition}:timestream:${Region}:${Account}:database/${DatabaseName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/Timestream/latest/UserGuide/table-example-policies.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTable(databaseName, tableName, account, region, partition) {
        var arn = 'arn:${Partition}:timestream:${Region}:${Account}:database/${DatabaseName}/table/${TableName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Timestream = Timestream;
//# sourceMappingURL=data:application/json;base64,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