"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Xray = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [xray](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsx-ray.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Xray extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [xray](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsx-ray.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'xray';
        this.accessLevelList = {
            "List": [
                "BatchGetTraces",
                "ListTagsForResource"
            ],
            "Write": [
                "CreateGroup",
                "CreateSamplingRule",
                "DeleteGroup",
                "DeleteSamplingRule",
                "PutTelemetryRecords",
                "PutTraceSegments",
                "UpdateGroup",
                "UpdateSamplingRule"
            ],
            "Permissions management": [
                "GetEncryptionConfig",
                "PutEncryptionConfig"
            ],
            "Read": [
                "GetGroup",
                "GetGroups",
                "GetInsight",
                "GetInsightEvents",
                "GetInsightImpactGraph",
                "GetInsightSummaries",
                "GetSamplingRules",
                "GetSamplingStatisticSummaries",
                "GetSamplingTargets",
                "GetServiceGraph",
                "GetTimeSeriesServiceStatistics",
                "GetTraceGraph",
                "GetTraceSummaries"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to retrieve a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html
     */
    toBatchGetTraces() {
        this.to('xray:BatchGetTraces');
        return this;
    }
    /**
     * Grants permission to create a group resource with a name and a filter expression
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html
     */
    toCreateGroup() {
        this.to('xray:CreateGroup');
        return this;
    }
    /**
     * Grants permission to create a rule to control sampling behavior for instrumented applications
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html
     */
    toCreateSamplingRule() {
        this.to('xray:CreateSamplingRule');
        return this;
    }
    /**
     * Grants permission to delete a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html
     */
    toDeleteGroup() {
        this.to('xray:DeleteGroup');
        return this;
    }
    /**
     * Grants permission to delete a sampling rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html
     */
    toDeleteSamplingRule() {
        this.to('xray:DeleteSamplingRule');
        return this;
    }
    /**
     * Grants permission to retrieve the current encryption configuration for X-Ray data
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html
     */
    toGetEncryptionConfig() {
        this.to('xray:GetEncryptionConfig');
        return this;
    }
    /**
     * Grants permission to retrieve group resource details
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html
     */
    toGetGroup() {
        this.to('xray:GetGroup');
        return this;
    }
    /**
     * Grants permission to retrieve all active group details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html
     */
    toGetGroups() {
        this.to('xray:GetGroups');
        return this;
    }
    /**
     * Grants permission to retrieve the details of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsight.html
     */
    toGetInsight() {
        this.to('xray:GetInsight');
        return this;
    }
    /**
     * Grants permission to retrieve the events of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightEvents.html
     */
    toGetInsightEvents() {
        this.to('xray:GetInsightEvents');
        return this;
    }
    /**
     * Grants permission to retrieve the part of the service graph which is impacted for a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightImpactGraph.html
     */
    toGetInsightImpactGraph() {
        this.to('xray:GetInsightImpactGraph');
        return this;
    }
    /**
     * Grants permission to retrieve the summary of all insights for a group and time range with optional filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightSummaries.html
     */
    toGetInsightSummaries() {
        this.to('xray:GetInsightSummaries');
        return this;
    }
    /**
     * Grants permission to retrieve all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html
     */
    toGetSamplingRules() {
        this.to('xray:GetSamplingRules');
        return this;
    }
    /**
     * Grants permission to retrieve information about recent sampling results for all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html
     */
    toGetSamplingStatisticSummaries() {
        this.to('xray:GetSamplingStatisticSummaries');
        return this;
    }
    /**
     * Grants permission to request a sampling quota for rules that the service is using to sample requests
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html
     */
    toGetSamplingTargets() {
        this.to('xray:GetSamplingTargets');
        return this;
    }
    /**
     * Grants permission to retrieve a document that describes services that process incoming requests, and downstream services that they call as a result
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html
     */
    toGetServiceGraph() {
        this.to('xray:GetServiceGraph');
        return this;
    }
    /**
     * Grants permission to retrieve an aggregation of service statistics defined by a specific time range bucketed into time intervals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html
     */
    toGetTimeSeriesServiceStatistics() {
        this.to('xray:GetTimeSeriesServiceStatistics');
        return this;
    }
    /**
     * Grants permission to retrieve a service graph for one or more specific trace IDs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html
     */
    toGetTraceGraph() {
        this.to('xray:GetTraceGraph');
        return this;
    }
    /**
     * Grants permission to retrieve IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html
     */
    toGetTraceSummaries() {
        this.to('xray:GetTraceSummaries');
        return this;
    }
    /**
     * Grants permission to list tags for an X-Ray resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('xray:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to update the encryption configuration for X-Ray data
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html
     */
    toPutEncryptionConfig() {
        this.to('xray:PutEncryptionConfig');
        return this;
    }
    /**
     * Grants permission to send AWS X-Ray daemon telemetry to the service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html
     */
    toPutTelemetryRecords() {
        this.to('xray:PutTelemetryRecords');
        return this;
    }
    /**
     * Grants permission to upload segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html
     */
    toPutTraceSegments() {
        this.to('xray:PutTraceSegments');
        return this;
    }
    /**
     * Grants permission to add tags to an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('xray:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('xray:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html
     */
    toUpdateGroup() {
        this.to('xray:UpdateGroup');
        return this;
    }
    /**
     * Grants permission to modify a sampling rule's configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html
     */
    toUpdateSamplingRule() {
        this.to('xray:UpdateSamplingRule');
        return this;
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups
     *
     * @param groupName - Identifier for the groupName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupName, id, account, region, partition) {
        var arn = 'arn:${Partition}:xray:${Region}:${Account}:group/${GroupName}/${Id}';
        arn = arn.replace('${GroupName}', groupName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sampling-rule to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling
     *
     * @param samplingRuleName - Identifier for the samplingRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSamplingRule(samplingRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:xray:${Region}:${Account}:sampling-rule/${SamplingRuleName}';
        arn = arn.replace('${SamplingRuleName}', samplingRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Xray = Xray;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzeC1yYXkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhd3N4LXJheS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxzQ0FBNEM7QUFFNUM7Ozs7R0FJRztBQUNILE1BQWEsSUFBSyxTQUFRLHdCQUFlO0lBR3ZDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxNQUFNLENBQUM7UUE2V3BCLG9CQUFlLEdBQW9CO1lBQzNDLE1BQU0sRUFBRTtnQkFDTixnQkFBZ0I7Z0JBQ2hCLHFCQUFxQjthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLG9CQUFvQjtnQkFDcEIsYUFBYTtnQkFDYixvQkFBb0I7Z0JBQ3BCLHFCQUFxQjtnQkFDckIsa0JBQWtCO2dCQUNsQixhQUFhO2dCQUNiLG9CQUFvQjthQUNyQjtZQUNELHdCQUF3QixFQUFFO2dCQUN4QixxQkFBcUI7Z0JBQ3JCLHFCQUFxQjthQUN0QjtZQUNELE1BQU0sRUFBRTtnQkFDTixVQUFVO2dCQUNWLFdBQVc7Z0JBQ1gsWUFBWTtnQkFDWixrQkFBa0I7Z0JBQ2xCLHVCQUF1QjtnQkFDdkIscUJBQXFCO2dCQUNyQixrQkFBa0I7Z0JBQ2xCLCtCQUErQjtnQkFDL0Isb0JBQW9CO2dCQUNwQixpQkFBaUI7Z0JBQ2pCLGdDQUFnQztnQkFDaEMsZUFBZTtnQkFDZixtQkFBbUI7YUFDcEI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQTFZRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ3pCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzFCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBMENEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxPQUFPLENBQUMsU0FBaUIsRUFBRSxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsSUFBSSxHQUFHLEdBQUcscUVBQXFFLENBQUM7UUFDaEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGNBQWMsQ0FBQyxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxJQUFJLEdBQUcsR0FBRyw4RUFBOEUsQ0FBQztRQUN6RixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUFuY0Qsb0JBbWNDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSBcIi4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWxcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCIuLi9zaGFyZWRcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3hyYXldKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c3gtcmF5Lmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgWHJheSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3hyYXknO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3hyYXldKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c3gtcmF5Lmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiB0cmFjZXMgc3BlY2lmaWVkIGJ5IElELiBFYWNoIHRyYWNlIGlzIGEgY29sbGVjdGlvbiBvZiBzZWdtZW50IGRvY3VtZW50cyB0aGF0IG9yaWdpbmF0ZXMgZnJvbSBhIHNpbmdsZSByZXF1ZXN0LiBVc2UgR2V0VHJhY2VTdW1tYXJpZXMgdG8gZ2V0IGEgbGlzdCBvZiB0cmFjZSBJRHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0JhdGNoR2V0VHJhY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoR2V0VHJhY2VzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6QmF0Y2hHZXRUcmFjZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBncm91cCByZXNvdXJjZSB3aXRoIGEgbmFtZSBhbmQgYSBmaWx0ZXIgZXhwcmVzc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0NyZWF0ZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUdyb3VwKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6Q3JlYXRlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBydWxlIHRvIGNvbnRyb2wgc2FtcGxpbmcgYmVoYXZpb3IgZm9yIGluc3RydW1lbnRlZCBhcHBsaWNhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9DcmVhdGVTYW1wbGluZ1J1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2FtcGxpbmdSdWxlKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6Q3JlYXRlU2FtcGxpbmdSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgZ3JvdXAgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfRGVsZXRlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlR3JvdXAoKSB7XG4gICAgdGhpcy50bygneHJheTpEZWxldGVHcm91cCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHNhbXBsaW5nIHJ1bGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfRGVsZXRlU2FtcGxpbmdSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNhbXBsaW5nUnVsZSgpIHtcbiAgICB0aGlzLnRvKCd4cmF5OkRlbGV0ZVNhbXBsaW5nUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBjdXJyZW50IGVuY3J5cHRpb24gY29uZmlndXJhdGlvbiBmb3IgWC1SYXkgZGF0YVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0RW5jcnlwdGlvbkNvbmZpZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRFbmNyeXB0aW9uQ29uZmlnKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0RW5jcnlwdGlvbkNvbmZpZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGdyb3VwIHJlc291cmNlIGRldGFpbHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRHcm91cCgpIHtcbiAgICB0aGlzLnRvKCd4cmF5OkdldEdyb3VwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYWxsIGFjdGl2ZSBncm91cCBkZXRhaWxzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0R3JvdXBzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0R3JvdXBzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGRldGFpbHMgb2YgYSBzcGVjaWZpYyBpbnNpZ2h0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRJbnNpZ2h0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEluc2lnaHQoKSB7XG4gICAgdGhpcy50bygneHJheTpHZXRJbnNpZ2h0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGV2ZW50cyBvZiBhIHNwZWNpZmljIGluc2lnaHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldEluc2lnaHRFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW5zaWdodEV2ZW50cygpIHtcbiAgICB0aGlzLnRvKCd4cmF5OkdldEluc2lnaHRFdmVudHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgcGFydCBvZiB0aGUgc2VydmljZSBncmFwaCB3aGljaCBpcyBpbXBhY3RlZCBmb3IgYSBzcGVjaWZpYyBpbnNpZ2h0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRJbnNpZ2h0SW1wYWN0R3JhcGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW5zaWdodEltcGFjdEdyYXBoKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0SW5zaWdodEltcGFjdEdyYXBoJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIHN1bW1hcnkgb2YgYWxsIGluc2lnaHRzIGZvciBhIGdyb3VwIGFuZCB0aW1lIHJhbmdlIHdpdGggb3B0aW9uYWwgZmlsdGVyc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0SW5zaWdodFN1bW1hcmllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnNpZ2h0U3VtbWFyaWVzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0SW5zaWdodFN1bW1hcmllcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFsbCBzYW1wbGluZyBydWxlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0U2FtcGxpbmdSdWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTYW1wbGluZ1J1bGVzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0U2FtcGxpbmdSdWxlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHJlY2VudCBzYW1wbGluZyByZXN1bHRzIGZvciBhbGwgc2FtcGxpbmcgcnVsZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFNhbXBsaW5nU3RhdGlzdGljU3VtbWFyaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNhbXBsaW5nU3RhdGlzdGljU3VtbWFyaWVzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0U2FtcGxpbmdTdGF0aXN0aWNTdW1tYXJpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXF1ZXN0IGEgc2FtcGxpbmcgcXVvdGEgZm9yIHJ1bGVzIHRoYXQgdGhlIHNlcnZpY2UgaXMgdXNpbmcgdG8gc2FtcGxlIHJlcXVlc3RzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRTYW1wbGluZ1RhcmdldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2FtcGxpbmdUYXJnZXRzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0U2FtcGxpbmdUYXJnZXRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBkb2N1bWVudCB0aGF0IGRlc2NyaWJlcyBzZXJ2aWNlcyB0aGF0IHByb2Nlc3MgaW5jb21pbmcgcmVxdWVzdHMsIGFuZCBkb3duc3RyZWFtIHNlcnZpY2VzIHRoYXQgdGhleSBjYWxsIGFzIGEgcmVzdWx0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9HZXRTZXJ2aWNlR3JhcGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2VydmljZUdyYXBoKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0U2VydmljZUdyYXBoJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYW4gYWdncmVnYXRpb24gb2Ygc2VydmljZSBzdGF0aXN0aWNzIGRlZmluZWQgYnkgYSBzcGVjaWZpYyB0aW1lIHJhbmdlIGJ1Y2tldGVkIGludG8gdGltZSBpbnRlcnZhbHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFRpbWVTZXJpZXNTZXJ2aWNlU3RhdGlzdGljcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRUaW1lU2VyaWVzU2VydmljZVN0YXRpc3RpY3MoKSB7XG4gICAgdGhpcy50bygneHJheTpHZXRUaW1lU2VyaWVzU2VydmljZVN0YXRpc3RpY3MnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIHNlcnZpY2UgZ3JhcGggZm9yIG9uZSBvciBtb3JlIHNwZWNpZmljIHRyYWNlIElEc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfR2V0VHJhY2VHcmFwaC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRUcmFjZUdyYXBoKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0VHJhY2VHcmFwaCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIElEcyBhbmQgbWV0YWRhdGEgZm9yIHRyYWNlcyBhdmFpbGFibGUgZm9yIGEgc3BlY2lmaWVkIHRpbWUgZnJhbWUgdXNpbmcgYW4gb3B0aW9uYWwgZmlsdGVyLiBUbyBnZXQgdGhlIGZ1bGwgdHJhY2VzLCBwYXNzIHRoZSB0cmFjZSBJRHMgdG8gQmF0Y2hHZXRUcmFjZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX0dldFRyYWNlU3VtbWFyaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFRyYWNlU3VtbWFyaWVzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6R2V0VHJhY2VTdW1tYXJpZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3MgZm9yIGFuIFgtUmF5IHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygneHJheTpMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBlbmNyeXB0aW9uIGNvbmZpZ3VyYXRpb24gZm9yIFgtUmF5IGRhdGFcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1B1dEVuY3J5cHRpb25Db25maWcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0RW5jcnlwdGlvbkNvbmZpZygpIHtcbiAgICB0aGlzLnRvKCd4cmF5OlB1dEVuY3J5cHRpb25Db25maWcnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzZW5kIEFXUyBYLVJheSBkYWVtb24gdGVsZW1ldHJ5IHRvIHRoZSBzZXJ2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfUHV0VGVsZW1ldHJ5UmVjb3Jkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRUZWxlbWV0cnlSZWNvcmRzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6UHV0VGVsZW1ldHJ5UmVjb3JkcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwbG9hZCBzZWdtZW50IGRvY3VtZW50cyB0byBBV1MgWC1SYXkuIFRoZSBYLVJheSBTREsgZ2VuZXJhdGVzIHNlZ21lbnQgZG9jdW1lbnRzIGFuZCBzZW5kcyB0aGVtIHRvIHRoZSBYLVJheSBkYWVtb24sIHdoaWNoIHVwbG9hZHMgdGhlbSBpbiBiYXRjaGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfUHV0VHJhY2VTZWdtZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRUcmFjZVNlZ21lbnRzKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6UHV0VHJhY2VTZWdtZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCB0YWdzIHRvIGFuIFgtUmF5IHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCd4cmF5OlRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRhZ3MgZnJvbSBhbiBYLVJheSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2FwaS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6VW50YWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGdyb3VwIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS94cmF5L2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUdyb3VwKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6VXBkYXRlR3JvdXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgYSBzYW1wbGluZyBydWxlJ3MgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvYXBpL0FQSV9VcGRhdGVTYW1wbGluZ1J1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlU2FtcGxpbmdSdWxlKCkge1xuICAgIHRoaXMudG8oJ3hyYXk6VXBkYXRlU2FtcGxpbmdSdWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgXCJMaXN0XCI6IFtcbiAgICAgIFwiQmF0Y2hHZXRUcmFjZXNcIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiXG4gICAgXSxcbiAgICBcIldyaXRlXCI6IFtcbiAgICAgIFwiQ3JlYXRlR3JvdXBcIixcbiAgICAgIFwiQ3JlYXRlU2FtcGxpbmdSdWxlXCIsXG4gICAgICBcIkRlbGV0ZUdyb3VwXCIsXG4gICAgICBcIkRlbGV0ZVNhbXBsaW5nUnVsZVwiLFxuICAgICAgXCJQdXRUZWxlbWV0cnlSZWNvcmRzXCIsXG4gICAgICBcIlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgIFwiVXBkYXRlR3JvdXBcIixcbiAgICAgIFwiVXBkYXRlU2FtcGxpbmdSdWxlXCJcbiAgICBdLFxuICAgIFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiOiBbXG4gICAgICBcIkdldEVuY3J5cHRpb25Db25maWdcIixcbiAgICAgIFwiUHV0RW5jcnlwdGlvbkNvbmZpZ1wiXG4gICAgXSxcbiAgICBcIlJlYWRcIjogW1xuICAgICAgXCJHZXRHcm91cFwiLFxuICAgICAgXCJHZXRHcm91cHNcIixcbiAgICAgIFwiR2V0SW5zaWdodFwiLFxuICAgICAgXCJHZXRJbnNpZ2h0RXZlbnRzXCIsXG4gICAgICBcIkdldEluc2lnaHRJbXBhY3RHcmFwaFwiLFxuICAgICAgXCJHZXRJbnNpZ2h0U3VtbWFyaWVzXCIsXG4gICAgICBcIkdldFNhbXBsaW5nUnVsZXNcIixcbiAgICAgIFwiR2V0U2FtcGxpbmdTdGF0aXN0aWNTdW1tYXJpZXNcIixcbiAgICAgIFwiR2V0U2FtcGxpbmdUYXJnZXRzXCIsXG4gICAgICBcIkdldFNlcnZpY2VHcmFwaFwiLFxuICAgICAgXCJHZXRUaW1lU2VyaWVzU2VydmljZVN0YXRpc3RpY3NcIixcbiAgICAgIFwiR2V0VHJhY2VHcmFwaFwiLFxuICAgICAgXCJHZXRUcmFjZVN1bW1hcmllc1wiXG4gICAgXSxcbiAgICBcIlRhZ2dpbmdcIjogW1xuICAgICAgXCJUYWdSZXNvdXJjZVwiLFxuICAgICAgXCJVbnRhZ1Jlc291cmNlXCJcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGdyb3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3hyYXkvbGF0ZXN0L2Rldmd1aWRlL3hyYXktY29uY2VwdHMuaHRtbCN4cmF5LWNvbmNlcHRzLWdyb3Vwc1xuICAgKlxuICAgKiBAcGFyYW0gZ3JvdXBOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGdyb3VwTmFtZS5cbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkdyb3VwKGdyb3VwTmFtZTogc3RyaW5nLCBpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp4cmF5OiR7UmVnaW9ufToke0FjY291bnR9Omdyb3VwLyR7R3JvdXBOYW1lfS8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7R3JvdXBOYW1lfScsIGdyb3VwTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNhbXBsaW5nLXJ1bGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20veHJheS9sYXRlc3QvZGV2Z3VpZGUveHJheS1jb25jZXB0cy5odG1sI3hyYXktY29uY2VwdHMtc2FtcGxpbmdcbiAgICpcbiAgICogQHBhcmFtIHNhbXBsaW5nUnVsZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2FtcGxpbmdSdWxlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TYW1wbGluZ1J1bGUoc2FtcGxpbmdSdWxlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTp4cmF5OiR7UmVnaW9ufToke0FjY291bnR9OnNhbXBsaW5nLXJ1bGUvJHtTYW1wbGluZ1J1bGVOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7U2FtcGxpbmdSdWxlTmFtZX0nLCBzYW1wbGluZ1J1bGVOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19