"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Datapipeline = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [datapipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_datapipeline.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Datapipeline extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [datapipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_datapipeline.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'datapipeline';
        this.accessLevelList = {
            "Write": [
                "ActivatePipeline",
                "CreatePipeline",
                "DeactivatePipeline",
                "DeletePipeline",
                "PollForTask",
                "PutAccountLimits",
                "PutPipelineDefinition",
                "ReportTaskProgress",
                "ReportTaskRunnerHeartbeat",
                "SetStatus",
                "SetTaskStatus"
            ],
            "Tagging": [
                "AddTags",
                "RemoveTags"
            ],
            "Read": [
                "DescribeObjects",
                "EvaluateExpression",
                "GetPipelineDefinition",
                "QueryObjects",
                "ValidatePipelineDefinition"
            ],
            "List": [
                "DescribePipelines",
                "GetAccountLimits",
                "ListPipelines"
            ]
        };
    }
    /**
     * Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation, activation fails.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ActivatePipeline.html
     */
    toActivatePipeline() {
        this.to('datapipeline:ActivatePipeline');
        return this;
    }
    /**
     * Adds or modifies tags for the specified pipeline.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        this.to('datapipeline:AddTags');
        return this;
    }
    /**
     * Creates a new, empty pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        this.to('datapipeline:CreatePipeline');
        return this;
    }
    /**
     * Deactivates the specified running pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeactivatePipeline.html
     */
    toDeactivatePipeline() {
        this.to('datapipeline:DeactivatePipeline');
        return this;
    }
    /**
     * Deletes a pipeline, its pipeline definition, and its run history.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        this.to('datapipeline:DeletePipeline');
        return this;
    }
    /**
     * Gets the object definitions for a set of objects associated with the pipeline.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribeObjects.html
     */
    toDescribeObjects() {
        this.to('datapipeline:DescribeObjects');
        return this;
    }
    /**
     * Retrieves metadata about one or more pipelines.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribePipelines.html
     */
    toDescribePipelines() {
        this.to('datapipeline:DescribePipelines');
        return this;
    }
    /**
     * Task runners call EvaluateExpression to evaluate a string in the context of the specified object.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_EvaluateExpression.html
     */
    toEvaluateExpression() {
        this.to('datapipeline:EvaluateExpression');
        return this;
    }
    /**
     * Description for GetAccountLimits
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetAccountLimits.html
     */
    toGetAccountLimits() {
        this.to('datapipeline:GetAccountLimits');
        return this;
    }
    /**
     * Gets the definition of the specified pipeline.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetPipelineDefinition.html
     */
    toGetPipelineDefinition() {
        this.to('datapipeline:GetPipelineDefinition');
        return this;
    }
    /**
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        this.to('datapipeline:ListPipelines');
        return this;
    }
    /**
     * Task runners call PollForTask to receive a task to perform from AWS Data Pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PollForTask.html
     */
    toPollForTask() {
        this.to('datapipeline:PollForTask');
        return this;
    }
    /**
     * Description for PutAccountLimits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutAccountLimits.html
     */
    toPutAccountLimits() {
        this.to('datapipeline:PutAccountLimits');
        return this;
    }
    /**
     * Adds tasks, schedules, and preconditions to the specified pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutPipelineDefinition.html
     */
    toPutPipelineDefinition() {
        this.to('datapipeline:PutPipelineDefinition');
        return this;
    }
    /**
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_QueryObjects.html
     */
    toQueryObjects() {
        this.to('datapipeline:QueryObjects');
        return this;
    }
    /**
     * Removes existing tags from the specified pipeline.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        this.to('datapipeline:RemoveTags');
        return this;
    }
    /**
     * Task runners call ReportTaskProgress when assigned a task to acknowledge that it has the task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskProgress.html
     */
    toReportTaskProgress() {
        this.to('datapipeline:ReportTaskProgress');
        return this;
    }
    /**
     * Task runners call ReportTaskRunnerHeartbeat every 15 minutes to indicate that they are operational.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskRunnerHeartbeat.html
     */
    toReportTaskRunnerHeartbeat() {
        this.to('datapipeline:ReportTaskRunnerHeartbeat');
        return this;
    }
    /**
     * Requests that the status of the specified physical or logical pipeline objects be updated in the specified pipeline.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetStatus.html
     */
    toSetStatus() {
        this.to('datapipeline:SetStatus');
        return this;
    }
    /**
     * Task runners call SetTaskStatus to notify AWS Data Pipeline that a task is completed and provide information about the final status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetTaskStatus.html
     */
    toSetTaskStatus() {
        this.to('datapipeline:SetTaskStatus');
        return this;
    }
    /**
     * Validates the specified pipeline definition to ensure that it is well formed and can be run without error.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineCreator()
     * - .ifTag()
     * - .ifWorkerGroup()
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ValidatePipelineDefinition.html
     */
    toValidatePipelineDefinition() {
        this.to('datapipeline:ValidatePipelineDefinition');
        return this;
    }
    /**
     * The IAM user that created the pipeline.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-example-tag-policies.html#ex3
     *
     * Applies to actions:
     * - .toActivatePipeline()
     * - .toAddTags()
     * - .toDeactivatePipeline()
     * - .toDeletePipeline()
     * - .toDescribeObjects()
     * - .toDescribePipelines()
     * - .toEvaluateExpression()
     * - .toGetPipelineDefinition()
     * - .toPutPipelineDefinition()
     * - .toQueryObjects()
     * - .toRemoveTags()
     * - .toSetStatus()
     * - .toValidatePipelineDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPipelineCreator(value, operator) {
        return this.if(`datapipeline:PipelineCreator`, value, operator || 'ArnLike');
    }
    /**
     * A customer-specified key/value pair that can be attached to a resource.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-resourcebased-access.html#dp-control-access-tags
     *
     * Applies to actions:
     * - .toActivatePipeline()
     * - .toAddTags()
     * - .toCreatePipeline()
     * - .toDeactivatePipeline()
     * - .toDeletePipeline()
     * - .toDescribeObjects()
     * - .toDescribePipelines()
     * - .toEvaluateExpression()
     * - .toGetPipelineDefinition()
     * - .toPutPipelineDefinition()
     * - .toQueryObjects()
     * - .toRemoveTags()
     * - .toSetStatus()
     * - .toValidatePipelineDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTag(value, operator) {
        return this.if(`datapipeline:Tag`, value, operator || 'ArnLike');
    }
    /**
     * The name of a worker group for which a Task Runner retrieves work.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-resourcebased-access.html#dp-control-access-workergroup
     *
     * Applies to actions:
     * - .toActivatePipeline()
     * - .toDeactivatePipeline()
     * - .toGetPipelineDefinition()
     * - .toPollForTask()
     * - .toPutPipelineDefinition()
     * - .toValidatePipelineDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifWorkerGroup(value, operator) {
        return this.if(`datapipeline:workerGroup`, value, operator || 'ArnLike');
    }
}
exports.Datapipeline = Datapipeline;
//# sourceMappingURL=data:application/json;base64,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