"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
/**
 * Use condition operators in the `Condition` element to match the condition key and value in the policy against values in the request context. For more information about the `Condition` element, see [IAM JSON Policy Elements: Condition](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 *
 * The condition operator that you can use in a policy depends on the condition key you choose. You can choose a global condition key or a service-specific condition key. To learn which condition operator you can use for a global condition key, see [AWS Global Condition Context Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html). To learn which condition operator you can use for a service-specific condition key, see [Actions, Resources, and Condition Keys for AWS Services](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_actions-resources-contextkeys.html) and choose the service that you want to view.
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html
 */
class Operator {
    constructor() {
        this.base = '';
        this.hasIfExists = false;
        this.hasForAllValues = false;
        this.hasForAnyValue = false;
    }
    setBase(base) {
        if (this.base.length) {
            throw new Error(`Operator already set: ${this.base}`);
        }
        this.base = base;
        return this;
    }
    toString() {
        let value = this.base;
        if (this.base.length == 0) {
            throw new Error('No operator set');
        }
        if (this.hasIfExists) {
            value += 'IfExists';
        }
        if (this.hasForAllValues) {
            value = `ForAllValues:${value}`;
        }
        if (this.hasForAnyValue) {
            value = `ForAnyValue:${value}`;
        }
        if (this.base == 'Null' && value != this.base) {
            throw new Error(`You cannot add modifiers to the "Null" operator: ${value}`);
        }
        return value;
    }
    /**
     * You can add `IfExists` to the end of any condition operator name except the `Null` condition. For example, `StringLikeIfExists`. You do this to say "If the policy key is present in the context of the request, process the key as specified in the policy. If the key is not present, evaluate the condition element as true." Other condition elements in the statement can still result in a nonmatch, but not a missing key when checked with `...IfExists`.
     */
    ifExists() {
        this.hasIfExists = true;
        return this;
    }
    /**
     * Tests whether the value of every member of the request set is a subset of the condition key set. The condition returns true if every key value in the request matches at least one value in the policy. It also returns true if there are no keys in the request, or if the key values resolve to a null data set, such as an empty string.
     */
    forAllValues() {
        this.hasForAllValues = true;
        return this;
    }
    /**
     * Tests whether at least one member of the set of request values matches at least one member of the set of condition key values. The condition returns true if any one of the key values in the request matches any one of the condition values in the policy. For no matching key or a null dataset, the condition returns false.
     */
    forAnyValue() {
        this.hasForAnyValue = true;
        return this;
    }
    /**
     * Exact match, case sensitive.
     */
    stringEquals() {
        return this.setBase('StringEquals');
    }
    /**
     * Negated exact match, case sensitive.
     */
    stringNotEquals() {
        return this.setBase('StringNotEquals');
    }
    /**
     * Exact match, ignore case.
     */
    stringEqualsIgnoreCase() {
        return this.setBase('StringEqualsIgnoreCase');
    }
    /**
     * Exact exact match, ignore case.
     */
    stringNotEqualsIgnoreCase() {
        return this.setBase('StringNotEqualsIgnoreCase');
    }
    /**
     * Case-sensitive match.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     */
    stringLike() {
        return this.setBase('StringLike');
    }
    /**
     * Negated case-sensitive matching.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     */
    stringNotLike() {
        return this.setBase('StringNotLike');
    }
    /**
     * Exact match.
     */
    numericEquals() {
        return this.setBase('NumericEquals');
    }
    /**
     * Negated exact match.
     */
    numericNotEquals() {
        return this.setBase('NumericNotEquals');
    }
    /**
     * Match numbers lower than value: `<`
     */
    numericLessThan() {
        return this.setBase('NumericLessThan');
    }
    /**
     * Match numbers lower or equal to value: `<=`
     */
    numericLessThanEquals() {
        return this.setBase('NumericLessThanEquals');
    }
    /**
     * Match numbers higher than value: `>`
     */
    numericGreaterThan() {
        return this.setBase('NumericGreaterThan');
    }
    /**
     * Match numbers higher or equal value: `>=`
     */
    numericGreaterThanEquals() {
        return this.setBase('NumericGreaterThanEquals');
    }
    /**
     * Match a specific date: `=`
     */
    dateEquals() {
        return this.setBase('DateEquals');
    }
    /**
     * Negated match for a specific date: `!=`
     */
    dateNotEquals() {
        return this.setBase('DateNotEquals');
    }
    /**
     * Match before a specific date and time: `<`
     */
    dateLessThan() {
        return this.setBase('DateLessThan');
    }
    /**
     * Match at or before a specific date and time: `<=`
     */
    dateLessThanEquals() {
        return this.setBase('DateLessThanEquals');
    }
    /**
     * Match after a specific a date and time: `>`
     */
    dateGreaterThan() {
        return this.setBase('DateGreaterThan');
    }
    /**
     * Match at or after a specific date and time: `>=`
     */
    dateGreaterThanEquals() {
        return this.setBase('DateGreaterThanEquals');
    }
    /**
     * Boolean match
     */
    bool() {
        return this.setBase('Bool');
    }
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     */
    binaryEquals() {
        return this.setBase('BinaryEquals');
    }
    /**
     * Match an IP address or range: `=`
     */
    ipAddress() {
        return this.setBase('IpAddress');
    }
    /**
     * All IP addresses except the specified IP address or range `!=`
     */
    notIpAddress() {
        return this.setBase('NotIpAddress');
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnEquals` and `ArnLike` behave identical.
     */
    arnEquals() {
        return this.setBase('ArnEquals');
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     */
    arnNotEquals() {
        return this.setBase('ArnNotEquals');
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     */
    arnLike() {
        return this.setBase('ArnLike');
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     */
    arnNotLike() {
        return this.setBase('ArnNotLike');
    }
    /**
     * Check if a key is present at the time of authorization. In the policy statement, use either true (the key doesn't exist — it is null) or false (the key exists and its value is not null).
     */
    null() {
        return this.setBase('Null');
    }
}
exports.Operator = Operator;
//# sourceMappingURL=data:application/json;base64,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