"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithCondition = void 0;
const operators_1 = require("../operators");
const _1_base_1 = require("./1-base");
/**
 * Adds "condition" functionality to the Policy Statement
 */
class PolicyStatementWithCondition extends _1_base_1.PolicyStatementBase {
    constructor() {
        super(...arguments);
        this.conditions = {};
        this.cdkConditionsApplied = false;
    }
    /**
     * Injects conditions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyConditions();
            return super.toJSON();
        }
        const statement = super.toJSON();
        if (this.hasConditions()) {
            statement.Condition = this.conditions;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyConditions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyConditions() {
        if (this.hasConditions() && !this.cdkConditionsApplied) {
            Object.keys(this.conditions).forEach((operator) => {
                Object.keys(this.conditions[operator]).forEach((key) => {
                    const condition = {};
                    condition[key] = this.conditions[operator][key];
                    // @ts-ignore only available after swapping 1-base
                    this.addCondition(operator, condition);
                });
            });
            this.cdkConditionsApplied = true;
        }
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.conditions).length > 0;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (typeof operator === 'undefined') {
            operator = new operators_1.Operator().stringLike();
        }
        var op = '';
        if (typeof operator === 'string') {
            op = operator;
        }
        else {
            op = operator.toString();
        }
        if (!(op in this.conditions)) {
            this.conditions[op] = {};
        }
        this.conditions[op][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifAwsCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator || new operators_1.Operator().forAnyValue().stringEquals());
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:CurrentTime', value, operator || new operators_1.Operator().dateLessThanEquals());
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:EpochTime', value, operator || new operators_1.Operator().dateLessThanEquals());
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator || new operators_1.Operator().numericLessThan());
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if(`aws:MultiFactorAuthPresent`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator || new operators_1.Operator().arnLike());
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if(`aws:SecureTransport`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator || new operators_1.Operator().arnLike());
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator || new operators_1.Operator().ipAddress());
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifAwsTokenIssueTime(value, operator) {
        return this.if('aws:TokenIssueTime', dateToString(value), operator || new operators_1.Operator().dateGreaterThanEquals());
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifAwsViaAWSService(value) {
        return this.if(`aws:ViaAWSService`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator || new operators_1.Operator().ipAddress());
    }
}
exports.PolicyStatementWithCondition = PolicyStatementWithCondition;
function dateToString(value) {
    if (typeof value.getMonth === 'function') {
        value = value.toISOString();
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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