"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const minimatch = require("minimatch");
const assets_1 = require("./assets");
const fs_1 = require("./fs");
const names_1 = require("./names");
const cache_1 = require("./private/cache");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
/**
 * Stages a file or directory from a location on the file system into a staging directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 *
 * @stability stable
 */
class AssetStaging extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = path.resolve(props.sourcePath);
        this.fingerprintOptions = props;
        const outdir = (_a = stage_1.Stage.of(this)) === null || _a === void 0 ? void 0 : _a.assetOutdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly asset output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.assetOutdir = outdir;
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        this.customSourceFingerprint = props.assetHash;
        this.hashType = determineHashType(props.assetHashType, this.customSourceFingerprint);
        // Decide what we're going to do, without actually doing it yet
        let stageThisAsset;
        let skip = false;
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_b = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _b !== void 0 ? _b : ['*'];
            skip = !bundlingStacks.find(pattern => minimatch(stack_1.Stack.of(this).stackName, pattern));
            const bundling = props.bundling;
            stageThisAsset = () => this.stageByBundling(bundling, skip);
        }
        else {
            stageThisAsset = () => this.stageByCopying();
        }
        // Calculate a cache key from the props. This way we can check if we already
        // staged this asset and reuse the result (e.g. the same asset with the same
        // configuration is used in multiple stacks). In this case we can completely
        // skip file system and bundling operations.
        //
        // The output directory and whether this asset is skipped or not should also be
        // part of the cache key to make sure we don't accidentally return the wrong
        // staged asset from the cache.
        this.cacheKey = calculateCacheKey({
            outdir: this.assetOutdir,
            sourcePath: path.resolve(props.sourcePath),
            bundling: props.bundling,
            assetHashType: this.hashType,
            customFingerprint: this.customSourceFingerprint,
            extraHash: props.extraHash,
            exclude: props.exclude,
            ignoreMode: props.ignoreMode,
            skip,
        });
        const staged = AssetStaging.assetCache.obtain(this.cacheKey, stageThisAsset);
        this.stagedPath = staged.stagedPath;
        this.absoluteStagedPath = staged.stagedPath;
        this.assetHash = staged.assetHash;
    }
    /**
     * Clears the asset hash cache.
     *
     * @stability stable
     */
    static clearAssetHashCache() {
        this.assetCache.clear();
    }
    /**
     * (deprecated) A cryptographic hash of the asset.
     *
     * @deprecated see `assetHash`.
     */
    get sourceHash() {
        return this.assetHash;
    }
    /**
     * Return the path to the staged asset, relative to the Cloud Assembly (manifest) directory of the given stack.
     *
     * Only returns a relative path if the asset was staged, returns an absolute path if
     * it was not staged.
     *
     * A bundled asset might end up in the outDir and still not count as
     * "staged"; if asset staging is disabled we're technically expected to
     * reference source directories, but we don't have a source directory for the
     * bundled outputs (as the bundle output is written to a temporary
     * directory). Nevertheless, we will still return an absolute path.
     *
     * A non-obvious directory layout may look like this:
     *
     * ```
     *    CLOUD ASSEMBLY ROOT
     *      +-- asset.12345abcdef/
     *      +-- assembly-Stage
     *            +-- MyStack.template.json
     *            +-- MyStack.assets.json <- will contain { "path": "../asset.12345abcdef" }
     * ```
     *
     * @stability stable
     */
    relativeStagedPath(stack) {
        var _a;
        const asmManifestDir = (_a = stage_1.Stage.of(stack)) === null || _a === void 0 ? void 0 : _a.outdir;
        if (!asmManifestDir) {
            return this.stagedPath;
        }
        const isOutsideAssetDir = path.relative(this.assetOutdir, this.stagedPath).startsWith('..');
        if (isOutsideAssetDir || this.stagingDisabled) {
            return this.stagedPath;
        }
        return path.relative(asmManifestDir, this.stagedPath);
    }
    /**
     * Stage the source to the target by copying
     *
     * Optionally skip if staging is disabled, in which case we pretend we did something but we don't really.
     */
    stageByCopying() {
        const assetHash = this.calculateHash(this.hashType);
        const stagedPath = this.stagingDisabled
            ? this.sourcePath
            : path.resolve(this.assetOutdir, renderAssetFilename(assetHash, path.extname(this.sourcePath)));
        this.stageAsset(this.sourcePath, stagedPath, 'copy');
        return { assetHash, stagedPath };
    }
    /**
     * Stage the source to the target by bundling
     *
     * Optionally skip, in which case we pretend we did something but we don't really.
     */
    stageByBundling(bundling, skip) {
        if (skip) {
            // We should have bundled, but didn't to save time. Still pretend to have a hash.
            // If the asset uses OUTPUT or BUNDLE, we use a CUSTOM hash to avoid fingerprinting
            // a potentially very large source directory. Other hash types are kept the same.
            let hashType = this.hashType;
            if (hashType === assets_1.AssetHashType.OUTPUT || hashType === assets_1.AssetHashType.BUNDLE) {
                this.customSourceFingerprint = names_1.Names.uniqueId(this);
                hashType = assets_1.AssetHashType.CUSTOM;
            }
            return {
                assetHash: this.calculateHash(hashType, bundling),
                stagedPath: this.sourcePath,
            };
        }
        // Try to calculate assetHash beforehand (if we can)
        let assetHash = this.hashType === assets_1.AssetHashType.SOURCE || this.hashType === assets_1.AssetHashType.CUSTOM
            ? this.calculateHash(this.hashType, bundling)
            : undefined;
        const bundleDir = this.determineBundleDir(this.assetOutdir, assetHash);
        this.bundle(bundling, bundleDir);
        // Calculate assetHash afterwards if we still must
        assetHash = assetHash !== null && assetHash !== void 0 ? assetHash : this.calculateHash(this.hashType, bundling, bundleDir);
        const stagedPath = path.resolve(this.assetOutdir, renderAssetFilename(assetHash));
        this.stageAsset(bundleDir, stagedPath, 'move');
        return { assetHash, stagedPath };
    }
    /**
     * Whether staging has been disabled
     */
    get stagingDisabled() {
        return !!this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
    }
    /**
     * Copies or moves the files from sourcePath to targetPath.
     *
     * Moving implies the source directory is temporary and can be trashed.
     *
     * Will not do anything if source and target are the same.
     */
    stageAsset(sourcePath, targetPath, style) {
        // Is the work already done?
        const isAlreadyStaged = fs.existsSync(targetPath);
        if (isAlreadyStaged) {
            if (style === 'move' && sourcePath !== targetPath) {
                fs.removeSync(sourcePath);
            }
            return;
        }
        // Moving can be done quickly
        if (style == 'move') {
            fs.renameSync(sourcePath, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${sourcePath}`);
        }
    }
    /**
     * Determine the directory where we're going to write the bundling output
     *
     * This is the target directory where we're going to write the staged output
     * files if we can (if the hash is fully known), or a temporary directory
     * otherwise.
     */
    determineBundleDir(outdir, sourceHash) {
        if (sourceHash) {
            return path.resolve(outdir, renderAssetFilename(sourceHash));
        }
        // When the asset hash isn't known in advance, bundler outputs to an
        // intermediate directory named after the asset's cache key
        return path.resolve(outdir, `bundling-temp-${this.cacheKey}`);
    }
    /**
     * Bundles an asset to the given directory
     *
     * If the given directory already exists, assume that everything's already
     * in order and don't do anything.
     *
     * @param options Bundling options
     * @param bundleDir Where to create the bundle directory
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, bundleDir) {
        var _a, _b, _c;
        if (fs.existsSync(bundleDir)) {
            return;
        }
        fs.ensureDirSync(bundleDir);
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_b = options.local) === null || _b === void 0 ? void 0 : _b.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image.run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_c = options.workingDirectory) !== null && _c !== void 0 ? _c : AssetStaging.BUNDLING_INPUT_DIR,
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
    }
    calculateHash(hashType, bundling, outputDir) {
        var _a;
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update((_a = this.customSourceFingerprint) !== null && _a !== void 0 ? _a : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
            case assets_1.AssetHashType.OUTPUT:
                if (!outputDir) {
                    throw new Error(`Cannot use \`${hashType}\` hash type when \`bundling\` is not specified.`);
                }
                return fs_1.FileSystem.fingerprint(outputDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * (experimental) The directory inside the bundling container into which the asset sources will be mounted.
 *
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * (experimental) The directory inside the bundling container into which the bundled output should be written.
 *
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
/**
 * Cache of asset hashes based on asset configuration to avoid repeated file
 * system and bundling operations.
 */
AssetStaging.assetCache = new cache_1.Cache();
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param customSourceFingerprint Asset hash seed given in the construct props
 */
function determineHashType(assetHashType, customSourceFingerprint) {
    const hashType = customSourceFingerprint
        ? (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.CUSTOM)
        : (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.SOURCE);
    if (customSourceFingerprint && hashType !== assets_1.AssetHashType.CUSTOM) {
        throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
    }
    if (hashType === assets_1.AssetHashType.CUSTOM && !customSourceFingerprint) {
        throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
    }
    return hashType;
}
/**
 * Calculates a cache key from the props. Normalize by sorting keys.
 */
function calculateCacheKey(props) {
    return crypto.createHash('sha256')
        .update(JSON.stringify(sortObject(props)))
        .digest('hex');
}
/**
 * Recursively sort object keys
 */
function sortObject(object) {
    if (typeof object !== 'object' || object instanceof Array) {
        return object;
    }
    const ret = {};
    for (const key of Object.keys(object).sort()) {
        ret[key] = sortObject(object[key]);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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